"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.syncLocallyPersistedTabState = exports.internalStateSlice = exports.createInternalStateStore = void 0;
var _uuid = require("uuid");
var _lodash = require("lodash");
var _toolkit = require("@reduxjs/toolkit");
var _discoverUtils = require("@kbn/discover-utils");
var _esQuery = require("@kbn/es-query");
var _runtime_state = require("./runtime_state");
var _types = require("./types");
var _actions = require("./actions");
var _selectors = require("./selectors");
var _data_sources = require("../../../../../common/data_sources");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const MIDDLEWARE_THROTTLE_MS = 300;
const MIDDLEWARE_THROTTLE_OPTIONS = {
  leading: false,
  trailing: true
};
const initialState = {
  initializationState: {
    hasESData: false,
    hasUserDataView: false
  },
  userId: undefined,
  spaceId: undefined,
  persistedDiscoverSession: undefined,
  hasUnsavedChanges: false,
  defaultProfileAdHocDataViewIds: [],
  savedDataViews: [],
  expandedDoc: undefined,
  isESQLToDataViewTransitionModalVisible: false,
  tabsBarVisibility: _types.TabsBarVisibility.default,
  tabs: {
    areInitializing: false,
    byId: {},
    allIds: [],
    unsavedIds: [],
    recentlyClosedTabsById: {},
    recentlyClosedTabIds: [],
    unsafeCurrentId: ''
  }
};
const withTab = (state, payload, fn) => {
  const tab = (0, _selectors.selectTab)(state, payload.tabId);
  if (tab) {
    fn(tab);
  }
};
const internalStateSlice = exports.internalStateSlice = (0, _toolkit.createSlice)({
  name: 'internalState',
  initialState,
  reducers: {
    setInitializationState: (state, action) => {
      state.initializationState = action.payload;
    },
    setTabs: (state, action) => {
      var _action$payload$updat;
      state.tabs.byId = action.payload.allTabs.reduce((acc, tab) => ({
        ...acc,
        [tab.id]: tab.id === action.payload.selectedTabId ? {
          ...tab,
          forceFetchOnSelect: false
        } : tab
      }), {});
      state.tabs.allIds = action.payload.allTabs.map(tab => tab.id);
      state.tabs.recentlyClosedTabsById = action.payload.recentlyClosedTabs.reduce((acc, tab) => ({
        ...acc,
        [tab.id]: tab
      }), {});
      state.tabs.recentlyClosedTabIds = action.payload.recentlyClosedTabs.map(tab => tab.id);
      state.tabs.unsafeCurrentId = action.payload.selectedTabId;
      state.persistedDiscoverSession = (_action$payload$updat = action.payload.updatedDiscoverSession) !== null && _action$payload$updat !== void 0 ? _action$payload$updat : state.persistedDiscoverSession;
    },
    setUnsavedChanges: (state, action) => {
      state.hasUnsavedChanges = action.payload.hasUnsavedChanges;
      state.tabs.unsavedIds = action.payload.unsavedTabIds;
    },
    setForceFetchOnSelect: (state, action) => withTab(state, action.payload, tab => {
      tab.forceFetchOnSelect = action.payload.forceFetchOnSelect;
    }),
    setIsDataViewLoading: (state, action) => withTab(state, action.payload, tab => {
      tab.isDataViewLoading = action.payload.isDataViewLoading;
    }),
    setDefaultProfileAdHocDataViewIds: (state, action) => {
      state.defaultProfileAdHocDataViewIds = action.payload;
    },
    setTabsBarVisibility: (state, action) => {
      state.tabsBarVisibility = action.payload;
    },
    setExpandedDoc: (state, action) => {
      state.expandedDoc = action.payload.expandedDoc;
      state.initialDocViewerTabId = action.payload.initialDocViewerTabId;
    },
    discardFlyoutsOnTabChange: state => {
      state.expandedDoc = undefined;
      state.initialDocViewerTabId = undefined;
    },
    setDataRequestParams: (state, action) => withTab(state, action.payload, tab => {
      tab.dataRequestParams = action.payload.dataRequestParams;
    }),
    /**
     * Set the tab global state, overwriting existing state and pushing to URL history
     */
    setGlobalState: (state, action) => withTab(state, action.payload, tab => {
      tab.globalState = action.payload.globalState;
    }),
    /**
     * Set the tab app state, overwriting existing state and pushing to URL history
     */
    setAppState: (state, action) => withTab(state, action.payload, tab => {
      let appState = action.payload.appState;

      // When updating to an ES|QL query, sync the data source
      if ((0, _esQuery.isOfAggregateQueryType)(appState.query)) {
        appState = {
          ...appState,
          dataSource: (0, _data_sources.createEsqlDataSource)()
        };
      }
      tab.previousAppState = tab.appState;
      tab.appState = appState;
    }),
    /**
     * Set the tab app state and previous app state, overwriting existing state and pushing to URL history
     */
    resetAppState: (state, action) => withTab(state, action.payload, tab => {
      tab.previousAppState = action.payload.appState;
      tab.appState = action.payload.appState;
    }),
    setOverriddenVisContextAfterInvalidation: (state, action) => withTab(state, action.payload, tab => {
      tab.overriddenVisContextAfterInvalidation = action.payload.overriddenVisContextAfterInvalidation;
    }),
    setControlGroupState: (state, action) => withTab(state, action.payload, tab => {
      tab.controlGroupState = action.payload.controlGroupState;
    }),
    setEsqlVariables: (state, action) => withTab(state, action.payload, tab => {
      tab.esqlVariables = action.payload.esqlVariables;
    }),
    setIsESQLToDataViewTransitionModalVisible: (state, action) => {
      state.isESQLToDataViewTransitionModalVisible = action.payload;
    },
    setResetDefaultProfileState: {
      prepare: payload => ({
        payload: {
          ...payload,
          resetDefaultProfileState: {
            ...payload.resetDefaultProfileState,
            resetId: (0, _uuid.v4)()
          }
        }
      }),
      reducer: (state, action) => withTab(state, action.payload, tab => {
        tab.resetDefaultProfileState = action.payload.resetDefaultProfileState;
      })
    },
    resetOnSavedSearchChange: (state, action) => withTab(state, action.payload, tab => {
      tab.overriddenVisContextAfterInvalidation = undefined;
      state.expandedDoc = undefined;
      state.initialDocViewerTabId = undefined;
    }),
    setESQLEditorUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.esqlEditor = action.payload.esqlEditorUiState;
    }),
    setDataGridUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.dataGrid = action.payload.dataGridUiState;
    }),
    setFieldListUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.fieldList = action.payload.fieldListUiState;
    }),
    setFieldListExistingFieldsInfoUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.fieldListExistingFieldsInfo = action.payload.fieldListExistingFieldsInfo;
    }),
    resetAffectedFieldListExistingFieldsInfoUiState: (state, action) => {
      Object.values(state.tabs.byId).forEach(tab => {
        var _tab$uiState$fieldLis;
        if (((_tab$uiState$fieldLis = tab.uiState.fieldListExistingFieldsInfo) === null || _tab$uiState$fieldLis === void 0 ? void 0 : _tab$uiState$fieldLis.dataViewId) === action.payload.dataViewId) {
          tab.uiState.fieldListExistingFieldsInfo = undefined;
        }
      });
    },
    setLayoutUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.layout = action.payload.layoutUiState;
    }),
    setSearchDraftUiState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.searchDraft = action.payload.searchDraftUiState;
    }),
    setMetricsGridState: (state, action) => withTab(state, action.payload, tab => {
      tab.uiState.metricsGrid = action.payload.metricsGridState;
    })
  },
  extraReducers: builder => {
    builder.addCase(_actions.loadDataViewList.fulfilled, (state, action) => {
      state.savedDataViews = action.payload;
    });
    builder.addCase(_actions.initializeTabs.pending, state => {
      state.tabs.areInitializing = true;
    });
    builder.addCase(_actions.initializeTabs.fulfilled, (state, action) => {
      state.userId = action.payload.userId;
      state.spaceId = action.payload.spaceId;
      state.persistedDiscoverSession = action.payload.persistedDiscoverSession;
    });
    builder.addCase(_actions.initializeSingleTab.pending, (state, action) => withTab(state, action.meta.arg, tab => {
      tab.initializationState = {
        initializationStatus: _types.TabInitializationStatus.InProgress
      };
    }));
    builder.addCase(_actions.initializeSingleTab.fulfilled, (state, action) => withTab(state, action.meta.arg, tab => {
      tab.initializationState = {
        initializationStatus: action.payload.showNoDataPage ? _types.TabInitializationStatus.NoData : _types.TabInitializationStatus.Complete
      };
    }));
    builder.addCase(_actions.initializeSingleTab.rejected, (state, action) => withTab(state, action.meta.arg, tab => {
      tab.initializationState = {
        initializationStatus: _types.TabInitializationStatus.Error,
        error: action.error
      };
    }));
    builder.addMatcher((0, _toolkit.isAnyOf)(_actions.initializeTabs.fulfilled, _actions.initializeTabs.rejected), state => {
      state.tabs.areInitializing = false;
    });
  }
});
const syncLocallyPersistedTabState = exports.syncLocallyPersistedTabState = (0, _toolkit.createAction)('internalState/syncLocallyPersistedTabState');
const createMiddleware = options => {
  const listenerMiddleware = (0, _toolkit.createListenerMiddleware)({
    extra: options
  });
  const startListening = listenerMiddleware.startListening;
  startListening({
    actionCreator: internalStateSlice.actions.setTabs,
    effect: (0, _lodash.throttle)((action, listenerApi) => {
      var _action$payload$updat2;
      const discoverSession = (_action$payload$updat2 = action.payload.updatedDiscoverSession) !== null && _action$payload$updat2 !== void 0 ? _action$payload$updat2 : listenerApi.getState().persistedDiscoverSession;
      const {
        runtimeStateManager,
        tabsStorageManager
      } = listenerApi.extra;
      const getTabInternalState = tabId => (0, _runtime_state.selectTabRuntimeInternalState)(runtimeStateManager, tabId);
      void tabsStorageManager.persistLocally(action.payload, getTabInternalState, discoverSession === null || discoverSession === void 0 ? void 0 : discoverSession.id);
    }, MIDDLEWARE_THROTTLE_MS, MIDDLEWARE_THROTTLE_OPTIONS)
  });
  startListening({
    actionCreator: syncLocallyPersistedTabState,
    effect: (0, _lodash.throttle)((action, listenerApi) => {
      const {
        runtimeStateManager,
        tabsStorageManager
      } = listenerApi.extra;
      withTab(listenerApi.getState(), action.payload, tab => {
        tabsStorageManager.updateTabStateLocally(action.payload.tabId, {
          internalState: (0, _runtime_state.selectTabRuntimeInternalState)(runtimeStateManager, tab.id),
          appState: tab.appState,
          globalState: tab.globalState
        });
      });
    }, MIDDLEWARE_THROTTLE_MS, MIDDLEWARE_THROTTLE_OPTIONS)
  });
  startListening({
    actionCreator: internalStateSlice.actions.discardFlyoutsOnTabChange,
    effect: () => {
      (0, _discoverUtils.dismissFlyouts)([_discoverUtils.DiscoverFlyouts.lensEdit, _discoverUtils.DiscoverFlyouts.metricInsights]);
    }
  });
  return listenerMiddleware.middleware;
};
const IS_JEST_ENVIRONMENT = typeof jest !== 'undefined';
const createInternalStateStore = options => {
  return (0, _toolkit.configureStore)({
    reducer: internalStateSlice.reducer,
    middleware: getDefaultMiddleware => getDefaultMiddleware({
      thunk: {
        extraArgument: options
      },
      serializableCheck: !IS_JEST_ENVIRONMENT
    }).prepend(createMiddleware(options)),
    devTools: {
      name: 'DiscoverInternalState'
    }
  });
};
exports.createInternalStateStore = createInternalStateStore;