"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildSecretsFromConnectorSpec = buildSecretsFromConnectorSpec;
exports.registerRoutes = registerRoutes;
var _configSchema = require("@kbn/config-schema");
var _connectorSpecs = require("@kbn/connector-specs");
var _saved_objects = require("../saved_objects");
var _schema = require("./schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Builds the secrets object for a connector based on its spec
 * @param connectorType - The connector type ID (e.g., '.notion')
 * @param token - The authentication token
 * @returns The secrets object to pass to the actions client
 * @throws Error if the connector spec is not found
 * @internal exported for testing
 */
function buildSecretsFromConnectorSpec(connectorType, token) {
  var _connectorSpec$auth;
  const connectorSpec = Object.values(_connectorSpecs.connectorsSpecs).find(spec => spec.metadata.id === connectorType);
  if (!connectorSpec) {
    throw new Error(`Stack connector spec not found for type "${connectorType}"`);
  }
  const hasBearerAuth = (_connectorSpec$auth = connectorSpec.auth) === null || _connectorSpec$auth === void 0 ? void 0 : _connectorSpec$auth.types.some(authType => {
    const typeId = typeof authType === 'string' ? authType : authType.type;
    return typeId === 'bearer';
  });
  const secrets = {};
  if (hasBearerAuth) {
    secrets.authType = 'bearer';
    secrets.token = token;
  } else {
    var _connectorSpec$auth2, _apiKeyHeaderAuth$def;
    const apiKeyHeaderAuth = (_connectorSpec$auth2 = connectorSpec.auth) === null || _connectorSpec$auth2 === void 0 ? void 0 : _connectorSpec$auth2.types.find(authType => {
      const typeId = typeof authType === 'string' ? authType : authType.type;
      return typeId === 'api_key_header';
    });
    const headerField = typeof apiKeyHeaderAuth !== 'string' && apiKeyHeaderAuth !== null && apiKeyHeaderAuth !== void 0 && (_apiKeyHeaderAuth$def = apiKeyHeaderAuth.defaults) !== null && _apiKeyHeaderAuth$def !== void 0 && _apiKeyHeaderAuth$def.headerField ? String(apiKeyHeaderAuth.defaults.headerField) : 'ApiKey'; // default fallback

    secrets.authType = 'api_key_header';
    secrets.apiKey = token;
    secrets.headerField = headerField;
  }
  return secrets;
}
function registerRoutes(router, logger, getStartServices) {
  // List all data connectors
  router.get({
    path: '/api/data_connectors',
    validate: false,
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    try {
      const savedObjectsClient = coreContext.savedObjects.client;
      const findResult = await savedObjectsClient.find({
        type: _saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE,
        perPage: 100
      });
      const connectors = findResult.saved_objects.map(savedObject => {
        return (0, _schema.convertSOtoAPIResponse)(savedObject);
      });
      return response.ok({
        body: {
          connectors,
          total: findResult.total
        }
      });
    } catch (error) {
      logger.error(`Failed to list all data connectors: ${error.message}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to list connectors: ${error.message}`
        }
      });
    }
  });

  // Get one data connector by ID
  router.get({
    path: '/api/data_connectors/{id}',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    try {
      const savedObjectsClient = coreContext.savedObjects.client;
      const savedObject = await savedObjectsClient.get(_saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE, request.params.id);
      return response.ok({
        body: (0, _schema.convertSOtoAPIResponse)(savedObject)
      });
    } catch (error) {
      logger.error(`Error fetching data connector: ${error.message}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to fetch data connector: ${error.message}`
        }
      });
    }
  });

  // Create data connector
  router.post({
    path: '/api/data_connectors',
    validate: {
      body: _schema.createDataConnectorRequestSchema
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    try {
      const {
        name,
        type,
        token
      } = request.body;
      const [, {
        actions,
        dataSourcesRegistry
      }] = await getStartServices();
      const dataCatalog = dataSourcesRegistry.getCatalog();
      const dataConnectorTypeDef = dataCatalog.get(type);
      if (!dataConnectorTypeDef) {
        return response.customError({
          statusCode: 400,
          body: {
            message: `Data connector type "${request.body.type}" not found`
          }
        });
      }
      const connectorType = dataConnectorTypeDef.stackConnector.type;
      const secrets = buildSecretsFromConnectorSpec(connectorType, token);
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const stackConnector = await actionsClient.create({
        action: {
          name: `${type} stack connector for data connector '${name}'`,
          actionTypeId: connectorType,
          config: {},
          secrets
        }
      });
      const savedObjectsClient = coreContext.savedObjects.client;
      const now = new Date().toISOString();
      const savedObject = await savedObjectsClient.create(_saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE, {
        name,
        type,
        config: {},
        features: [],
        createdAt: now,
        updatedAt: now,
        workflowIds: [],
        toolIds: [],
        kscIds: [stackConnector.id]
      });
      return response.ok({
        body: {
          message: 'Data connector created successfully!',
          dataConnectorId: savedObject.id,
          stackConnectorId: stackConnector.id
        }
      });
    } catch (error) {
      logger.error(`Error creating data connector: ${error.message}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to create data connector: ${error.message}`
        }
      });
    }
  });

  // Delete all data connectors
  router.delete({
    path: '/api/data_connectors',
    validate: false,
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    try {
      const savedObjectsClient = coreContext.savedObjects.client;
      const findResponse = await savedObjectsClient.find({
        type: _saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE,
        perPage: 1000
      });
      const connectors = findResponse.saved_objects;
      const kscIds = connectors.flatMap(connector => connector.attributes.kscIds);
      logger.info(`Found ${connectors.length} data connectors and ${kscIds.length} stack connectors to delete.`);
      const [, {
        actions
      }] = await getStartServices();
      const actionsClient = await actions.getActionsClientWithRequest(request);
      const deleteKscPromises = kscIds.map(kscId => actionsClient.delete({
        id: kscId
      }));
      await Promise.all(deleteKscPromises);
      const deletePromises = connectors.map(connector => savedObjectsClient.delete(_saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE, connector.id));
      await Promise.all(deletePromises);
      return response.ok({
        body: {
          success: true,
          deletedCount: connectors.length
        }
      });
    } catch (error) {
      logger.error(`Failed to delete all connectors: ${error.message}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to delete all connectors: ${error.message}`
        }
      });
    }
  });

  // Delete data connector by ID
  router.delete({
    path: '/api/data_connectors/{id}',
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    },
    security: {
      authz: {
        enabled: false,
        reason: 'This route is opted out from authorization'
      }
    }
  }, async (context, request, response) => {
    const coreContext = await context.core;
    try {
      const savedObjectsClient = coreContext.savedObjects.client;
      const savedObject = await savedObjectsClient.get(_saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE, request.params.id);
      const kscIds = savedObject.attributes.kscIds;
      const [, {
        actions
      }] = await getStartServices();
      const actionsClient = await actions.getActionsClientWithRequest(request);
      kscIds.forEach(kscId => actionsClient.delete({
        id: kscId
      }));
      logger.info(`Successfully deleted Kibana stack connectors: ${kscIds.join(', ')}`);
      await savedObjectsClient.delete(_saved_objects.DATA_CONNECTOR_SAVED_OBJECT_TYPE, savedObject.id);
      logger.info(`Successfully deleted data connector ${savedObject.id}`);
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      logger.error(`Failed to delete connector: ${error.message}`);
      return response.customError({
        statusCode: 500,
        body: {
          message: `Failed to delete connector: ${error.message}`
        }
      });
    }
  });
}