"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.stripUnmappedKeys = stripUnmappedKeys;
exports.throwOnUnmappedKeys = throwOnUnmappedKeys;
var _common = require("../../common");
var _kibana_services = require("../kibana_services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function stripUnmappedKeys(dashboardState) {
  const warnings = [];
  const {
    controlGroupInput,
    references,
    panels,
    ...rest
  } = dashboardState;
  if (controlGroupInput) {
    warnings.push(`Dropped unmapped key 'controlGroupInput' from dashboard`);
  }
  if (references) {
    warnings.push(`Dropped unmapped key 'references' from dashboard`);
  }
  function isMappedPanelType(panel) {
    const transforms = _kibana_services.embeddableService === null || _kibana_services.embeddableService === void 0 ? void 0 : _kibana_services.embeddableService.getTransforms(panel.type);
    if (transforms !== null && transforms !== void 0 && transforms.throwOnUnmappedPanel) {
      try {
        transforms.throwOnUnmappedPanel(panel.config);
      } catch (e) {
        warnings.push(`Dropped panel ${panel.uid}, panel config is not supported. Reason: ${e.message}.`);
        return false;
      }
    }
    if (!(transforms !== null && transforms !== void 0 && transforms.schema)) {
      warnings.push(`Dropped panel ${panel.uid}, panel schema not available for panel type: ${panel.type}. Panels without schemas are not supported by dashboard REST endpoints`);
    }
    return Boolean(transforms === null || transforms === void 0 ? void 0 : transforms.schema);
  }
  function removeEnhancements(panel) {
    const {
      enhancements,
      ...restOfConfig
    } = panel.config;
    if (enhancements) {
      warnings.push(`Dropped unmapped panel config key 'enhancements' from panel ${panel.uid}`);
    }
    return {
      ...panel,
      config: restOfConfig
    };
  }
  const mappedPanels = (panels !== null && panels !== void 0 ? panels : []).filter(panel => (0, _common.isDashboardSection)(panel) || isMappedPanelType(panel)).map(panel => {
    if (!(0, _common.isDashboardSection)(panel)) return removeEnhancements(panel);
    const {
      panels: sectionPanels,
      ...restOfSection
    } = panel;
    return {
      ...restOfSection,
      panels: sectionPanels.filter(isMappedPanelType).map(removeEnhancements)
    };
  });
  return {
    data: {
      ...rest,
      panels: mappedPanels
    },
    warnings
  };
}
function throwOnUnmappedKeys(dashboardState) {
  var _dashboardState$panel;
  if (dashboardState.controlGroupInput) {
    throw new Error('controlGroupInput key is not supported by dashboard REST endpoints.');
  }
  if (dashboardState.references) {
    throw new Error('references key is not supported by dashboard REST endpoints.');
  }
  function throwOnUnmappedPanelKeys(panel) {
    const transforms = _kibana_services.embeddableService === null || _kibana_services.embeddableService === void 0 ? void 0 : _kibana_services.embeddableService.getTransforms(panel.type);
    if (!(transforms !== null && transforms !== void 0 && transforms.schema)) {
      throw new Error(`Panel schema not available for panel type: ${panel.type}. Panels without schemas are not supported by dashboard REST endpoints`);
    }
    if (panel.config.enhancements) {
      throw new Error('enhancements panel config key is not supported by dashboard REST endpoints.');
    }
  }
  (_dashboardState$panel = dashboardState.panels) === null || _dashboardState$panel === void 0 ? void 0 : _dashboardState$panel.forEach(panel => {
    if ((0, _common.isDashboardSection)(panel)) {
      panel.panels.forEach(throwOnUnmappedPanelKeys);
    } else {
      throwOnUnmappedPanelKeys(panel);
    }
  });
}