"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initializeSettingsManager = initializeSettingsManager;
var _presentationPublishing = require("@kbn/presentation-publishing");
var _rxjs = require("rxjs");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const DEFAULT_SETTINGS = {
  ..._constants.DEFAULT_DASHBOARD_OPTIONS,
  description: undefined,
  tags: [],
  time_restore: false,
  project_routing_restore: false,
  title: ''
};
const comparators = {
  title: 'referenceEquality',
  description: 'referenceEquality',
  hide_panel_titles: 'referenceEquality',
  sync_colors: 'referenceEquality',
  sync_cursor: 'referenceEquality',
  sync_tooltips: 'referenceEquality',
  time_restore: 'referenceEquality',
  use_margins: 'referenceEquality',
  project_routing_restore: 'referenceEquality',
  tags: 'deepEquality'
};
function deserializeState(state) {
  return {
    ...state.options,
    description: state.description,
    tags: state.tags,
    time_restore: Boolean(state.time_range),
    project_routing_restore: Boolean(state.project_routing),
    title: state.title
  };
}
function initializeSettingsManager(initialState) {
  const stateManager = (0, _presentationPublishing.initializeStateManager)(deserializeState(initialState), DEFAULT_SETTINGS, comparators);
  function serializeSettings() {
    const {
      description,
      tags,
      time_restore,
      project_routing_restore,
      title,
      ...options
    } = stateManager.getLatestState();
    return {
      ...(description && {
        description
      }),
      tags,
      title,
      options
    };
  }
  return {
    api: {
      setTags: stateManager.api.setTags,
      getSettings: stateManager.getLatestState,
      setSettings: settings => {
        stateManager.reinitializeState({
          ...stateManager.getLatestState(),
          ...settings
        });
      },
      projectRoutingRestore$: stateManager.api.project_routing_restore$,
      title$: stateManager.api.title$,
      description$: stateManager.api.description$,
      timeRestore$: stateManager.api.time_restore$,
      hideTitle$: stateManager.api.hide_panel_titles$,
      settings: {
        syncColors$: stateManager.api.sync_colors$,
        syncCursor$: stateManager.api.sync_cursor$,
        syncTooltips$: stateManager.api.sync_tooltips$,
        useMargins$: stateManager.api.use_margins$
      }
    },
    internalApi: {
      serializeSettings,
      startComparing$: lastSavedState$ => {
        return stateManager.anyStateChange$.pipe((0, _rxjs.debounceTime)(100), (0, _rxjs.map)(() => stateManager.getLatestState()), (0, _rxjs.combineLatestWith)(lastSavedState$), (0, _rxjs.map)(([latestState, lastSavedState]) => {
          const {
            description,
            tags,
            time_restore,
            project_routing_restore,
            title,
            ...optionDiffs
          } = (0, _presentationPublishing.diffComparators)(comparators, deserializeState(lastSavedState), latestState, DEFAULT_SETTINGS);
          // options needs to contain all values and not just diffs since is spread into saved state
          const options = Object.keys(optionDiffs).length ? {
            ...serializeSettings().options,
            ...optionDiffs
          } : undefined;
          return {
            ...(description && {
              description
            }),
            ...(tags && {
              tags
            }),
            ...(title && {
              title
            }),
            ...(typeof time_restore === 'boolean' && {
              time_restore
            }),
            ...(typeof project_routing_restore === 'boolean' && {
              project_routing_restore
            }),
            ...(options && {
              options
            })
          };
        }));
      },
      reset: lastSavedState => {
        stateManager.reinitializeState(deserializeState(lastSavedState));
      }
    }
  };
}