"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.decorateEsError = decorateEsError;
var _lodash = require("lodash");
var _elasticsearch = require("@elastic/elasticsearch");
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const responseErrors = {
  isServiceUnavailable: statusCode => statusCode === 503,
  isConflict: statusCode => statusCode === 409,
  isNotAuthorized: statusCode => statusCode === 401,
  isForbidden: statusCode => statusCode === 403,
  isRequestEntityTooLarge: statusCode => statusCode === 413,
  isNotFound: statusCode => statusCode === 404,
  isBadRequest: statusCode => statusCode === 400,
  isTooManyRequests: statusCode => statusCode === 429
};
const {
  ConnectionError,
  NoLivingConnectionsError,
  TimeoutError
} = _elasticsearch.errors;
const SCRIPT_CONTEXT_DISABLED_REGEX = /(?:cannot execute scripts using \[)([a-z]*)(?:\] context)/;
const INLINE_SCRIPTS_DISABLED_MESSAGE = 'cannot execute [inline] scripts';
function decorateEsError(error) {
  if (!(error instanceof Error)) {
    throw new Error('Expected an instance of Error');
  }
  const {
    reason: esErrorReason
  } = (0, _lodash.get)(error, 'body.error', {
    reason: undefined
  });
  const reason = esErrorReason || undefined;
  if (error instanceof ConnectionError || error instanceof NoLivingConnectionsError || error instanceof TimeoutError || responseErrors.isServiceUnavailable(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateEsUnavailableError(error, reason);
  }
  if (responseErrors.isConflict(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateConflictError(error, reason);
  }
  if (responseErrors.isNotAuthorized(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateNotAuthorizedError(error, reason);
  }
  if (responseErrors.isForbidden(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateForbiddenError(error, reason);
  }
  if (responseErrors.isRequestEntityTooLarge(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateRequestEntityTooLargeError(error, reason);
  }
  if (responseErrors.isNotFound(error.statusCode)) {
    var _error$meta, _error$meta$body, _error$meta$body$erro, _error$meta$body$erro2, _error$meta2;
    const match = error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : (_error$meta$body = _error$meta.body) === null || _error$meta$body === void 0 ? void 0 : (_error$meta$body$erro = _error$meta$body.error) === null || _error$meta$body$erro === void 0 ? void 0 : (_error$meta$body$erro2 = _error$meta$body$erro.reason) === null || _error$meta$body$erro2 === void 0 ? void 0 : _error$meta$body$erro2.match(/no such index \[(.+)\] and \[require_alias\] request flag is \[true\] and \[.+\] is not an alias/);
    if (match && match.length > 0) {
      return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateIndexAliasNotFoundError(error, match[1]);
    }
    // Throw EsUnavailable error if the 404 is not from elasticsearch
    // Needed here to verify Product support for any non-ignored 404 responses from calls to ES
    if (!(0, _coreElasticsearchServerInternal.isSupportedEsServer)(error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : _error$meta2.headers)) {
      return _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
    }
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError();
  }
  if (responseErrors.isTooManyRequests(error.statusCode)) {
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateTooManyRequestsError(error, reason);
  }
  if (responseErrors.isBadRequest(error.statusCode)) {
    if (SCRIPT_CONTEXT_DISABLED_REGEX.test(reason || '') || reason === INLINE_SCRIPTS_DISABLED_MESSAGE) {
      return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateEsCannotExecuteScriptError(error, reason);
    }
    return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateBadRequestError(error, reason);
  }
  return _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateGeneralError(error, reason);
}