"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Fetch = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _url = require("url");
var _rxjs = require("rxjs");
var _coreExecutionContextBrowserInternal = require("@kbn/core-execution-context-browser-internal");
var _coreHttpCommon = require("@kbn/core-http-common");
var _http_fetch_error = require("./http_fetch_error");
var _http_intercept_controller = require("./http_intercept_controller");
var _intercept = require("./intercept");
var _http_intercept_halt_error = require("./http_intercept_halt_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const JSON_CONTENT = /^(application\/(json|x-javascript)|text\/(x-)?javascript|x-json)(;.*)?$/;
const NDJSON_CONTENT = /^(application\/ndjson)(;.*)?$/;
const ZIP_CONTENT = /^(application\/zip)(;.*)?$/;
const VECTOR_TILE = /^(application\/vnd.mapbox-vector-tile)(;.*)?$/;
const removedUndefined = obj => {
  return (0, _lodash.omitBy)(obj, v => v === undefined);
};
class Fetch {
  constructor(params) {
    (0, _defineProperty2.default)(this, "interceptors", new Set());
    (0, _defineProperty2.default)(this, "requestCount$", new _rxjs.BehaviorSubject(0));
    (0, _defineProperty2.default)(this, "delete", this.shorthand('DELETE'));
    (0, _defineProperty2.default)(this, "get", this.shorthand('GET'));
    (0, _defineProperty2.default)(this, "head", this.shorthand('HEAD'));
    (0, _defineProperty2.default)(this, "options", this.shorthand('options'));
    (0, _defineProperty2.default)(this, "patch", this.shorthand('PATCH'));
    (0, _defineProperty2.default)(this, "post", this.shorthand('POST'));
    (0, _defineProperty2.default)(this, "put", this.shorthand('PUT'));
    (0, _defineProperty2.default)(this, "fetch", async (pathOrOptions, options) => {
      const optionsWithPath = validateFetchArguments(pathOrOptions, options);
      const controller = new _http_intercept_controller.HttpInterceptController();

      // We wrap the interception in a separate promise to ensure that when
      // a halt is called we do not resolve or reject, halting handling of the promise.
      return new Promise(async (resolve, reject) => {
        try {
          this.requestCount$.next(this.requestCount$.value + 1);
          const interceptedOptions = await (0, _intercept.interceptRequest)(optionsWithPath, this.interceptors, controller);
          const initialResponse = (0, _intercept.interceptFetch)(this.fetchResponse.bind(this), interceptedOptions, this.interceptors, controller);
          const interceptedResponse = await (0, _intercept.interceptResponse)(interceptedOptions, initialResponse, this.interceptors, controller);
          if (optionsWithPath.asResponse) {
            resolve(interceptedResponse);
          } else {
            resolve(interceptedResponse.body);
          }
        } catch (error) {
          if (!(error instanceof _http_intercept_halt_error.HttpInterceptHaltError)) {
            reject(error);
          }
        } finally {
          this.requestCount$.next(this.requestCount$.value - 1);
        }
      });
    });
    this.params = params;
  }
  intercept(interceptor) {
    this.interceptors.add(interceptor);
    return () => {
      this.interceptors.delete(interceptor);
    };
  }
  removeAllInterceptors() {
    this.interceptors.clear();
  }
  getRequestCount$() {
    return this.requestCount$.asObservable();
  }
  createRequest(options) {
    const context = this.params.executionContext.withGlobalContext(options.context);
    const {
      version
    } = options;

    // Merge and destructure options out that are not applicable to the Fetch API.
    const {
      query,
      prependBasePath: shouldPrependBasePath,
      asResponse,
      asSystemRequest,
      ...fetchOptions
    } = {
      method: 'GET',
      credentials: 'same-origin',
      prependBasePath: true,
      ...options,
      // options can pass an `undefined` Content-Type to erase the default value.
      // however we can't pass it to `fetch` as it will send an `Content-Type: Undefined` header
      headers: removedUndefined({
        'Content-Type': 'application/json',
        ...options.headers,
        'kbn-version': this.params.kibanaVersion,
        [_coreHttpCommon.KIBANA_BUILD_NR_HEADER]: this.params.buildNumber,
        [_coreHttpCommon.ELASTIC_HTTP_VERSION_HEADER]: version,
        [_coreHttpCommon.X_ELASTIC_INTERNAL_ORIGIN_REQUEST]: 'Kibana',
        ...(!(0, _lodash.isEmpty)(context) ? new _coreExecutionContextBrowserInternal.ExecutionContextContainer(context).toHeader() : {})
      })
    };
    const url = (0, _url.format)({
      pathname: shouldPrependBasePath ? this.params.basePath.prepend(options.path) : options.path,
      query: removedUndefined(query)
    });

    // Make sure the system request header is only present if `asSystemRequest` is true.
    if (asSystemRequest) {
      fetchOptions.headers['kbn-system-request'] = 'true';
    }
    return new Request(url, fetchOptions);
  }
  async fetchResponse(fetchOptions) {
    const request = this.createRequest(fetchOptions);
    let response;
    let body = null;
    try {
      response = await window.fetch(request);
    } catch (err) {
      var _err$name;
      throw new _http_fetch_error.HttpFetchError(err.message, (_err$name = err.name) !== null && _err$name !== void 0 ? _err$name : 'Error', request);
    }
    const contentType = response.headers.get('Content-Type') || '';
    try {
      if (fetchOptions.rawResponse) {
        body = null;
      } else if (NDJSON_CONTENT.test(contentType) || ZIP_CONTENT.test(contentType)) {
        body = await response.blob();
      } else if (JSON_CONTENT.test(contentType)) {
        body = await response.json();
      } else if (VECTOR_TILE.test(contentType)) {
        body = await response.arrayBuffer();
      } else {
        const text = await response.text();
        try {
          body = JSON.parse(text);
        } catch (err) {
          body = text;
        }
      }
    } catch (err) {
      var _err$name2;
      throw new _http_fetch_error.HttpFetchError(err.message, (_err$name2 = err.name) !== null && _err$name2 !== void 0 ? _err$name2 : 'Error', request, response, body);
    }
    if (!response.ok) {
      throw new _http_fetch_error.HttpFetchError(response.statusText, 'Error', request, response, body);
    }
    return {
      fetchOptions,
      request,
      response,
      body
    };
  }
  shorthand(method) {
    return (pathOrOptions, options) => {
      const optionsWithPath = validateFetchArguments(pathOrOptions, options);
      return this.fetch({
        ...optionsWithPath,
        method
      });
    };
  }
}

/**
 * Ensure that the overloaded arguments to `HttpHandler` are valid.
 */
exports.Fetch = Fetch;
const validateFetchArguments = (pathOrOptions, options) => {
  var _fullOptions$headers, _fullOptions$headers2;
  let fullOptions;
  if (typeof pathOrOptions === 'string' && (typeof options === 'object' || options === undefined)) {
    fullOptions = {
      ...options,
      path: pathOrOptions
    };
  } else if (typeof pathOrOptions === 'object' && options === undefined) {
    fullOptions = pathOrOptions;
  } else {
    throw new Error(`Invalid fetch arguments, must either be (string, object) or (object, undefined), received (${typeof pathOrOptions}, ${typeof options})`);
  }
  if (fullOptions.rawResponse && !fullOptions.asResponse) {
    throw new Error('Invalid fetch arguments, rawResponse = true is only supported when asResponse = true');
  }
  const invalidKbnHeaders = Object.keys((_fullOptions$headers = fullOptions.headers) !== null && _fullOptions$headers !== void 0 ? _fullOptions$headers : {}).filter(headerName => headerName.startsWith('kbn-'));
  const invalidInternalOriginProducHeader = Object.keys((_fullOptions$headers2 = fullOptions.headers) !== null && _fullOptions$headers2 !== void 0 ? _fullOptions$headers2 : {}).filter(headerName => headerName.includes(_coreHttpCommon.X_ELASTIC_INTERNAL_ORIGIN_REQUEST));
  if (invalidKbnHeaders.length) {
    throw new Error(`Invalid fetch headers, headers beginning with "kbn-" are not allowed: [${invalidKbnHeaders.join(',')}]`);
  }
  if (invalidInternalOriginProducHeader.length) {
    throw new Error(`Invalid fetch headers, headers beginning with "x-elastic-internal-" are not allowed: [${invalidInternalOriginProducHeader.join(',')}]`);
  }
  return fullOptions;
};