"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ProjectNavigationService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _history = require("history");
var _reactFastCompare = _interopRequireDefault(require("react-fast-compare"));
var _utils = require("./utils");
var _breadcrumbs = require("./breadcrumbs");
var _cloud_links = require("./cloud_links");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

class ProjectNavigationService {
  constructor(isServerless) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "projectHome$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "kibanaName$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "feedbackUrlParams$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "navigationTree$", new _rxjs.BehaviorSubject(undefined));
    // The flattened version of the navigation tree for quicker access
    (0, _defineProperty2.default)(this, "projectNavigationNavTreeFlattened", {});
    // The navigation tree for the Side nav UI that still contains layout information (body, footer, etc.)
    (0, _defineProperty2.default)(this, "navigationTreeUi$", new _rxjs.BehaviorSubject(null));
    (0, _defineProperty2.default)(this, "activeNodes$", new _rxjs.BehaviorSubject([]));
    (0, _defineProperty2.default)(this, "projectBreadcrumbs$", new _rxjs.BehaviorSubject({
      breadcrumbs: [],
      params: {
        absolute: false
      }
    }));
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.ReplaySubject(1));
    (0, _defineProperty2.default)(this, "solutionNavDefinitions$", new _rxjs.BehaviorSubject({}));
    // As the active definition **id** and the definitions are set independently, one before the other without
    // any guarantee of order, we need to store the next active definition id in a separate BehaviorSubject
    (0, _defineProperty2.default)(this, "nextSolutionNavDefinitionId$", new _rxjs.BehaviorSubject(null));
    // The active solution navigation definition id that has been initiated and is currently active
    (0, _defineProperty2.default)(this, "activeSolutionNavDefinitionId$", new _rxjs.BehaviorSubject(null));
    (0, _defineProperty2.default)(this, "activeDataTestSubj$", new _rxjs.BehaviorSubject(undefined));
    (0, _defineProperty2.default)(this, "location$", new _rxjs.BehaviorSubject((0, _history.createLocation)('/')));
    (0, _defineProperty2.default)(this, "deepLinksMap$", (0, _rxjs.of)({}));
    (0, _defineProperty2.default)(this, "cloudLinks$", new _rxjs.BehaviorSubject({}));
    (0, _defineProperty2.default)(this, "application", void 0);
    (0, _defineProperty2.default)(this, "navLinksService", void 0);
    (0, _defineProperty2.default)(this, "_http", void 0);
    (0, _defineProperty2.default)(this, "uiSettings", void 0);
    (0, _defineProperty2.default)(this, "navigationChangeSubscription", void 0);
    (0, _defineProperty2.default)(this, "unlistenHistory", void 0);
    this.isServerless = isServerless;
  }
  start({
    application,
    navLinksService,
    http,
    chromeBreadcrumbs$,
    logger,
    uiSettings
  }) {
    this.application = application;
    this.navLinksService = navLinksService;
    this._http = http;
    this.logger = logger;
    this.uiSettings = uiSettings;
    this.onHistoryLocationChange(application.history.location);
    this.unlistenHistory = application.history.listen(this.onHistoryLocationChange.bind(this));
    this.handleActiveNodesChange();
    this.handleSolutionNavDefinitionChange();
    this.deepLinksMap$ = navLinksService.getNavLinks$().pipe((0, _rxjs.map)(navLinks => {
      return navLinks.reduce((acc, navLink) => {
        acc[navLink.id] = navLink;
        return acc;
      }, {});
    }));
    return {
      setProjectHome: this.setProjectHome.bind(this),
      getProjectHome$: () => {
        return this.projectHome$.pipe((0, _rxjs.map)(home => {
          var _this$uiSettings;
          return ((_this$uiSettings = this.uiSettings) === null || _this$uiSettings === void 0 ? void 0 : _this$uiSettings.get('defaultRoute')) || home;
        }));
      },
      setCloudUrls: cloudUrls => {
        this.cloudLinks$.next((0, _cloud_links.getCloudLinks)(cloudUrls));
      },
      setFeedbackUrlParams: feedbackUrlParams => {
        this.feedbackUrlParams$.next(feedbackUrlParams);
      },
      setKibanaName: kibanaName => {
        this.kibanaName$.next(kibanaName);
      },
      getKibanaName$: () => {
        return this.kibanaName$.asObservable();
      },
      getFeedbackUrlParams$: () => {
        return this.feedbackUrlParams$.asObservable();
      },
      initNavigation: (id, navTreeDefinition$, config) => {
        this.initNavigation(id, navTreeDefinition$, config);
      },
      getNavigationTreeUi$: this.getNavigationTreeUi$.bind(this),
      getActiveNodes$: () => {
        return this.activeNodes$.pipe((0, _rxjs.takeUntil)(this.stop$), (0, _rxjs.distinctUntilChanged)(_reactFastCompare.default));
      },
      setProjectBreadcrumbs: (breadcrumbs, params) => {
        this.projectBreadcrumbs$.next({
          breadcrumbs: Array.isArray(breadcrumbs) ? breadcrumbs : [breadcrumbs],
          params: {
            absolute: false,
            ...params
          }
        });
      },
      getProjectBreadcrumbs$: () => {
        return (0, _rxjs.combineLatest)([this.projectBreadcrumbs$, this.activeNodes$, chromeBreadcrumbs$, this.kibanaName$, this.cloudLinks$]).pipe((0, _rxjs.map)(([projectBreadcrumbs, activeNodes, chromeBreadcrumbs, kibanaName, cloudLinks]) => {
          return (0, _breadcrumbs.buildBreadcrumbs)({
            kibanaName,
            projectBreadcrumbs,
            activeNodes,
            chromeBreadcrumbs,
            cloudLinks,
            isServerless: this.isServerless
          });
        }));
      },
      /** In stateful Kibana, get the registered solution navigations */
      getSolutionsNavDefinitions$: this.getSolutionsNavDefinitions$.bind(this),
      /** In stateful Kibana, update the registered solution navigations */
      updateSolutionNavigations: this.updateSolutionNavigations.bind(this),
      /** In stateful Kibana, change the active solution navigation */
      changeActiveSolutionNavigation: this.changeActiveSolutionNavigation.bind(this),
      /** In stateful Kibana, get the active solution navigation definition */
      getActiveSolutionNavDefinition$: this.getActiveSolutionNavDefinition$.bind(this),
      /** In stateful Kibana, get the id of the active solution navigation */
      getActiveSolutionNavId$: () => this.activeSolutionNavDefinitionId$.asObservable(),
      getActiveDataTestSubj$: () => this.activeDataTestSubj$.asObservable()
    };
  }

  /**
   * Initialize a "serverless style" navigation. For stateful deployments (not serverless), this
   * handler initialize one of the solution navigations registered.
   *
   * @param id Id for the navigation tree definition
   * @param navTreeDefinition$ The navigation tree definition
   * @param config Optional configuration object, currently only supports `dataTestSubj` to set the data-test-subj attribute for the navigation container
   */
  initNavigation(id, navTreeDefinition$, config) {
    if (this.activeSolutionNavDefinitionId$.getValue() === id) return;
    if (config !== null && config !== void 0 && config.dataTestSubj) {
      this.activeDataTestSubj$.next(config.dataTestSubj);
    }
    if (this.navigationChangeSubscription) {
      this.navigationChangeSubscription.unsubscribe();
    }
    let initialised = false;
    this.projectNavigationNavTreeFlattened = {};
    this.navigationChangeSubscription = (0, _rxjs.combineLatest)([navTreeDefinition$, this.deepLinksMap$, this.cloudLinks$]).pipe((0, _rxjs.takeUntil)(this.stop$), (0, _rxjs.map)(([def, deepLinksMap, cloudLinks]) => {
      return (0, _utils.parseNavigationTree)(id, def, {
        deepLinks: deepLinksMap,
        cloudLinks
      });
    })).subscribe({
      next: ({
        navigationTree,
        navigationTreeUI
      }) => {
        this.navigationTree$.next(navigationTree);
        this.navigationTreeUi$.next(navigationTreeUI);
        this.projectNavigationNavTreeFlattened = (0, _utils.flattenNav)(navigationTree);
        this.updateActiveProjectNavigationNodes();
        if (!initialised) {
          this.activeSolutionNavDefinitionId$.next(id);
          initialised = true;
        }
      },
      error: err => {
        var _this$logger;
        (_this$logger = this.logger) === null || _this$logger === void 0 ? void 0 : _this$logger.error(err);
      }
    });
  }
  getNavigationTreeUi$() {
    return this.navigationTreeUi$.asObservable().pipe((0, _rxjs.filter)(v => v !== null));
  }
  findActiveNodes({
    location: _location,
    flattendTree = this.projectNavigationNavTreeFlattened
  } = {}) {
    var _this$http$basePath$p, _this$http, _this$http2;
    if (!this.application) return [];
    if (!Object.keys(flattendTree).length) return [];
    const location = _location !== null && _location !== void 0 ? _location : this.application.history.location;
    let currentPathname = (_this$http$basePath$p = (_this$http = this.http) === null || _this$http === void 0 ? void 0 : _this$http.basePath.prepend(location.pathname)) !== null && _this$http$basePath$p !== void 0 ? _this$http$basePath$p : location.pathname;

    // We add possible hash to the current pathname
    // e.g. /app/kibana#/management
    currentPathname = (0, _utils.stripQueryParams)(`${currentPathname}${location.hash}`);
    return (0, _utils.findActiveNodes)(currentPathname, flattendTree, location, (_this$http2 = this.http) === null || _this$http2 === void 0 ? void 0 : _this$http2.basePath.prepend);
  }

  /**
   * Find the active nodes in the navigation tree based on the current location (or a location passed in params)
   * and update the activeNodes$ Observable.
   *
   * @param location Optional location to use to detect the active node in the new navigation tree, if not set the current location is used
   * @param forceUpdate Optional flag to force the update of the active nodes even if the active nodes are the same
   */
  updateActiveProjectNavigationNodes({
    location
  } = {}) {
    const activeNodes = this.findActiveNodes({
      location
    });
    this.activeNodes$.next(activeNodes);
    return activeNodes;
  }
  onHistoryLocationChange(location) {
    this.updateActiveProjectNavigationNodes({
      location
    });
    this.location$.next(location);
  }
  handleActiveNodesChange() {
    this.activeNodes$.pipe((0, _rxjs.takeUntil)(this.stop$),
    // skip while the project navigation is not set
    (0, _rxjs.skipWhile)(() => !this.navigationTree$.getValue()),
    // only reset when the active breadcrumb path changes, use ids to get more stable reference
    (0, _rxjs.distinctUntilChanged)((prevNodes, nextNodes) => {
      var _prevNodes$, _nextNodes$;
      return (0, _reactFastCompare.default)(prevNodes === null || prevNodes === void 0 ? void 0 : (_prevNodes$ = prevNodes[0]) === null || _prevNodes$ === void 0 ? void 0 : _prevNodes$.map(node => node.id), nextNodes === null || nextNodes === void 0 ? void 0 : (_nextNodes$ = nextNodes[0]) === null || _nextNodes$ === void 0 ? void 0 : _nextNodes$.map(node => node.id));
    }),
    // skip the initial state, we only want to reset the breadcrumbs when the active nodes change
    (0, _rxjs.skip)(1)).subscribe(() => {
      // reset the breadcrumbs when the active nodes change
      this.projectBreadcrumbs$.next({
        breadcrumbs: [],
        params: {
          absolute: false
        }
      });
    });
  }
  handleSolutionNavDefinitionChange() {
    (0, _rxjs.combineLatest)([this.solutionNavDefinitions$, this.nextSolutionNavDefinitionId$.pipe((0, _rxjs.distinctUntilChanged)())]).pipe((0, _rxjs.takeUntil)(this.stop$)).subscribe(([definitions, nextId]) => {
      const definition = typeof nextId === 'string' ? definitions[nextId] : undefined;
      const noActiveDefinition = Object.keys(definitions).length === 0 || !definition || nextId === null;
      if (noActiveDefinition) {
        this.navigationTree$.next(undefined);
        this.activeNodes$.next([]);
        return;
      }
      const {
        homePage = ''
      } = definition;
      this.waitForLink(homePage, navLink => {
        this.setProjectHome(navLink.href);
      });
      this.initNavigation(nextId, definition.navigationTree$, {
        dataTestSubj: definition.dataTestSubj
      });
    });
  }

  /**
   * This method waits for the chrome nav link to be available and then calls the callback.
   * This is necessary to avoid race conditions when we register the solution navigation
   * before the deep links are available (plugins can register them later).
   *
   * @param linkId The chrome nav link id
   * @param cb The callback to call when the link is found
   * @returns
   */
  waitForLink(linkId, cb) {
    if (!this.navLinksService) return;
    let navLink = this.navLinksService.get(linkId);
    if (navLink) {
      cb(navLink);
      return;
    }
    const stop$ = new _rxjs.Subject();
    const tenSeconds = (0, _rxjs.timer)(10000);
    this.deepLinksMap$.pipe((0, _rxjs.takeUntil)(tenSeconds), (0, _rxjs.takeUntil)(stop$)).subscribe(navLinks => {
      navLink = navLinks[linkId];
      if (navLink) {
        cb(navLink);
        stop$.next();
      }
    });
  }
  setProjectHome(homeHref) {
    this.projectHome$.next(homeHref);
  }
  changeActiveSolutionNavigation(id) {
    if (this.nextSolutionNavDefinitionId$.getValue() === id) return;
    this.nextSolutionNavDefinitionId$.next(id);
  }
  getSolutionsNavDefinitions$() {
    return this.solutionNavDefinitions$.asObservable();
  }
  getActiveSolutionNavDefinition$() {
    return (0, _rxjs.combineLatest)([this.solutionNavDefinitions$, this.activeSolutionNavDefinitionId$]).pipe((0, _rxjs.takeUntil)(this.stop$), (0, _rxjs.map)(([definitions, id]) => {
      if (id === null) return null;
      if (Object.keys(definitions).length === 0) return null;
      if (!definitions[id]) return null;
      return definitions[id];
    }));
  }
  updateSolutionNavigations(solutionNavs, replace = false) {
    if (replace) {
      this.solutionNavDefinitions$.next(solutionNavs);
    } else {
      this.solutionNavDefinitions$.next({
        ...this.solutionNavDefinitions$.getValue(),
        ...solutionNavs
      });
    }
  }
  get http() {
    if (!this._http) {
      throw new Error('Http service not provided.');
    }
    return this._http;
  }
  stop() {
    var _this$unlistenHistory;
    this.stop$.next();
    (_this$unlistenHistory = this.unlistenHistory) === null || _this$unlistenHistory === void 0 ? void 0 : _this$unlistenHistory.call(this);
  }
}
exports.ProjectNavigationService = ProjectNavigationService;