"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateConnectorName = void 0;
var _uuid = require("uuid");
var _searchConnectors = require("@kbn/search-connectors");
var _error_codes = require("../../../common/types/error_codes");
var _exists_index = require("../indices/exists_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const addIndexPrefix = (indexName, isManagedConnector) => {
  const indexPrefix = isManagedConnector ? _searchConnectors.MANAGED_CONNECTOR_INDEX_PREFIX : 'connector-';
  return `${indexPrefix}${indexName}`;
};
const addConnectorPrefix = indexName => {
  return `connector-${indexName}`;
};
const generateConnectorName = async (client, connectorType, userConnectorName, isManagedConnector = false) => {
  const prefix = (0, _searchConnectors.toAlphanumeric)(connectorType);
  if (!prefix || prefix.length === 0) {
    throw new Error('Connector type or connectorName is required');
  }
  if (userConnectorName) {
    let indexName = addIndexPrefix(userConnectorName, isManagedConnector);
    const resultSameName = await (0, _exists_index.indexOrAliasExists)(client, indexName);
    // index with same name doesn't exist
    if (!resultSameName) {
      return {
        apiKeyName: userConnectorName,
        connectorName: userConnectorName,
        indexName
      };
    }
    // if the index name already exists, we will generate until it doesn't for 20 times
    for (let i = 0; i < 20; i++) {
      const randomizedConnectorName = `${userConnectorName}-${(0, _uuid.v4)().split('-')[1].slice(0, 4)}`;
      indexName = addIndexPrefix(randomizedConnectorName, isManagedConnector);
      const result = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!result) {
        return {
          apiKeyName: addConnectorPrefix(randomizedConnectorName),
          connectorName: userConnectorName,
          indexName
        };
      }
    }
  } else {
    for (let i = 0; i < 20; i++) {
      const randomizedConnectorName = `${prefix}-${(0, _uuid.v4)().split('-')[1].slice(0, 4)}`;
      const indexName = addIndexPrefix(randomizedConnectorName, isManagedConnector);
      const result = await (0, _exists_index.indexOrAliasExists)(client, indexName);
      if (!result) {
        return {
          apiKeyName: addConnectorPrefix(randomizedConnectorName),
          connectorName: randomizedConnectorName,
          indexName
        };
      }
    }
  }
  throw new Error(_error_codes.ErrorCode.GENERATE_INDEX_NAME_ERROR);
};
exports.generateConnectorName = generateConnectorName;