"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildChildrenTree = buildChildrenTree;
exports.buildFocusedTraceItems = buildFocusedTraceItems;
exports.findRootItem = findRootItem;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_NUMBER_OF_CHILDREN = 2;
function buildChildrenTree({
  initialTraceDoc,
  itemsGroupedByParentId,
  maxNumberOfChildren
}) {
  let _processedItemsCount = 0;
  function findChildren(traceDoc) {
    const id = traceDoc.id;
    if (!id) {
      return [];
    }
    const children = [];
    const _children = itemsGroupedByParentId[id];
    if ((0, _lodash.isEmpty)(_children)) {
      return [];
    }
    for (let i = 0; i < _children.length; i++) {
      const child = _children[i];
      _processedItemsCount++;
      if (_processedItemsCount > maxNumberOfChildren) {
        break;
      }
      children.push({
        traceDoc: child,
        children: findChildren(child)
      });
    }
    return children;
  }
  return findChildren(initialTraceDoc);
}
function findRootItem(traceItems) {
  return traceItems.find(item => !item.parentId);
}
function buildFocusedTraceItems({
  traceItems,
  docId
}) {
  var _itemsGroupedByParent;
  const itemsById = traceItems.reduce((acc, curr) => {
    const id = curr.id;
    if (!id) {
      return acc;
    }
    acc[id] = curr;
    return acc;
  }, {});
  const focusedTraceDoc = itemsById[docId];
  if (!focusedTraceDoc) {
    return undefined;
  }
  const parentDoc = focusedTraceDoc.parentId ? itemsById[focusedTraceDoc.parentId] : undefined;
  const itemsGroupedByParentId = traceItems.reduce((acc, curr) => {
    const parentId = curr.parentId;
    if (!parentId) {
      acc.root = [curr];
      return acc;
    }
    const group = acc[parentId] || [];
    group.push(curr);
    acc[parentId] = group;
    return acc;
  }, {});
  const rootDoc = (_itemsGroupedByParent = itemsGroupedByParentId.root) === null || _itemsGroupedByParent === void 0 ? void 0 : _itemsGroupedByParent[0];
  if (!rootDoc) {
    return undefined;
  }
  const focusedTraceTree = buildChildrenTree({
    initialTraceDoc: focusedTraceDoc,
    itemsGroupedByParentId,
    maxNumberOfChildren: MAX_NUMBER_OF_CHILDREN
  });
  return {
    rootDoc,
    parentDoc,
    focusedTraceDoc,
    focusedTraceTree
  };
}