"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServiceTransactionGroupDetailedStatisticsPeriods = getServiceTransactionGroupDetailedStatisticsPeriods;
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _apm = require("../../../common/es_fields/apm");
var _environment_query = require("../../../common/utils/environment_query");
var _get_offset_in_ms = require("../../../common/utils/get_offset_in_ms");
var _offset_previous_period_coordinate = require("../../../common/utils/offset_previous_period_coordinate");
var _latency_aggregation_type = require("../../lib/helpers/latency_aggregation_type");
var _transactions = require("../../lib/helpers/transactions");
var _transaction_error_rate = require("../../lib/helpers/transaction_error_rate");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getServiceTransactionGroupDetailedStatistics({
  environment,
  kuery,
  serviceName,
  transactionNames,
  apmEventClient,
  transactionType,
  latencyAggregationType,
  start,
  end,
  offset,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  useDurationSummary
}) {
  var _response$aggregation, _response$aggregation2, _response$aggregation3;
  const {
    startWithOffset,
    endWithOffset
  } = (0, _get_offset_in_ms.getOffsetInMs)({
    start,
    end,
    offset
  });
  const intervalString = `${bucketSizeInSeconds}s`;
  const field = (0, _transactions.getDurationFieldForTransactions)(documentType, useDurationSummary);
  const response = await apmEventClient.search('get_service_transaction_group_detailed_statistics', {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            [_apm.SERVICE_NAME]: serviceName
          }
        }, {
          term: {
            [_apm.TRANSACTION_TYPE]: transactionType
          }
        }, ...(0, _server.rangeQuery)(startWithOffset, endWithOffset), ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.kqlQuery)(kuery)]
      }
    },
    aggs: {
      total_duration: {
        sum: {
          field
        }
      },
      transaction_groups: {
        terms: {
          field: _apm.TRANSACTION_NAME,
          include: transactionNames,
          size: transactionNames.length
        },
        aggs: {
          transaction_group_total_duration: {
            sum: {
              field
            }
          },
          timeseries: {
            date_histogram: {
              field: '@timestamp',
              fixed_interval: intervalString,
              min_doc_count: 0,
              extended_bounds: {
                min: startWithOffset,
                max: endWithOffset
              }
            },
            aggs: {
              ...(0, _latency_aggregation_type.getLatencyAggregation)(latencyAggregationType, field),
              ...(0, _transaction_error_rate.getOutcomeAggregation)(documentType)
            }
          }
        }
      }
    }
  });
  const buckets = (_response$aggregation = (_response$aggregation2 = response.aggregations) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.transaction_groups.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const totalDuration = (_response$aggregation3 = response.aggregations) === null || _response$aggregation3 === void 0 ? void 0 : _response$aggregation3.total_duration.value;
  return buckets.map(bucket => {
    const transactionName = bucket.key;
    const latency = bucket.timeseries.buckets.map(timeseriesBucket => ({
      x: timeseriesBucket.key,
      y: (0, _latency_aggregation_type.getLatencyValue)({
        latencyAggregationType,
        aggregation: timeseriesBucket.latency
      })
    }));
    const throughput = bucket.timeseries.buckets.map(timeseriesBucket => ({
      x: timeseriesBucket.key,
      y: timeseriesBucket.doc_count // sparklines only shows trend (no axis)
    }));
    const errorRate = bucket.timeseries.buckets.map(timeseriesBucket => ({
      x: timeseriesBucket.key,
      y: (0, _transaction_error_rate.calculateFailedTransactionRate)(timeseriesBucket)
    }));
    const transactionGroupTotalDuration = bucket.transaction_group_total_duration.value || 0;
    return {
      transactionName,
      latency,
      throughput,
      errorRate,
      impact: totalDuration ? transactionGroupTotalDuration * 100 / totalDuration : 0
    };
  });
}
async function getServiceTransactionGroupDetailedStatisticsPeriods({
  serviceName,
  transactionNames,
  apmEventClient,
  transactionType,
  documentType,
  rollupInterval,
  bucketSizeInSeconds,
  useDurationSummary,
  latencyAggregationType,
  environment,
  kuery,
  start,
  end,
  offset
}) {
  const commonProps = {
    apmEventClient,
    serviceName,
    transactionNames,
    transactionType,
    latencyAggregationType: latencyAggregationType,
    environment,
    kuery,
    documentType,
    rollupInterval,
    bucketSizeInSeconds,
    useDurationSummary
  };
  const currentPeriodPromise = getServiceTransactionGroupDetailedStatistics({
    ...commonProps,
    start,
    end
  });
  const previousPeriodPromise = offset ? getServiceTransactionGroupDetailedStatistics({
    ...commonProps,
    start,
    end,
    offset
  }) : [];
  const [currentPeriod, previousPeriod] = await Promise.all([currentPeriodPromise, previousPeriodPromise]);
  const firstCurrentPeriod = currentPeriod === null || currentPeriod === void 0 ? void 0 : currentPeriod[0];
  return {
    currentPeriod: (0, _lodash.keyBy)(currentPeriod, 'transactionName'),
    previousPeriod: (0, _lodash.keyBy)(previousPeriod.map(data => {
      return {
        ...data,
        errorRate: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
          currentPeriodTimeseries: firstCurrentPeriod === null || firstCurrentPeriod === void 0 ? void 0 : firstCurrentPeriod.errorRate,
          previousPeriodTimeseries: data.errorRate
        }),
        throughput: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
          currentPeriodTimeseries: firstCurrentPeriod === null || firstCurrentPeriod === void 0 ? void 0 : firstCurrentPeriod.throughput,
          previousPeriodTimeseries: data.throughput
        }),
        latency: (0, _offset_previous_period_coordinate.offsetPreviousPeriodCoordinates)({
          currentPeriodTimeseries: firstCurrentPeriod === null || firstCurrentPeriod === void 0 ? void 0 : firstCurrentPeriod.latency,
          previousPeriodTimeseries: data.latency
        })
      };
    }), 'transactionName')
  };
}