"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchExitSpanSamplesFromTraceIds = fetchExitSpanSamplesFromTraceIds;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _event_outcome = require("../../../common/event_outcome");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _apm = require("../../../common/es_fields/apm");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// https://opentelemetry.io/docs/specs/otel/configuration/sdk-environment-variables/#span-limits
const SPAN_LINK_IDS_LIMIT = 128;
const MAX_EXIT_SPANS = 10000;
const MAX_SPAN_LINKS = 1000;
async function fetchExitSpanSamplesFromTraceIds({
  apmEventClient,
  traceIds,
  start,
  end
}) {
  const [exitSpansSample, {
    outgoingSpanLinksSample,
    incomingSpanLinksSample
  }] = await Promise.all([fetchExitSpanIdsFromTraceIds({
    apmEventClient,
    traceIds,
    start,
    end
  }), fetchSpanLinksFromTraceIds({
    apmEventClient,
    traceIds,
    start,
    end
  })]);
  const [transactionsFromExitSpans, spansFromSpanLinks] = await Promise.all([fetchTransactionsFromExitSpans({
    apmEventClient,
    exitSpansSample,
    start,
    end
  }), fetchSpansFromSpanLinks({
    apmEventClient,
    outgoingSpanLinksSample,
    incomingSpanLinksSample,
    start,
    end
  })]);
  return transactionsFromExitSpans.concat(spansFromSpanLinks);
}
async function fetchExitSpanIdsFromTraceIds({
  apmEventClient,
  traceIds,
  start,
  end
}) {
  var _sampleExitSpans$aggr;
  const sampleExitSpans = await apmEventClient.search('get_service_map_exit_span_samples', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _server.termsQuery)(_apm.TRACE_ID, ...traceIds), ...(0, _server.existsQuery)(_apm.SPAN_DESTINATION_SERVICE_RESOURCE)]
      }
    },
    aggs: {
      exitSpans: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([{
            serviceName: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            spanDestinationServiceResource: {
              terms: {
                field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
              }
            }
          }]),
          size: MAX_EXIT_SPANS
        },
        aggs: {
          eventOutcomeGroup: {
            filters: {
              filters: {
                success: {
                  term: {
                    [_apm.EVENT_OUTCOME]: _event_outcome.EventOutcome.success
                  }
                },
                others: {
                  bool: {
                    must_not: {
                      term: {
                        [_apm.EVENT_OUTCOME]: _event_outcome.EventOutcome.success
                      }
                    }
                  }
                }
              }
            },
            aggs: {
              sample: {
                top_metrics: {
                  size: 1,
                  sort: {
                    [_apm.AT_TIMESTAMP]: 'asc'
                  },
                  metrics: (0, _as_mutable_array.asMutableArray)([{
                    field: _apm.SPAN_ID
                  }, {
                    field: _apm.SPAN_TYPE
                  }, {
                    field: _apm.SPAN_SUBTYPE
                  }, {
                    field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
                  }, {
                    field: _apm.SERVICE_NAME
                  }, {
                    field: _apm.SERVICE_ENVIRONMENT
                  }, {
                    field: _apm.AGENT_NAME
                  }])
                }
              }
            }
          }
        }
      }
    }
  });
  const destinationsBySpanId = new Map();
  (_sampleExitSpans$aggr = sampleExitSpans.aggregations) === null || _sampleExitSpans$aggr === void 0 ? void 0 : _sampleExitSpans$aggr.exitSpans.buckets.forEach(bucket => {
    var _eventOutcomeGroup$sa;
    const {
      success,
      others
    } = bucket.eventOutcomeGroup.buckets;
    const eventOutcomeGroup = success.sample.top.length > 0 ? success : others.sample.top.length > 0 ? others : undefined;
    const sample = eventOutcomeGroup === null || eventOutcomeGroup === void 0 ? void 0 : (_eventOutcomeGroup$sa = eventOutcomeGroup.sample.top[0]) === null || _eventOutcomeGroup$sa === void 0 ? void 0 : _eventOutcomeGroup$sa.metrics;
    if (!sample) {
      return;
    }
    const spanId = sample[_apm.SPAN_ID];
    destinationsBySpanId.set(spanId, {
      spanId,
      spanDestinationServiceResource: bucket.key.spanDestinationServiceResource,
      spanType: sample[_apm.SPAN_TYPE],
      spanSubtype: sample[_apm.SPAN_SUBTYPE],
      agentName: sample[_apm.AGENT_NAME],
      serviceName: bucket.key.serviceName,
      serviceEnvironment: sample[_apm.SERVICE_ENVIRONMENT]
    });
  });
  return destinationsBySpanId;
}
async function fetchSpanLinksFromTraceIds({
  apmEventClient,
  traceIds,
  start,
  end
}) {
  var _sampleExitSpans$aggr2, _sampleExitSpans$aggr3;
  const sampleExitSpans = await apmEventClient.search('get_service_map_span_link_samples', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), {
          bool: {
            minimum_should_match: 1,
            should: [...(0, _server.existsQuery)(_apm.SPAN_LINKS_TRACE_ID), ...(0, _server.existsQuery)(_apm.OTEL_SPAN_LINKS_TRACE_ID)]
          }
        }, {
          bool: {
            minimum_should_match: 1,
            should: [...(0, _server.termsQuery)(_apm.SPAN_LINKS_TRACE_ID, ...traceIds), ...(0, _server.termsQuery)(_apm.OTEL_SPAN_LINKS_TRACE_ID, ...traceIds),
            // needed for focused service map
            // containing incoming links in the root transaction
            ...(0, _server.termsQuery)(_apm.TRACE_ID, ...traceIds)]
          }
        }]
      }
    },
    aggs: {
      outgoingSpanLinks: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([{
            serviceName: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            spanName: {
              terms: {
                field: _apm.SPAN_NAME
              }
            }
          }]),
          size: MAX_SPAN_LINKS
        },
        aggs: {
          linkedSpanId: {
            terms: {
              field: _apm.SPAN_LINKS_SPAN_ID,
              size: SPAN_LINK_IDS_LIMIT
            }
          },
          otelLinkedSpanId: {
            terms: {
              field: _apm.OTEL_SPAN_LINKS_SPAN_ID,
              size: SPAN_LINK_IDS_LIMIT
            }
          },
          sample: {
            top_metrics: {
              size: 1,
              sort: {
                [_apm.AT_TIMESTAMP]: 'asc'
              },
              metrics: (0, _as_mutable_array.asMutableArray)([{
                field: _apm.SPAN_TYPE
              }, {
                field: _apm.SPAN_SUBTYPE
              }, {
                field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
              }, {
                field: _apm.SERVICE_NAME
              }, {
                field: _apm.SERVICE_ENVIRONMENT
              }, {
                field: _apm.AGENT_NAME
              }])
            }
          }
        }
      },
      incomingSpanLinks: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([{
            serviceName: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            transactionName: {
              terms: {
                field: _apm.TRANSACTION_NAME
              }
            }
          }]),
          size: MAX_SPAN_LINKS
        },
        aggs: {
          linkedSpanId: {
            terms: {
              field: _apm.SPAN_LINKS_SPAN_ID,
              size: SPAN_LINK_IDS_LIMIT
            }
          },
          otelLinkedSpanId: {
            terms: {
              field: _apm.OTEL_SPAN_LINKS_SPAN_ID,
              size: SPAN_LINK_IDS_LIMIT
            }
          },
          sample: {
            top_metrics: {
              size: 1,
              sort: {
                [_apm.AT_TIMESTAMP]: 'asc'
              },
              metrics: (0, _as_mutable_array.asMutableArray)([{
                field: _apm.SERVICE_NAME
              }, {
                field: _apm.SERVICE_ENVIRONMENT
              }, {
                field: _apm.AGENT_NAME
              }])
            }
          }
        }
      }
    }
  });
  const outgoingSpanLinksSample = new Map();
  (_sampleExitSpans$aggr2 = sampleExitSpans.aggregations) === null || _sampleExitSpans$aggr2 === void 0 ? void 0 : _sampleExitSpans$aggr2.outgoingSpanLinks.buckets.forEach(bucket => {
    var _bucket$sample$top$;
    const sample = (_bucket$sample$top$ = bucket.sample.top[0]) === null || _bucket$sample$top$ === void 0 ? void 0 : _bucket$sample$top$.metrics;
    if (!sample) {
      return;
    }
    const spanIds = new Set([...bucket.otelLinkedSpanId.buckets, ...bucket.linkedSpanId.buckets].map(item => item.key));
    for (const spanId of spanIds) {
      var _sample$SPAN_DESTINAT;
      outgoingSpanLinksSample.set(spanId, {
        spanId,
        spanDestinationServiceResource: (_sample$SPAN_DESTINAT = sample[_apm.SPAN_DESTINATION_SERVICE_RESOURCE]) !== null && _sample$SPAN_DESTINAT !== void 0 ? _sample$SPAN_DESTINAT : bucket.key.spanName,
        spanType: sample[_apm.SPAN_TYPE],
        spanSubtype: sample[_apm.SPAN_SUBTYPE],
        agentName: sample[_apm.AGENT_NAME],
        serviceName: bucket.key.serviceName,
        serviceEnvironment: sample[_apm.SERVICE_ENVIRONMENT]
      });
    }
  });
  const incomingSpanLinksSample = new Map();
  (_sampleExitSpans$aggr3 = sampleExitSpans.aggregations) === null || _sampleExitSpans$aggr3 === void 0 ? void 0 : _sampleExitSpans$aggr3.incomingSpanLinks.buckets.forEach(bucket => {
    var _bucket$sample$top$2;
    const sample = (_bucket$sample$top$2 = bucket.sample.top[0]) === null || _bucket$sample$top$2 === void 0 ? void 0 : _bucket$sample$top$2.metrics;
    if (!sample) {
      return;
    }
    const spanIds = new Set([...bucket.otelLinkedSpanId.buckets, ...bucket.linkedSpanId.buckets].map(item => item.key));
    for (const spanId of spanIds) {
      incomingSpanLinksSample.set(spanId, {
        agentName: sample[_apm.AGENT_NAME],
        serviceName: bucket.key.serviceName,
        serviceEnvironment: sample[_apm.SERVICE_ENVIRONMENT],
        transactionName: bucket.key.transactionName
      });
    }
  });
  return {
    outgoingSpanLinksSample,
    incomingSpanLinksSample
  };
}
async function fetchTransactionsFromExitSpans({
  apmEventClient,
  exitSpansSample,
  start,
  end
}) {
  const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SERVICE_ENVIRONMENT]);
  const requiredFields = (0, _as_mutable_array.asMutableArray)([_apm.SERVICE_NAME, _apm.AGENT_NAME, _apm.PARENT_ID]);
  const servicesResponse = await apmEventClient.search('get_transactions_from_exit_spans', {
    apm: {
      events: [_common.ProcessorEvent.transaction]
    },
    track_total_hits: false,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _server.termsQuery)(_apm.PARENT_ID, ...exitSpansSample.keys())]
      }
    },
    size: exitSpansSample.size,
    fields: [...requiredFields, ...optionalFields]
  });
  const destinationsBySpanId = new Map(exitSpansSample);
  servicesResponse.hits.hits.forEach(hit => {
    const transaction = (0, _utils.accessKnownApmEventFields)(hit.fields).requireFields(requiredFields);
    const spanId = transaction[_apm.PARENT_ID];
    const destination = destinationsBySpanId.get(spanId);
    if (destination) {
      destinationsBySpanId.set(spanId, {
        ...destination,
        destinationService: {
          agentName: transaction[_apm.AGENT_NAME],
          serviceEnvironment: transaction[_apm.SERVICE_ENVIRONMENT],
          serviceName: transaction[_apm.SERVICE_NAME]
        }
      });
    }
  });
  return Array.from(destinationsBySpanId.values());
}
async function fetchSpansFromSpanLinks({
  apmEventClient,
  outgoingSpanLinksSample,
  incomingSpanLinksSample,
  start,
  end
}) {
  var _servicesResponse$agg;
  const spanIds = new Set([...outgoingSpanLinksSample.keys(), ...incomingSpanLinksSample.keys()]);
  const servicesResponse = await apmEventClient.search('get_spans_for_span_links', {
    apm: {
      events: [_common.ProcessorEvent.span]
    },
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [...(0, _server.rangeQuery)(start, end), ...(0, _server.termsQuery)(_apm.SPAN_ID, ...spanIds)]
      }
    },
    aggs: {
      links: {
        composite: {
          sources: (0, _as_mutable_array.asMutableArray)([{
            serviceName: {
              terms: {
                field: _apm.SERVICE_NAME
              }
            }
          }, {
            spanName: {
              terms: {
                field: _apm.SPAN_NAME
              }
            }
          }]),
          size: MAX_SPAN_LINKS
        },
        aggs: {
          sample: {
            top_metrics: {
              size: 1,
              sort: {
                [_apm.AT_TIMESTAMP]: 'asc'
              },
              metrics: (0, _as_mutable_array.asMutableArray)([{
                field: _apm.SPAN_ID
              }, {
                field: _apm.SPAN_NAME
              }, {
                field: _apm.SPAN_DESTINATION_SERVICE_RESOURCE
              }, {
                field: _apm.SPAN_TYPE
              }, {
                field: _apm.SPAN_SUBTYPE
              }, {
                field: _apm.AGENT_NAME
              }, {
                field: _apm.SERVICE_NAME
              }, {
                field: _apm.SERVICE_ENVIRONMENT
              }])
            }
          }
        }
      }
    }
  });
  const spanLinksDestination = new Map();
  (_servicesResponse$agg = servicesResponse.aggregations) === null || _servicesResponse$agg === void 0 ? void 0 : _servicesResponse$agg.links.buckets.forEach(bucket => {
    var _bucket$sample$top$3;
    const sample = bucket === null || bucket === void 0 ? void 0 : (_bucket$sample$top$3 = bucket.sample.top[0]) === null || _bucket$sample$top$3 === void 0 ? void 0 : _bucket$sample$top$3.metrics;
    if (!sample) {
      return;
    }
    const spanId = sample[_apm.SPAN_ID];
    const serviceFromSample = {
      agentName: sample[_apm.AGENT_NAME],
      serviceEnvironment: sample[_apm.SERVICE_ENVIRONMENT],
      serviceName: sample[_apm.SERVICE_NAME]
    };
    const outgoingDestination = outgoingSpanLinksSample.get(spanId);
    if (outgoingDestination) {
      var _outgoingDestination$;
      spanLinksDestination.set(spanId, {
        ...outgoingDestination,
        spanDestinationServiceResource: (_outgoingDestination$ = outgoingDestination.spanDestinationServiceResource) !== null && _outgoingDestination$ !== void 0 ? _outgoingDestination$ : bucket.key.spanName,
        destinationService: serviceFromSample
      });
    }
    const incomingDestination = incomingSpanLinksSample.get(spanId);
    if (incomingDestination) {
      var _sample$SPAN_DESTINAT2;
      spanLinksDestination.set(spanId, {
        spanId,
        ...serviceFromSample,
        spanDestinationServiceResource: (_sample$SPAN_DESTINAT2 = sample[_apm.SPAN_DESTINATION_SERVICE_RESOURCE]) !== null && _sample$SPAN_DESTINAT2 !== void 0 ? _sample$SPAN_DESTINAT2 : sample[_apm.SPAN_NAME],
        spanType: sample[_apm.SPAN_TYPE],
        spanSubtype: sample[_apm.SPAN_SUBTYPE],
        destinationService: incomingDestination
      });
    }
  });
  return Array.from(spanLinksDestination.values());
}