"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerFleetPolicyCallbacks = registerFleetPolicyCallbacks;
var _lodash = require("lodash");
var _merge_package_policy_with_apm = require("./merge_package_policy_with_apm");
var _add_api_keys_to_policies_if_missing = require("./api_keys/add_api_keys_to_policies_if_missing");
var _get_package_policy_decorators = require("./get_package_policy_decorators");
var _create_internal_es_client = require("../../lib/helpers/create_es_client/create_internal_es_client");
var _get_internal_saved_objects_client = require("../../lib/helpers/get_internal_saved_objects_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function registerFleetPolicyCallbacks({
  logger,
  coreStartPromise,
  plugins
}) {
  if (!plugins.fleet) {
    return;
  }
  const fleetPluginStart = await plugins.fleet.start();
  const {
    getApmIndices
  } = plugins.apmDataAccess.setup;
  const coreStart = await coreStartPromise;
  fleetPluginStart.registerExternalCallback('packagePolicyUpdate', onPackagePolicyCreateOrUpdate({
    fleetPluginStart,
    getApmIndices,
    coreStart
  }));
  fleetPluginStart.registerExternalCallback('packagePolicyCreate', onPackagePolicyCreateOrUpdate({
    fleetPluginStart,
    getApmIndices,
    coreStart
  }));
  fleetPluginStart.registerExternalCallback('packagePolicyDelete', onPackagePolicyDelete({
    coreStart,
    logger
  }));
  fleetPluginStart.registerExternalCallback('packagePolicyPostCreate', onPackagePolicyPostCreate({
    fleet: fleetPluginStart,
    coreStart,
    logger
  }));
}
function onPackagePolicyDelete({
  coreStart,
  logger
}) {
  return async packagePolicies => {
    const promises = packagePolicies.map(async packagePolicy => {
      var _packagePolicy$packag;
      if (((_packagePolicy$packag = packagePolicy.package) === null || _packagePolicy$packag === void 0 ? void 0 : _packagePolicy$packag.name) !== 'apm') {
        return packagePolicy;
      }
      const internalESClient = coreStart.elasticsearch.client.asInternalUser;
      const [agentConfigApiKeyId] = (0, _lodash.get)(packagePolicy, _get_package_policy_decorators.AGENT_CONFIG_API_KEY_PATH, '').split(':');
      const [sourceMapApiKeyId] = (0, _lodash.get)(packagePolicy, _get_package_policy_decorators.SOURCE_MAP_API_KEY_PATH, '').split(':');
      logger.debug(`Deleting API keys: ${agentConfigApiKeyId}, ${sourceMapApiKeyId} (package policy: ${packagePolicy.id})`);
      try {
        await internalESClient.security.invalidateApiKey({
          ids: [agentConfigApiKeyId, sourceMapApiKeyId],
          owner: true
        });
      } catch (e) {
        logger.error(`Failed to delete API keys: ${agentConfigApiKeyId}, ${sourceMapApiKeyId} (package policy: ${packagePolicy.id})`);
      }
    });
    await Promise.all(promises);
  };
}
function onPackagePolicyPostCreate({
  fleet,
  coreStart,
  logger
}) {
  return async (packagePolicy, savedObjectsClient) => {
    var _packagePolicy$packag2;
    if (((_packagePolicy$packag2 = packagePolicy.package) === null || _packagePolicy$packag2 === void 0 ? void 0 : _packagePolicy$packag2.name) !== 'apm') {
      return packagePolicy;
    }

    // add api key to new package policy
    await (0, _add_api_keys_to_policies_if_missing.addApiKeysToPackagePolicyIfMissing)({
      policy: packagePolicy,
      savedObjectsClient,
      coreStart,
      fleet,
      logger
    });
    return packagePolicy;
  };
}

/*
 * This is called when a new package policy is created.
 * It will add an API key to the package policy.
 */
function onPackagePolicyCreateOrUpdate({
  fleetPluginStart,
  getApmIndices,
  coreStart
}) {
  return async packagePolicy => {
    var _packagePolicy$packag3;
    if (((_packagePolicy$packag3 = packagePolicy.package) === null || _packagePolicy$packag3 === void 0 ? void 0 : _packagePolicy$packag3.name) !== 'apm') {
      return packagePolicy;
    }
    const {
      asInternalUser
    } = coreStart.elasticsearch.client;
    const savedObjectsClient = await (0, _get_internal_saved_objects_client.getInternalSavedObjectsClient)(coreStart);
    const apmIndices = await getApmIndices(savedObjectsClient);
    const internalESClient = await (0, _create_internal_es_client.createInternalESClient)({
      debug: false,
      elasticsearchClient: asInternalUser
    });
    return (0, _merge_package_policy_with_apm.decoratePackagePolicyWithAgentConfigAndSourceMap)({
      internalESClient,
      fleetPluginStart,
      packagePolicy,
      apmIndices
    });
  };
}