/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
package org.elasticsearch.xpack.esql.core.expression;

import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.xpack.esql.core.capabilities.Unresolvable;
import org.elasticsearch.xpack.esql.core.capabilities.UnresolvedException;
import org.elasticsearch.xpack.esql.core.tree.NodeInfo;
import org.elasticsearch.xpack.esql.core.tree.Source;
import org.elasticsearch.xpack.esql.core.type.DataType;
import org.elasticsearch.xpack.esql.core.util.CollectionUtils;

import java.io.IOException;
import java.util.List;
import java.util.Objects;

/**
 * An unresolved attribute. We build these while walking the syntax tree and then resolve them into other {@link Attribute} subclasses
 * during analysis. {@link Alias}es whose types are not yet resolved also return {@link UnresolvedAttribute}s when calling
 * {@link Alias#toAttribute()}.
 * <p>
 * Note that the {@link #id()} is respected in {@link #equals(Object)}. Two unresolved attributes with the same name but different
 * {@link NameId} can occur e.g. when resolving {@code EVAL x = 2*x, y = 3*x}. In the first expression, {@code x} is referring to an
 * upstream attribute, while in the second expression it is referring to the alias defined in the first expression. This allows us to
 * distinguish the attributes generated by the {@code EVAL} from attributes referenced by it.
 */
public class UnresolvedAttribute extends Attribute implements Unresolvable {
    private final boolean customMessage;
    protected final String unresolvedMsg;

    // TODO: Check usage of constructors without qualifiers, that's likely where qualifiers need to be plugged into resolution logic.
    public UnresolvedAttribute(Source source, String name) {
        this(source, name, null);
    }

    public UnresolvedAttribute(Source source, String name, @Nullable String unresolvedMessage) {
        this(source, null, name, unresolvedMessage);
    }

    public UnresolvedAttribute(Source source, @Nullable String qualifier, String name, @Nullable String unresolvedMessage) {
        this(source, qualifier, name, null, unresolvedMessage);
    }

    public UnresolvedAttribute(Source source, String name, @Nullable NameId id, @Nullable String unresolvedMessage) {
        this(source, null, name, id, unresolvedMessage);
    }

    @SuppressWarnings("this-escape")
    public UnresolvedAttribute(
        Source source,
        @Nullable String qualifier,
        String name,
        @Nullable NameId id,
        @Nullable String unresolvedMessage
    ) {
        super(source, qualifier, name, id);
        this.customMessage = unresolvedMessage != null;
        this.unresolvedMsg = unresolvedMessage != null ? unresolvedMessage : defaultUnresolvedMessage(null);
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        throw new UnsupportedOperationException("doesn't escape the node");
    }

    @Override
    public String getWriteableName() {
        throw new UnsupportedOperationException("doesn't escape the node");
    }

    @Override
    protected NodeInfo<? extends UnresolvedAttribute> info() {
        return NodeInfo.create(this, UnresolvedAttribute::new, qualifier(), name(), id(), unresolvedMsg);
    }

    public boolean customMessage() {
        return customMessage;
    }

    @Override
    public boolean resolved() {
        return false;
    }

    @Override
    protected Attribute clone(
        Source source,
        String qualifier,
        String name,
        DataType dataType,
        Nullability nullability,
        NameId id,
        boolean synthetic
    ) {
        // TODO: This looks like a bug; making clones should allow for changes.
        return this;
    }

    // Cannot just use the super method because that requires a data type.
    @Override
    public UnresolvedAttribute withId(NameId id) {
        return new UnresolvedAttribute(source(), qualifier(), name(), id, unresolvedMessage());
    }

    public UnresolvedAttribute withUnresolvedMessage(String unresolvedMessage) {
        return new UnresolvedAttribute(source(), qualifier(), name(), id(), unresolvedMessage);
    }

    @Override
    protected TypeResolution resolveType() {
        return new TypeResolution("unresolved attribute [" + name() + "]");
    }

    @Override
    public DataType dataType() {
        throw new UnresolvedException("dataType", this);
    }

    @Override
    public String toString() {
        return UNRESOLVED_PREFIX + qualifiedName();
    }

    @Override
    protected String label() {
        return UNRESOLVED_PREFIX;
    }

    @Override
    public boolean isDimension() {
        // We don't want to use this during the analysis phase, and this class does not exist after analysis
        throw new UnsupportedOperationException("This should never be called before the attribute is resolved");
    }

    @Override
    public boolean isMetric() {
        // We don't want to use this during the analysis phase, and this class does not exist after analysis
        throw new UnsupportedOperationException("This should never be called before the attribute is resolved");
    }

    @Override
    public String nodeString() {
        return toString();
    }

    @Override
    public String unresolvedMessage() {
        return unresolvedMsg;
    }

    public String defaultUnresolvedMessage(@Nullable List<String> potentialMatches) {
        return errorMessage(qualifier() != null ? qualifiedName() : name(), potentialMatches);
    }

    public static String errorMessage(String name, List<String> potentialMatches) {
        String msg = "Unknown column [" + name + "]";
        if (CollectionUtils.isEmpty(potentialMatches) == false) {
            msg += ", did you mean "
                + (potentialMatches.size() == 1 ? "[" + potentialMatches.get(0) + "]" : "any of " + potentialMatches.toString())
                + "?";
        }
        return msg;
    }

    @Override
    protected int innerHashCode(boolean ignoreIds) {
        return Objects.hash(super.innerHashCode(ignoreIds), unresolvedMsg);
    }

    @Override
    protected boolean innerEquals(Object o, boolean ignoreIds) {
        var other = (UnresolvedAttribute) o;
        return super.innerEquals(o, ignoreIds) && Objects.equals(unresolvedMsg, other.unresolvedMsg);
    }
}
