/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.search.retriever.rankdoc;

import java.io.IOException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.LeafReaderContext;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.DocIdSetIterator;
import org.apache.lucene.search.Explanation;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Matches;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.QueryVisitor;
import org.apache.lucene.search.ScoreMode;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.ScorerSupplier;
import org.apache.lucene.search.Weight;
import org.elasticsearch.common.lucene.search.Queries;
import org.elasticsearch.search.rank.RankDoc;

public class RankDocsQuery
extends Query {
    private final RankDoc[] docs;
    private final Query topQuery;
    private final Query tailQuery;
    private final boolean onlyRankDocs;

    public RankDocsQuery(IndexReader reader, RankDoc[] rankDocs, Query[] sources, String[] queryNames, boolean onlyRankDocs) {
        assert (sources.length == queryNames.length);
        this.docs = (RankDoc[])rankDocs.clone();
        Arrays.sort(this.docs, Comparator.comparingInt(a -> a.doc));
        this.topQuery = new TopQuery(this.docs, sources, queryNames, RankDocsQuery.findSegmentStarts(reader, this.docs), reader.getContext().id());
        if (sources.length > 0 && !onlyRankDocs) {
            BooleanQuery.Builder bq = new BooleanQuery.Builder();
            for (Query source : sources) {
                bq.add(source, BooleanClause.Occur.SHOULD);
            }
            this.tailQuery = bq.build();
        } else {
            this.tailQuery = null;
        }
        this.onlyRankDocs = onlyRankDocs;
    }

    private RankDocsQuery(RankDoc[] docs, Query topQuery, Query tailQuery, boolean onlyRankDocs) {
        this.docs = docs;
        this.topQuery = topQuery;
        this.tailQuery = tailQuery;
        this.onlyRankDocs = onlyRankDocs;
    }

    private static int binarySearch(RankDoc[] docs, int fromIndex, int toIndex, int key) {
        return Arrays.binarySearch(docs, fromIndex, toIndex, new RankDoc(key, Float.NaN, -1), Comparator.comparingInt(a -> a.doc));
    }

    private static int[] findSegmentStarts(IndexReader reader, RankDoc[] docs) {
        int[] starts = new int[reader.leaves().size() + 1];
        starts[starts.length - 1] = docs.length;
        if (starts.length == 2) {
            return starts;
        }
        int resultIndex = 0;
        for (int i = 1; i < starts.length - 1; ++i) {
            int upper = ((LeafReaderContext)reader.leaves().get((int)i)).docBase;
            if ((resultIndex = RankDocsQuery.binarySearch(docs, resultIndex, docs.length, upper)) < 0) {
                resultIndex = -1 - resultIndex;
            }
            starts[i] = resultIndex;
        }
        return starts;
    }

    public RankDoc[] rankDocs() {
        return this.docs;
    }

    public Query rewrite(IndexSearcher searcher) throws IOException {
        Query tailRewrite;
        if (this.tailQuery == null) {
            return this.topQuery;
        }
        boolean hasChanged = false;
        Query topRewrite = this.topQuery.rewrite(searcher);
        if (topRewrite != this.topQuery) {
            hasChanged = true;
        }
        if ((tailRewrite = this.tailQuery.rewrite(searcher)) != this.tailQuery) {
            hasChanged = true;
        }
        return hasChanged ? new RankDocsQuery(this.docs, topRewrite, tailRewrite, this.onlyRankDocs) : this;
    }

    public Weight createWeight(IndexSearcher searcher, ScoreMode scoreMode, float boost) throws IOException {
        if (this.tailQuery == null) {
            throw new IllegalArgumentException("[tailQuery] should not be null; maybe missing a rewrite?");
        }
        BooleanQuery combined = new BooleanQuery.Builder().add(this.topQuery, this.onlyRankDocs ? BooleanClause.Occur.MUST : BooleanClause.Occur.SHOULD).add(this.tailQuery, BooleanClause.Occur.FILTER).build();
        final Weight topWeight = this.topQuery.createWeight(searcher, scoreMode, boost);
        final Weight combinedWeight = searcher.rewrite((Query)combined).createWeight(searcher, scoreMode, boost);
        return new Weight(this, this){

            public int count(LeafReaderContext context) throws IOException {
                return combinedWeight.count(context);
            }

            public Explanation explain(LeafReaderContext context, int doc) throws IOException {
                return topWeight.explain(context, doc);
            }

            public boolean isCacheable(LeafReaderContext ctx) {
                return combinedWeight.isCacheable(ctx);
            }

            public Matches matches(LeafReaderContext context, int doc) throws IOException {
                return combinedWeight.matches(context, doc);
            }

            public ScorerSupplier scorerSupplier(LeafReaderContext context) throws IOException {
                return combinedWeight.scorerSupplier(context);
            }
        };
    }

    public String toString(String field) {
        return ((Object)((Object)this)).getClass().getSimpleName() + "{rank_docs:" + Arrays.toString(this.docs) + "}";
    }

    public void visit(QueryVisitor visitor) {
        this.topQuery.visit(visitor);
        if (this.tailQuery != null) {
            this.tailQuery.visit(visitor);
        }
    }

    public boolean equals(Object obj) {
        if (!this.sameClassAs(obj)) {
            return false;
        }
        RankDocsQuery other = (RankDocsQuery)((Object)obj);
        return Objects.equals(this.topQuery, other.topQuery) && Objects.equals(this.tailQuery, other.tailQuery) && this.onlyRankDocs == other.onlyRankDocs;
    }

    public int hashCode() {
        return Objects.hash(this.classHash(), this.topQuery, this.tailQuery, this.onlyRankDocs);
    }

    public static class TopQuery
    extends Query {
        private final RankDoc[] docs;
        private final Query[] sources;
        private final String[] queryNames;
        private final int[] segmentStarts;
        private final Object contextIdentity;

        TopQuery(RankDoc[] docs, Query[] sources, String[] queryNames, int[] segmentStarts, Object contextIdentity) {
            assert (sources.length == queryNames.length);
            this.docs = docs;
            this.sources = sources;
            this.queryNames = queryNames;
            this.segmentStarts = segmentStarts;
            this.contextIdentity = contextIdentity;
            for (RankDoc doc : docs) {
                if (doc.score >= 0.0f) continue;
                throw new IllegalArgumentException("RankDoc scores must be positive values. Missing a normalization step?");
            }
        }

        public Query rewrite(IndexSearcher searcher) throws IOException {
            if (this.docs.length == 0) {
                return Queries.NO_DOCS_INSTANCE;
            }
            boolean changed = false;
            Query[] newSources = new Query[this.sources.length];
            for (int i = 0; i < this.sources.length; ++i) {
                newSources[i] = this.sources[i].rewrite(searcher);
                changed |= newSources[i] != this.sources[i];
            }
            if (changed) {
                return new TopQuery(this.docs, newSources, this.queryNames, this.segmentStarts, this.contextIdentity);
            }
            return this;
        }

        public Weight createWeight(IndexSearcher searcher, ScoreMode scoreMode, float boost) throws IOException {
            if (searcher.getIndexReader().getContext().id() != this.contextIdentity) {
                throw new IllegalStateException("This RankDocsDocQuery was created by a different reader");
            }
            final Weight[] weights = new Weight[this.sources.length];
            for (int i = 0; i < this.sources.length; ++i) {
                weights[i] = this.sources[i].createWeight(searcher, scoreMode, boost);
            }
            return new Weight(this){

                public int count(LeafReaderContext context) {
                    return segmentStarts[context.ord + 1] - segmentStarts[context.ord];
                }

                public Explanation explain(LeafReaderContext context, int doc) throws IOException {
                    int found = RankDocsQuery.binarySearch(docs, 0, docs.length, doc + context.docBase);
                    if (found < 0) {
                        return Explanation.noMatch((String)("doc not found in top " + docs.length + " rank docs"), (Explanation[])new Explanation[0]);
                    }
                    Explanation[] sourceExplanations = new Explanation[sources.length];
                    for (int i = 0; i < sources.length; ++i) {
                        sourceExplanations[i] = weights[i].explain(context, doc);
                    }
                    return docs[found].explain(sourceExplanations, queryNames);
                }

                public ScorerSupplier scorerSupplier(final LeafReaderContext context) throws IOException {
                    Scorer scorer = new Scorer(){
                        final int lower;
                        final int upper;
                        int upTo;
                        float score;
                        {
                            this.lower = segmentStarts[context.ord];
                            this.upper = segmentStarts[context.ord + 1];
                            this.upTo = -1;
                        }

                        public DocIdSetIterator iterator() {
                            return new DocIdSetIterator(){

                                public int docID() {
                                    return this.currentDocId();
                                }

                                public int nextDoc() {
                                    upTo = upTo == -1 ? lower : ++upTo;
                                    return this.currentDocId();
                                }

                                public int advance(int target) throws IOException {
                                    return this.slowAdvance(target);
                                }

                                public long cost() {
                                    return upper - lower;
                                }
                            };
                        }

                        public float getMaxScore(int docId) {
                            if (docId != Integer.MAX_VALUE) {
                                docId += context.docBase;
                            }
                            float maxScore = 0.0f;
                            for (int idx = Math.max(this.lower, this.upTo); idx < this.upper && docs[idx].doc <= docId; ++idx) {
                                maxScore = Math.max(maxScore, docs[idx].score);
                            }
                            return maxScore;
                        }

                        public float score() {
                            return Math.max(docs[this.upTo].score, Float.MIN_VALUE);
                        }

                        public int docID() {
                            return this.currentDocId();
                        }

                        private int currentDocId() {
                            if (this.upTo == -1) {
                                return -1;
                            }
                            if (this.upTo >= this.upper) {
                                return Integer.MAX_VALUE;
                            }
                            return docs[this.upTo].doc - context.docBase;
                        }
                    };
                    return new Weight.DefaultScorerSupplier(scorer);
                }

                public boolean isCacheable(LeafReaderContext ctx) {
                    return true;
                }
            };
        }

        public String toString(String field) {
            return ((Object)((Object)this)).getClass().getSimpleName() + "{rank_docs:" + Arrays.toString(this.docs) + "}";
        }

        public void visit(QueryVisitor visitor) {
            visitor.visitLeaf((Query)this);
        }

        public boolean equals(Object obj) {
            if (!this.sameClassAs(obj)) {
                return false;
            }
            TopQuery other = (TopQuery)((Object)obj);
            return Arrays.equals(this.docs, other.docs) && Arrays.equals(this.segmentStarts, other.segmentStarts) && this.contextIdentity == other.contextIdentity;
        }

        public int hashCode() {
            return Objects.hash(this.classHash(), Arrays.hashCode(this.docs), Arrays.hashCode(this.segmentStarts), this.contextIdentity);
        }
    }
}

