/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingRequest;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.client.internal.OriginSettingClient;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ProjectState;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.cluster.health.ClusterIndexHealth;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.project.ProjectResolver;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.gateway.GatewayService;
import org.elasticsearch.indices.SystemIndexDescriptor;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.xcontent.XContentType;

public class SystemIndexMappingUpdateService
implements ClusterStateListener {
    private static final Logger logger = LogManager.getLogger(SystemIndexMappingUpdateService.class);
    public static final Set<String> MANAGED_SYSTEM_INDEX_SETTING_UPDATE_ALLOWLIST;
    private final SystemIndices systemIndices;
    private final Client client;
    private final ProjectResolver projectResolver;
    private final AtomicBoolean isUpgradeInProgress;

    public SystemIndexMappingUpdateService(SystemIndices systemIndices, Client client, ProjectResolver projectResolver) {
        this.systemIndices = systemIndices;
        this.client = client;
        this.projectResolver = projectResolver;
        this.isUpgradeInProgress = new AtomicBoolean(false);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        ClusterState state = event.state();
        if (state.blocks().hasGlobalBlock(GatewayService.STATE_NOT_RECOVERED_BLOCK)) {
            logger.debug("Waiting until state has been recovered");
            return;
        }
        if (!state.nodes().isLocalNodeElectedMaster()) {
            return;
        }
        if (state.nodes().isMixedVersionCluster()) {
            logger.debug("Skipping system indices up-to-date check as cluster has mixed versions");
            return;
        }
        if (this.isUpgradeInProgress.compareAndSet(false, true)) {
            try (RefCountingRunnable refs = new RefCountingRunnable(() -> this.isUpgradeInProgress.set(false));){
                state.forEachProject(project -> {
                    for (SystemIndexDescriptor systemIndexDescriptor : this.getEligibleDescriptors(project.metadata())) {
                        UpgradeStatus upgradeStatus;
                        try {
                            upgradeStatus = SystemIndexMappingUpdateService.getUpgradeStatus(project, systemIndexDescriptor);
                        }
                        catch (Exception e) {
                            logger.warn(Strings.format((String)"Failed to calculate upgrade status for project [%s]: %s", (Object[])new Object[]{project.projectId(), e.getMessage()}), (Throwable)e);
                            continue;
                        }
                        if (upgradeStatus != UpgradeStatus.NEEDS_MAPPINGS_UPDATE) continue;
                        this.upgradeIndexMappings(project.projectId(), systemIndexDescriptor, ActionListener.releasing(refs.acquire()));
                    }
                });
            }
        } else {
            logger.trace("Update already in progress");
        }
    }

    List<SystemIndexDescriptor> getEligibleDescriptors(ProjectMetadata projectMetadata) {
        return this.systemIndices.getSystemIndexDescriptors().stream().filter(SystemIndexDescriptor::isAutomaticallyManaged).filter(d -> projectMetadata.hasIndexAbstraction(d.getPrimaryIndex())).toList();
    }

    static UpgradeStatus getUpgradeStatus(ProjectState projectState, SystemIndexDescriptor descriptor) {
        State indexState = SystemIndexMappingUpdateService.calculateIndexState(projectState, descriptor);
        String indexDescription = "[" + descriptor.getPrimaryIndex() + "] (alias [" + descriptor.getAliasName() + "])";
        if (indexState == null) {
            logger.debug("Index {} does not exist yet", (Object)indexDescription);
            return UpgradeStatus.UP_TO_DATE;
        }
        if (indexState.indexState == IndexMetadata.State.CLOSE) {
            logger.debug("Index {} is closed. This is likely to prevent some features from functioning correctly", (Object)indexDescription);
            return UpgradeStatus.CLOSED;
        }
        if (indexState.indexHealth == ClusterHealthStatus.RED) {
            logger.debug("Index {} health status is RED, any pending mapping upgrades will wait until this changes", (Object)indexDescription);
            return UpgradeStatus.UNHEALTHY;
        }
        if (!indexState.isIndexUpToDate) {
            logger.debug("Index {} is not on the current version. Features relying on the index will not be available until the index is upgraded", (Object)indexDescription);
            return UpgradeStatus.NEEDS_UPGRADE;
        }
        if (indexState.mappingUpToDate) {
            logger.trace("Index {} is up-to-date, no action required", (Object)indexDescription);
            return UpgradeStatus.UP_TO_DATE;
        }
        logger.info("Index {} mappings are not up-to-date and will be updated", (Object)indexDescription);
        return UpgradeStatus.NEEDS_MAPPINGS_UPDATE;
    }

    private void upgradeIndexMappings(ProjectId projectId, SystemIndexDescriptor descriptor, final ActionListener<AcknowledgedResponse> listener) {
        this.projectResolver.executeOnProject(projectId, () -> {
            final String indexName = descriptor.getPrimaryIndex();
            PutMappingRequest request = new PutMappingRequest(indexName).source(descriptor.getMappings(), XContentType.JSON);
            OriginSettingClient originSettingClient = new OriginSettingClient(this.client, descriptor.getOrigin());
            originSettingClient.admin().indices().putMapping(request, new ActionListener<AcknowledgedResponse>(this){

                @Override
                public void onResponse(AcknowledgedResponse response) {
                    if (!response.isAcknowledged()) {
                        String message = "Put mapping request for [" + indexName + "] was not acknowledged";
                        logger.error(message);
                        listener.onFailure(new ElasticsearchException(message, new Object[0]));
                    } else {
                        listener.onResponse(response);
                    }
                }

                @Override
                public void onFailure(Exception e) {
                    logger.error("Put mapping request for [" + indexName + "] failed", (Throwable)e);
                    listener.onFailure(e);
                }
            });
        });
    }

    static State calculateIndexState(ProjectState state, SystemIndexDescriptor descriptor) {
        ClusterHealthStatus indexHealth;
        IndexMetadata indexMetadata = state.metadata().index(descriptor.getPrimaryIndex());
        if (indexMetadata == null) {
            return null;
        }
        boolean isIndexUpToDate = IndexMetadata.INDEX_FORMAT_SETTING.get(indexMetadata.getSettings()).intValue() == descriptor.getIndexFormat();
        boolean isMappingIsUpToDate = SystemIndexMappingUpdateService.checkIndexMappingUpToDate(descriptor, indexMetadata);
        String concreteIndexName = indexMetadata.getIndex().getName();
        IndexMetadata.State indexState = indexMetadata.getState();
        if (indexState == IndexMetadata.State.CLOSE) {
            indexHealth = null;
            logger.warn("Index [{}] (alias [{}]) is closed. This is likely to prevent some features from functioning correctly", (Object)concreteIndexName, (Object)descriptor.getAliasName());
        } else {
            IndexRoutingTable routingTable = state.routingTable().index(indexMetadata.getIndex());
            indexHealth = new ClusterIndexHealth(indexMetadata, routingTable).getStatus();
        }
        return new State(indexState, indexHealth, isIndexUpToDate, isMappingIsUpToDate);
    }

    private static boolean checkIndexMappingUpToDate(SystemIndexDescriptor descriptor, IndexMetadata indexMetadata) {
        MappingMetadata mappingMetadata = indexMetadata.mapping();
        if (mappingMetadata == null) {
            return false;
        }
        return descriptor.getMappingsVersion().version() <= SystemIndexMappingUpdateService.readMappingVersion(descriptor, mappingMetadata);
    }

    private static int readMappingVersion(SystemIndexDescriptor descriptor, MappingMetadata mappingMetadata) {
        String indexName = descriptor.getPrimaryIndex();
        try {
            Map meta = (Map)mappingMetadata.sourceAsMap().get("_meta");
            if (meta == null) {
                logger.warn("Missing _meta field in mapping [{}] of index [{}], assuming mappings update required", (Object)mappingMetadata.type(), (Object)indexName);
                return -1;
            }
            Object rawVersion = meta.get("managed_index_mappings_version");
            if (rawVersion == null) {
                logger.warn("No value found in mappings for [_meta.{}], assuming mappings update required", (Object)"managed_index_mappings_version");
                return -1;
            }
            if (!(rawVersion instanceof Integer)) {
                logger.warn("Value in [_meta.{}] was not an integer, assuming mappings update required", (Object)"managed_index_mappings_version");
                return -1;
            }
            return (Integer)rawVersion;
        }
        catch (IllegalArgumentException | ElasticsearchParseException e) {
            logger.error(() -> "Cannot parse the mapping for index [" + indexName + "]", (Throwable)e);
            return -1;
        }
    }

    static {
        HashSet<String> allowlist = new HashSet<String>();
        for (IndexMetadata.APIBlock blockType : IndexMetadata.APIBlock.values()) {
            allowlist.add(blockType.settingName());
        }
        MANAGED_SYSTEM_INDEX_SETTING_UPDATE_ALLOWLIST = Collections.unmodifiableSet(allowlist);
    }

    static class State {
        final IndexMetadata.State indexState;
        final ClusterHealthStatus indexHealth;
        final boolean isIndexUpToDate;
        final boolean mappingUpToDate;

        State(IndexMetadata.State indexState, ClusterHealthStatus indexHealth, boolean isIndexUpToDate, boolean mappingUpToDate) {
            this.indexState = indexState;
            this.indexHealth = indexHealth;
            this.isIndexUpToDate = isIndexUpToDate;
            this.mappingUpToDate = mappingUpToDate;
        }
    }

    static enum UpgradeStatus {
        CLOSED,
        UNHEALTHY,
        NEEDS_UPGRADE,
        UP_TO_DATE,
        NEEDS_MAPPINGS_UPDATE;

    }
}

