/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.lucene.search.FieldComparatorSource;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.SortedNumericSortField;
import org.apache.lucene.search.SortedSetSortField;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.fielddata.IndexFieldData;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.search.MultiValueMode;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.search.sort.SortOrder;

public final class IndexSortConfig {
    private static final DeprecationLogger DEPRECATION_LOGGER = DeprecationLogger.getLogger(IndexSortConfig.class);
    public static final Setting<List<String>> INDEX_SORT_FIELD_SETTING = Setting.stringListSettingWithDefaultProvider("index.sort.field", IndexSortConfigDefaults::getDefaultSortFields, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<SortOrder>> INDEX_SORT_ORDER_SETTING = Setting.listSetting("index.sort.order", IndexSortConfigDefaults::getDefaultSortOrder, IndexSortConfig::parseOrderMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<MultiValueMode>> INDEX_SORT_MODE_SETTING = Setting.listSetting("index.sort.mode", IndexSortConfigDefaults::getDefaultSortMode, IndexSortConfig::parseMultiValueMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<String>> INDEX_SORT_MISSING_SETTING = Setting.listSetting("index.sort.missing", IndexSortConfigDefaults::getDefaultSortMissing, IndexSortConfig::validateMissingValue, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    final FieldSortSpec[] sortSpecs;
    private final IndexVersion indexCreatedVersion;
    private final String indexName;
    private final IndexMode indexMode;
    private static final EnumSet<SortField.Type> ALLOWED_INDEX_SORT_TYPES = EnumSet.of(SortField.Type.STRING, SortField.Type.LONG, SortField.Type.INT, SortField.Type.DOUBLE, SortField.Type.FLOAT);

    private static String validateMissingValue(String missing) {
        if (!"_last".equals(missing) && !"_first".equals(missing)) {
            throw new IllegalArgumentException("Illegal missing value:[" + missing + "], must be one of [_last, _first]");
        }
        return missing;
    }

    private static SortOrder parseOrderMode(String value) {
        try {
            return SortOrder.fromString(value);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Illegal sort order:" + value);
        }
    }

    private static MultiValueMode parseMultiValueMode(String value) {
        MultiValueMode mode = MultiValueMode.fromString(value);
        if (mode != MultiValueMode.MAX && mode != MultiValueMode.MIN) {
            throw new IllegalArgumentException("Illegal index sort mode:[" + String.valueOf(mode) + "], must be one of [" + String.valueOf(MultiValueMode.MAX) + ", " + String.valueOf(MultiValueMode.MIN) + "]");
        }
        return mode;
    }

    private static void checkSizeMismatch(String firstKey, List<?> first, String secondKey, List<?> second) {
        if (first.size() != second.size()) {
            throw new IllegalArgumentException(firstKey + ":" + String.valueOf(first) + " " + secondKey + ":" + String.valueOf(second) + ", size mismatch");
        }
    }

    private static void validateSortSettings(Settings settings) {
        if (!INDEX_SORT_FIELD_SETTING.exists(settings)) {
            for (Setting setting : new Setting[]{INDEX_SORT_ORDER_SETTING, INDEX_SORT_MODE_SETTING, INDEX_SORT_MISSING_SETTING}) {
                if (!setting.exists(settings)) continue;
                throw new IllegalArgumentException("setting [" + setting.getKey() + "] requires [" + INDEX_SORT_FIELD_SETTING.getKey() + "] to be configured");
            }
        }
        List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
        List<SortOrder> order = INDEX_SORT_ORDER_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_ORDER_SETTING.getKey(), order);
        List<MultiValueMode> mode = INDEX_SORT_MODE_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_MODE_SETTING.getKey(), mode);
        List<String> missing = INDEX_SORT_MISSING_SETTING.get(settings);
        IndexSortConfig.checkSizeMismatch(INDEX_SORT_FIELD_SETTING.getKey(), fields, INDEX_SORT_MISSING_SETTING.getKey(), missing);
    }

    public IndexSortConfig(IndexSettings indexSettings) {
        Settings settings = indexSettings.getSettings();
        this.indexCreatedVersion = indexSettings.getIndexVersionCreated();
        this.indexName = indexSettings.getIndex().getName();
        this.indexMode = indexSettings.getMode();
        IndexSortConfig.validateSortSettings(settings);
        List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
        this.sortSpecs = (FieldSortSpec[])fields.stream().map(FieldSortSpec::new).toArray(FieldSortSpec[]::new);
        List<SortOrder> orders = INDEX_SORT_ORDER_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].order = orders.get(i);
        }
        List<MultiValueMode> modes = INDEX_SORT_MODE_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].mode = modes.get(i);
        }
        List<String> missingValues = INDEX_SORT_MISSING_SETTING.get(settings);
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            this.sortSpecs[i].missingValue = missingValues.get(i);
        }
    }

    public boolean hasIndexSort() {
        return this.sortSpecs.length > 0;
    }

    public boolean hasPrimarySortOnField(String field) {
        return this.sortSpecs.length > 0 && this.sortSpecs[0].field.equals(field);
    }

    public Sort buildIndexSort(Function<String, MappedFieldType> fieldTypeLookup, BiFunction<MappedFieldType, Supplier<SearchLookup>, IndexFieldData<?>> fieldDataLookup) {
        if (!this.hasIndexSort()) {
            return null;
        }
        SortField[] sortFields = new SortField[this.sortSpecs.length];
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            IndexFieldData<?> fieldData;
            FieldSortSpec sortSpec = this.sortSpecs[i];
            MappedFieldType ft = fieldTypeLookup.apply(sortSpec.field);
            if (ft == null) {
                String err = "unknown index sort field:[" + sortSpec.field + "]";
                if (this.indexMode == IndexMode.TIME_SERIES) {
                    err = err + " required by [" + IndexSettings.MODE.getKey() + "=time_series]";
                }
                throw new IllegalArgumentException(err);
            }
            if (!Objects.equals(ft.name(), sortSpec.field)) {
                if (this.indexCreatedVersion.onOrAfter(IndexVersions.V_7_13_0)) {
                    throw new IllegalArgumentException("Cannot use alias [" + sortSpec.field + "] as an index sort field");
                }
                DEPRECATION_LOGGER.warn(DeprecationCategory.MAPPINGS, "index-sort-aliases", "Index sort for index [" + this.indexName + "] defined on field [" + sortSpec.field + "] which resolves to field [" + ft.name() + "]. You will not be able to define an index sort over aliased fields in new indexes", new Object[0]);
            }
            boolean reverse = sortSpec.order == null ? false : sortSpec.order == SortOrder.DESC;
            MultiValueMode mode = sortSpec.mode;
            if (mode == null) {
                mode = reverse ? MultiValueMode.MAX : MultiValueMode.MIN;
            }
            try {
                fieldData = fieldDataLookup.apply(ft, () -> {
                    throw new UnsupportedOperationException("index sorting not supported on runtime field [" + ft.name() + "]");
                });
            }
            catch (Exception e) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]", e);
            }
            if (fieldData == null) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]");
            }
            sortFields[i] = fieldData.indexSort(this.indexCreatedVersion, sortSpec.missingValue, mode, reverse);
            IndexSortConfig.validateIndexSortField(sortFields[i]);
        }
        return new Sort(sortFields);
    }

    private static void validateIndexSortField(SortField sortField) {
        SortField.Type type = IndexSortConfig.getSortFieldType(sortField);
        if (!ALLOWED_INDEX_SORT_TYPES.contains(type)) {
            throw new IllegalArgumentException("invalid index sort field:[" + sortField.getField() + "]");
        }
    }

    public boolean hasSortOnField(String fieldName) {
        for (FieldSortSpec sortSpec : this.sortSpecs) {
            if (!sortSpec.field.equals(fieldName)) continue;
            return true;
        }
        return false;
    }

    public static SortField.Type getSortFieldType(SortField sortField) {
        if (sortField instanceof SortedSetSortField) {
            return SortField.Type.STRING;
        }
        if (sortField instanceof SortedNumericSortField) {
            return ((SortedNumericSortField)sortField).getNumericType();
        }
        FieldComparatorSource fieldComparatorSource = sortField.getComparatorSource();
        if (fieldComparatorSource instanceof IndexFieldData.XFieldComparatorSource) {
            IndexFieldData.XFieldComparatorSource fcs = (IndexFieldData.XFieldComparatorSource)fieldComparatorSource;
            return fcs.sortType();
        }
        return sortField.getType();
    }

    public static class FieldSortSpec {
        final String field;
        SortOrder order;
        MultiValueMode mode;
        String missingValue;

        FieldSortSpec(String field) {
            this.field = field;
        }

        public String getField() {
            return this.field;
        }

        public SortOrder getOrder() {
            return this.order;
        }
    }

    public static class IndexSortConfigDefaults {
        public static final SortDefault NO_SORT = new SortDefault(Collections.emptyList(), Collections.emptyList(), Collections.emptyList(), Collections.emptyList());
        public static final SortDefault TIME_SERIES_SORT = new SortDefault(List.of("_tsid", "@timestamp"), List.of("asc", "desc"), List.of("min", "max"), List.of("_last", "_last"));
        public static final SortDefault TIMESTAMP_SORT = new SortDefault(List.of("@timestamp"), List.of("desc"), List.of("max"), List.of("_last"));
        public static final SortDefault HOSTNAME_TIMESTAMP_SORT = new SortDefault(List.of("host.name", "@timestamp"), List.of("asc", "desc"), List.of("min", "max"), List.of("_last", "_last"));
        public static final SortDefault HOSTNAME_TIMESTAMP_BWC_SORT;
        public static final SortDefault MESSAGE_PATTERN_TIMESTAMP_SORT;
        public static final SortDefault HOSTNAME_MESSAGE_PATTERN_TIMESTAMP_SORT;

        static SortDefault getSortDefault(Settings settings) {
            if (settings.isEmpty()) {
                return NO_SORT;
            }
            String indexMode = settings.get(IndexSettings.MODE.getKey());
            if (indexMode != null) {
                indexMode = indexMode.toLowerCase(Locale.ROOT);
            }
            if (IndexMode.TIME_SERIES.getName().equals(indexMode)) {
                return TIME_SERIES_SORT;
            }
            if (IndexMode.LOGSDB.getName().equals(indexMode)) {
                IndexVersion version = IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(settings);
                if (version.onOrAfter(IndexVersions.LOGSB_OPTIONAL_SORTING_ON_HOST_NAME) || version.between(IndexVersions.LOGSB_OPTIONAL_SORTING_ON_HOST_NAME_BACKPORT, IndexVersions.UPGRADE_TO_LUCENE_10_0_0)) {
                    boolean sortOnHostName = settings.getAsBoolean(IndexSettings.LOGSDB_SORT_ON_HOST_NAME.getKey(), false);
                    boolean sortOnMessageTemplate = settings.getAsBoolean(IndexSettings.LOGSDB_SORT_ON_MESSAGE_TEMPLATE.getKey(), false);
                    if (sortOnHostName && sortOnMessageTemplate) {
                        return HOSTNAME_MESSAGE_PATTERN_TIMESTAMP_SORT;
                    }
                    if (sortOnHostName) {
                        return HOSTNAME_TIMESTAMP_SORT;
                    }
                    if (sortOnMessageTemplate) {
                        return MESSAGE_PATTERN_TIMESTAMP_SORT;
                    }
                    return TIMESTAMP_SORT;
                }
                return HOSTNAME_TIMESTAMP_BWC_SORT;
            }
            return NO_SORT;
        }

        public static List<String> getDefaultSortFields(Settings settings) {
            return IndexSortConfigDefaults.getSortDefault(settings).fields();
        }

        public static List<String> getDefaultSortOrder(Settings settings) {
            if (!settings.hasValue(INDEX_SORT_FIELD_SETTING.getKey())) {
                return IndexSortConfigDefaults.getSortDefault(settings).order();
            }
            List<String> sortFields = settings.getAsList(INDEX_SORT_FIELD_SETTING.getKey());
            ArrayList<String> order = new ArrayList<String>(sortFields.size());
            for (int i = 0; i < sortFields.size(); ++i) {
                order.add("asc");
            }
            return order;
        }

        public static List<String> getDefaultSortMode(Settings settings) {
            if (!settings.hasValue(INDEX_SORT_FIELD_SETTING.getKey())) {
                return IndexSortConfigDefaults.getSortDefault(settings).mode();
            }
            List<String> sortFields = settings.getAsList(INDEX_SORT_FIELD_SETTING.getKey());
            List<String> sortOrder = settings.getAsList(INDEX_SORT_ORDER_SETTING.getKey(), null);
            ArrayList<String> mode = new ArrayList<String>(sortFields.size());
            for (int i = 0; i < sortFields.size(); ++i) {
                if (sortOrder != null && sortOrder.get(i).equals(SortOrder.DESC.toString())) {
                    mode.add("max");
                    continue;
                }
                mode.add("min");
            }
            return mode;
        }

        public static List<String> getDefaultSortMissing(Settings settings) {
            if (!settings.hasValue(INDEX_SORT_FIELD_SETTING.getKey())) {
                return IndexSortConfigDefaults.getSortDefault(settings).missing();
            }
            List<String> sortFields = settings.getAsList(INDEX_SORT_FIELD_SETTING.getKey());
            ArrayList<String> missing = new ArrayList<String>(sortFields.size());
            for (int i = 0; i < sortFields.size(); ++i) {
                missing.add("_last");
            }
            return missing;
        }

        static {
            MESSAGE_PATTERN_TIMESTAMP_SORT = new SortDefault(List.of("message.template_id", "@timestamp"), List.of("asc", "desc"), List.of("min", "max"), List.of("_last", "_last"));
            HOSTNAME_MESSAGE_PATTERN_TIMESTAMP_SORT = new SortDefault(List.of("host.name", "message.template_id", "@timestamp"), List.of("asc", "asc", "desc"), List.of("min", "min", "max"), List.of("_last", "_last", "_last"));
            HOSTNAME_TIMESTAMP_BWC_SORT = new SortDefault(List.of("host.name", "@timestamp"), List.of("asc", "asc"), List.of("min", "min"), List.of("_last", "_last"));
        }

        public record SortDefault(List<String> fields, List<String> order, List<String> mode, List<String> missing) {
            public SortDefault {
                assert (fields.size() == order.size());
                assert (fields.size() == mode.size());
                assert (fields.size() == missing.size());
            }
        }
    }
}

