/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.blobstore;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.common.blobstore.OperationPurpose;
import org.elasticsearch.core.IOUtils;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.repositories.blobstore.RequestedRangeNotSatisfiedException;

public abstract class RetryingInputStream<V>
extends InputStream {
    private static final Logger logger = LogManager.getLogger(RetryingInputStream.class);
    public static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final BlobStoreServices<V> blobStoreServices;
    private final OperationPurpose purpose;
    private final long start;
    private final long end;
    private final List<Exception> failures;
    protected SingleAttemptInputStream<V> currentStream;
    private long offset = 0L;
    private int attempt = 1;
    private int failuresAfterMeaningfulProgress = 0;
    private boolean closed = false;

    protected RetryingInputStream(BlobStoreServices<V> blobStoreServices, OperationPurpose purpose) throws IOException {
        this(blobStoreServices, purpose, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    protected RetryingInputStream(BlobStoreServices<V> blobStoreServices, OperationPurpose purpose, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobStoreServices = blobStoreServices;
        this.purpose = purpose;
        this.failures = new ArrayList<Exception>(10);
        this.start = start;
        this.end = end;
        int initialAttempt = this.attempt;
        this.openStreamWithRetry();
        this.maybeLogAndRecordMetricsForSuccess(initialAttempt, StreamAction.OPEN);
    }

    private void openStreamWithRetry() throws IOException {
        while (true) {
            if (this.offset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) assert (this.start + this.offset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.offset + " end=" + this.end;
            try {
                this.currentStream = this.blobStoreServices.getInputStream(this.currentStream != null ? (Object)this.currentStream.getVersion() : null, Math.addExact(this.start, this.offset), this.end);
                return;
            }
            catch (NoSuchFileException | RequestedRangeNotSatisfiedException e) {
                throw this.addSuppressedExceptions(e);
            }
            catch (RuntimeException e) {
                this.retryOrAbortOnOpen(e);
                continue;
            }
            catch (IOException e) {
                this.retryOrAbortOnOpen(e);
                continue;
            }
            break;
        }
    }

    private <T extends Exception> void retryOrAbortOnOpen(T exception) throws T {
        if (this.attempt == 1) {
            this.blobStoreServices.onRetryStarted(StreamAction.OPEN);
        }
        long delayInMillis = this.maybeLogAndComputeRetryDelay(StreamAction.OPEN, exception);
        this.delayBeforeRetry(StreamAction.OPEN, exception, delayInMillis);
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int result = this.currentStream.read();
                if (result != -1) {
                    ++this.offset;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, StreamAction.READ);
                return result;
            }
            catch (IOException e) {
                this.retryOrAbortOnRead(initialAttempt, e);
                continue;
            }
            catch (RuntimeException e) {
                this.retryOrAbortOnRead(initialAttempt, e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        int initialAttempt = this.attempt;
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead != -1) {
                    this.offset += (long)bytesRead;
                }
                this.maybeLogAndRecordMetricsForSuccess(initialAttempt, StreamAction.READ);
                return bytesRead;
            }
            catch (IOException e) {
                this.retryOrAbortOnRead(initialAttempt, e);
                continue;
            }
            catch (RuntimeException e) {
                this.retryOrAbortOnRead(initialAttempt, e);
                continue;
            }
            break;
        }
    }

    private <T extends Exception> void retryOrAbortOnRead(int initialAttempt, T exception) throws T, IOException {
        if (this.attempt == initialAttempt) {
            this.blobStoreServices.onRetryStarted(StreamAction.READ);
        }
        this.reopenStreamOrFail(exception);
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using RetryingInputStream after close";
            throw new IllegalStateException("Stream is closed");
        }
    }

    private <T extends Exception> void reopenStreamOrFail(T e) throws T, IOException {
        long meaningfulProgressSize = this.blobStoreServices.getMeaningfulProgressSize();
        if (this.currentStreamProgress() >= meaningfulProgressSize) {
            ++this.failuresAfterMeaningfulProgress;
        }
        long delayInMillis = this.maybeLogAndComputeRetryDelay(StreamAction.READ, e);
        IOUtils.closeWhileHandlingException(this.currentStream);
        this.delayBeforeRetry(StreamAction.READ, e, delayInMillis);
        this.openStreamWithRetry();
    }

    private <T extends Exception> long maybeLogAndComputeRetryDelay(StreamAction action, T e) throws T {
        if (!this.shouldRetry(action, e, this.attempt)) {
            T finalException = this.addSuppressedExceptions(e);
            this.logForFailure(action, finalException);
            throw finalException;
        }
        this.logForRetry(Integer.bitCount(this.attempt) == 1 ? Level.INFO : Level.DEBUG, action, e);
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        long delayInMillis = this.getRetryDelayInMillis();
        ++this.attempt;
        return delayInMillis;
    }

    private void logForFailure(StreamAction action, Exception e) {
        logger.warn(() -> Strings.format((String)"failed %s [%s] at offset [%s] with purpose [%s]", (Object[])new Object[]{action.getPresentTense(), this.blobStoreServices.getBlobDescription(), this.start + this.offset, this.purpose.getKey()}), (Throwable)e);
    }

    private void logForRetry(Level level, StreamAction action, Exception e) {
        logger.log(level, () -> Strings.format((String)"failed %s [%s] at offset [%s] with purpose [%s]; this was attempt [%s] to read this blob which yielded [%s] bytes; in total [%s] of the attempts to read this blob have made meaningful progress and do not count towards the maximum number of retries; the maximum number of read attempts which do not make meaningful progress is [%s]", (Object[])new Object[]{action.getPresentTense(), this.blobStoreServices.getBlobDescription(), this.start + this.offset, this.purpose.getKey(), this.attempt, this.currentStreamProgress(), this.failuresAfterMeaningfulProgress, this.maxRetriesForNoMeaningfulProgress()}), (Throwable)e);
    }

    private void maybeLogAndRecordMetricsForSuccess(int initialAttempt, StreamAction action) {
        if (this.attempt > initialAttempt) {
            int numberOfRetries = this.attempt - initialAttempt;
            logger.info("successfully {} input stream for [{}] with purpose [{}] after [{}] retries", (Object)action.getPastTense(), (Object)this.blobStoreServices.getBlobDescription(), (Object)this.purpose.getKey(), (Object)numberOfRetries);
            this.blobStoreServices.onRetrySucceeded(action, numberOfRetries);
        }
    }

    private long currentStreamProgress() {
        if (this.currentStream == null) {
            return 0L;
        }
        return Math.subtractExact(Math.addExact(this.start, this.offset), this.currentStream.getFirstOffset());
    }

    private boolean shouldRetry(StreamAction action, Exception exception, int attempt) {
        if (!this.blobStoreServices.isRetryableException(action, exception)) {
            return false;
        }
        if (this.purpose == OperationPurpose.REPOSITORY_ANALYSIS) {
            return false;
        }
        if (this.purpose == OperationPurpose.INDICES) {
            return true;
        }
        int maxAttempts = this.blobStoreServices.getMaxRetries() + 1;
        return attempt < maxAttempts + this.failuresAfterMeaningfulProgress;
    }

    private int maxRetriesForNoMeaningfulProgress() {
        return this.purpose == OperationPurpose.INDICES ? Integer.MAX_VALUE : this.blobStoreServices.getMaxRetries() + 1;
    }

    private void delayBeforeRetry(StreamAction action, Exception exception, long delayInMillis) {
        try {
            assert (this.shouldRetry(action, exception, this.attempt - 1)) : "should not have retried";
            Thread.sleep(delayInMillis);
        }
        catch (InterruptedException e) {
            logger.info("retrying input stream delay interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    protected long getRetryDelayInMillis() {
        return 10L << Math.min(this.attempt - 1, 10);
    }

    @Override
    public void close() throws IOException {
        try {
            this.currentStream.close();
        }
        finally {
            this.closed = true;
        }
    }

    @Override
    public long skip(long n) throws IOException {
        return super.skip(n);
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (Exception failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    protected static interface BlobStoreServices<V> {
        public SingleAttemptInputStream<V> getInputStream(@Nullable V var1, long var2, long var4) throws IOException;

        public void onRetryStarted(StreamAction var1);

        public void onRetrySucceeded(StreamAction var1, long var2);

        public long getMeaningfulProgressSize();

        public int getMaxRetries();

        public String getBlobDescription();

        public boolean isRetryableException(StreamAction var1, Exception var2);
    }

    public static enum StreamAction {
        OPEN("open", "opening"),
        READ("read", "reading");

        private final String pastTense;
        private final String presentTense;

        private StreamAction(String pastTense, String presentTense) {
            this.pastTense = pastTense;
            this.presentTense = presentTense;
        }

        public String getPastTense() {
            return this.pastTense;
        }

        public String getPresentTense() {
            return this.presentTense;
        }
    }

    protected static final class SingleAttemptInputStream<V>
    extends FilterInputStream {
        private final long firstOffset;
        private final V version;

        public SingleAttemptInputStream(InputStream in, long firstOffset, V version) {
            super(in);
            this.firstOffset = firstOffset;
            this.version = version;
        }

        public long getFirstOffset() {
            return this.firstOffset;
        }

        public V getVersion() {
            return this.version;
        }

        public <T> T unwrap(Class<T> clazz) {
            return clazz.cast(this.in);
        }
    }
}

