/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.metadata;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ResourceAlreadyExistsException;
import org.elasticsearch.TransportVersion;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.alias.Alias;
import org.elasticsearch.action.admin.indices.create.CreateIndexClusterStateUpdateRequest;
import org.elasticsearch.action.admin.indices.shrink.ResizeType;
import org.elasticsearch.action.support.ActiveShardCount;
import org.elasticsearch.action.support.ActiveShardsObserver;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.action.support.master.ShardsAcknowledgedResponse;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.ProjectState;
import org.elasticsearch.cluster.block.ClusterBlockLevel;
import org.elasticsearch.cluster.block.ClusterBlocks;
import org.elasticsearch.cluster.metadata.AliasMetadata;
import org.elasticsearch.cluster.metadata.AliasValidator;
import org.elasticsearch.cluster.metadata.ComponentTemplate;
import org.elasticsearch.cluster.metadata.ComposableIndexTemplate;
import org.elasticsearch.cluster.metadata.DataStreamFailureStoreDefinition;
import org.elasticsearch.cluster.metadata.DiffableStringMap;
import org.elasticsearch.cluster.metadata.IndexAbstraction;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexTemplateMetadata;
import org.elasticsearch.cluster.metadata.MappingMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.MetadataIndexTemplateService;
import org.elasticsearch.cluster.metadata.ProjectId;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.GlobalRoutingTable;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.ShardRoutingRoleStrategy;
import org.elasticsearch.cluster.routing.ShardRoutingState;
import org.elasticsearch.cluster.routing.allocation.AllocationService;
import org.elasticsearch.cluster.routing.allocation.DataTier;
import org.elasticsearch.cluster.routing.allocation.allocator.AllocationActionListener;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Priority;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.ValidationException;
import org.elasticsearch.common.compress.CompressedXContent;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.IndexScopedSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.xcontent.XContentHelper;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.PathUtils;
import org.elasticsearch.core.SuppressForbidden;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexModule;
import org.elasticsearch.index.IndexNotFoundException;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.IndexSettingProvider;
import org.elasticsearch.index.IndexSettingProviders;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexSortConfig;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.mapper.DocumentMapper;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.query.SearchExecutionContext;
import org.elasticsearch.indices.IndexCreationException;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.indices.InvalidIndexNameException;
import org.elasticsearch.indices.ShardLimitValidator;
import org.elasticsearch.indices.SystemIndexDescriptor;
import org.elasticsearch.indices.SystemIndices;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.xcontent.NamedXContentRegistry;

public class MetadataCreateIndexService {
    public static final Setting<Priority> CREATE_INDEX_PRIORITY_SETTING = Setting.enumSetting(Priority.class, "cluster.service.create_index.priority", Priority.URGENT, Setting.Property.NodeScope);
    private static final Logger logger = LogManager.getLogger(MetadataCreateIndexService.class);
    private static final DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(MetadataCreateIndexService.class);
    public static final int MAX_INDEX_NAME_BYTES = 255;
    public static final String USE_INDEX_REFRESH_BLOCK_SETTING_NAME = "stateless.indices.use_refresh_block_upon_index_creation";
    private final Settings settings;
    private final ClusterService clusterService;
    private final IndicesService indicesService;
    private final AllocationService allocationService;
    private final Environment env;
    private final IndexScopedSettings indexScopedSettings;
    private final NamedXContentRegistry xContentRegistry;
    private final SystemIndices systemIndices;
    private final ShardLimitValidator shardLimitValidator;
    private final boolean forbidPrivateIndexSettings;
    private final Set<IndexSettingProvider> indexSettingProviders;
    private final ThreadPool threadPool;
    private final ClusterBlocksTransformer blocksTransformerUponIndexCreation;
    private final Priority clusterStateUpdateTaskPriority;
    private static final Set<String> UNMODIFIABLE_SETTINGS_DURING_RESIZE = Set.of(IndexSettings.MODE.getKey(), IndexSettings.INDEX_MAPPER_SOURCE_MODE_SETTING.getKey(), IndexSettings.RECOVERY_USE_SYNTHETIC_SOURCE_SETTING.getKey(), IndexSortConfig.INDEX_SORT_FIELD_SETTING.getKey(), IndexSortConfig.INDEX_SORT_ORDER_SETTING.getKey(), IndexSortConfig.INDEX_SORT_MODE_SETTING.getKey(), IndexSortConfig.INDEX_SORT_MISSING_SETTING.getKey());

    public MetadataCreateIndexService(Settings settings, ClusterService clusterService, IndicesService indicesService, AllocationService allocationService, ShardLimitValidator shardLimitValidator, Environment env, IndexScopedSettings indexScopedSettings, ThreadPool threadPool, NamedXContentRegistry xContentRegistry, SystemIndices systemIndices, boolean forbidPrivateIndexSettings, IndexSettingProviders indexSettingProviders) {
        this.settings = settings;
        this.clusterService = clusterService;
        this.indicesService = indicesService;
        this.allocationService = allocationService;
        this.env = env;
        this.indexScopedSettings = indexScopedSettings;
        this.xContentRegistry = xContentRegistry;
        this.systemIndices = systemIndices;
        this.forbidPrivateIndexSettings = forbidPrivateIndexSettings;
        this.shardLimitValidator = shardLimitValidator;
        this.indexSettingProviders = indexSettingProviders.getIndexSettingProviders();
        this.threadPool = threadPool;
        this.blocksTransformerUponIndexCreation = MetadataCreateIndexService.createClusterBlocksTransformerForIndexCreation(settings);
        this.clusterStateUpdateTaskPriority = CREATE_INDEX_PRIORITY_SETTING.get(settings);
    }

    public static void validateIndexName(String index, ProjectMetadata projectMetadata, RoutingTable routingTable) {
        MetadataCreateIndexService.validateIndexOrAliasName(index, InvalidIndexNameException::new);
        if (!index.toLowerCase(Locale.ROOT).equals(index)) {
            throw new InvalidIndexNameException(index, "must be lowercase");
        }
        if (routingTable.hasIndex(index)) {
            throw new ResourceAlreadyExistsException(routingTable.index(index).getIndex());
        }
        if (projectMetadata.hasIndex(index)) {
            throw new ResourceAlreadyExistsException(projectMetadata.index(index).getIndex());
        }
        if (projectMetadata.hasAlias(index)) {
            throw new InvalidIndexNameException(index, "already exists as alias");
        }
    }

    public boolean validateDotIndex(String index, @Nullable Boolean isHidden) {
        boolean isSystem = false;
        if (index.charAt(0) == '.') {
            isSystem = this.systemIndices.isSystemName(index);
            if (isSystem) {
                logger.trace("index [{}] is a system index", (Object)index);
            } else if (isHidden.booleanValue()) {
                logger.trace("index [{}] is a hidden index", (Object)index);
            } else {
                deprecationLogger.warn(DeprecationCategory.INDICES, "index_name_starts_with_dot", "index name [{}] starts with a dot '.', in the next major version, index names starting with a dot are reserved for hidden indices and system indices", index);
            }
        }
        return isSystem;
    }

    public SystemIndices getSystemIndices() {
        return this.systemIndices;
    }

    public static void validateIndexOrAliasName(String index, BiFunction<String, String, ? extends RuntimeException> exceptionCtor) {
        if (index == null || index.isEmpty()) {
            throw exceptionCtor.apply(index, "must not be empty");
        }
        if (!Strings.validFileName(index)) {
            throw exceptionCtor.apply(index, "must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
        if (index.contains("#")) {
            throw exceptionCtor.apply(index, "must not contain '#'");
        }
        if (index.contains(":")) {
            throw exceptionCtor.apply(index, "must not contain ':'");
        }
        if (index.charAt(0) == '_' || index.charAt(0) == '-' || index.charAt(0) == '+') {
            throw exceptionCtor.apply(index, "must not start with '_', '-', or '+'");
        }
        int byteCount = index.getBytes(StandardCharsets.UTF_8).length;
        if (byteCount > 255) {
            throw exceptionCtor.apply(index, "index name is too long, (" + byteCount + " > 255)");
        }
        if (index.equals(".") || index.equals("..")) {
            throw exceptionCtor.apply(index, "must not be '.' or '..'");
        }
    }

    public void createIndex(TimeValue masterNodeTimeout, TimeValue ackTimeout, @Nullable TimeValue waitForActiveShardsTimeout, CreateIndexClusterStateUpdateRequest request, ActionListener<ShardsAcknowledgedResponse> listener) {
        logger.trace("createIndex[{}]", (Object)request);
        this.onlyCreateIndex(masterNodeTimeout, ackTimeout, request, listener.delegateFailureAndWrap((delegate, response) -> {
            if (response.isAcknowledged()) {
                logger.trace("[{}] index creation in project [{}] acknowledged, waiting for active shards [{}]", (Object)request.index(), (Object)request.projectId(), (Object)request.waitForActiveShards());
                ActiveShardsObserver.waitForActiveShards(this.clusterService, request.projectId(), new String[]{request.index()}, request.waitForActiveShards(), waitForActiveShardsTimeout, delegate.map(shardsAcknowledged -> {
                    if (!shardsAcknowledged.booleanValue()) {
                        logger.debug("[{}] index created, but the operation timed out while waiting for enough shards to be started.", (Object)request.index());
                    } else {
                        logger.trace("[{}] index created and shards acknowledged", (Object)request.index());
                    }
                    return ShardsAcknowledgedResponse.of(true, shardsAcknowledged);
                }));
            } else {
                logger.trace("index creation not acknowledged for [{}]", (Object)request);
                delegate.onResponse(ShardsAcknowledgedResponse.NOT_ACKNOWLEDGED);
            }
        }));
    }

    private void onlyCreateIndex(TimeValue masterNodeTimeout, TimeValue ackTimeout, final CreateIndexClusterStateUpdateRequest request, ActionListener<AcknowledgedResponse> listener) {
        try {
            this.normalizeRequestSetting(request);
        }
        catch (Exception e) {
            listener.onFailure(e);
            return;
        }
        final AllocationActionListener<AcknowledgedResponse> delegate = new AllocationActionListener<AcknowledgedResponse>(listener, this.threadPool.getThreadContext());
        this.submitUnbatchedTask("create-index [" + request.index() + "], in project [" + String.valueOf(request.projectId()) + "], cause [" + request.cause() + "]", new AckedClusterStateUpdateTask(this.clusterStateUpdateTaskPriority, masterNodeTimeout, ackTimeout, delegate.clusterStateUpdate()){

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                return MetadataCreateIndexService.this.applyCreateIndexRequest(currentState, request, false, null, delegate.reroute());
            }

            @Override
            public void onFailure(Exception e) {
                if (e instanceof ResourceAlreadyExistsException) {
                    logger.trace(() -> "[" + request.index() + "] failed to create, in project [" + String.valueOf(request.projectId()) + "]", (Throwable)e);
                } else {
                    logger.debug(() -> "[" + request.index() + "] failed to create, in project [" + String.valueOf(request.projectId()) + "]", (Throwable)e);
                }
                super.onFailure(e);
            }
        });
    }

    @SuppressForbidden(reason="legacy usage of unbatched task")
    private void submitUnbatchedTask(String source, ClusterStateUpdateTask task) {
        this.clusterService.submitUnbatchedStateUpdateTask(source, task);
    }

    private void normalizeRequestSetting(CreateIndexClusterStateUpdateRequest createIndexClusterStateRequest) {
        Settings build = Settings.builder().put(createIndexClusterStateRequest.settings()).normalizePrefix("index.").build();
        this.indexScopedSettings.validate(build, true);
        createIndexClusterStateRequest.settings(build);
    }

    public ClusterState applyCreateIndexRequest(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, BiConsumer<ProjectMetadata.Builder, IndexMetadata> metadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        String name;
        IndexMetadata sourceMetadata;
        this.normalizeRequestSetting(request);
        logger.trace("executing IndexCreationTask for [{}] against cluster state version [{}]", (Object)request, (Object)currentState.version());
        ProjectMetadata projectMetadata = currentState.getMetadata().projects().get(request.projectId());
        RoutingTable routingTable = currentState.globalRoutingTable().routingTables().get(request.projectId());
        if (projectMetadata == null || routingTable == null) {
            throw new IndexCreationException(request.index(), new IllegalArgumentException("no such project [" + String.valueOf(request.projectId()) + "]"));
        }
        this.validate(request, projectMetadata, routingTable);
        Index recoverFromIndex = request.recoverFrom();
        IndexMetadata indexMetadata = sourceMetadata = recoverFromIndex == null ? null : projectMetadata.getIndexSafe(recoverFromIndex);
        if (sourceMetadata != null) {
            return this.applyCreateIndexRequestWithExistingMetadata(currentState, request, silent, sourceMetadata, metadataTransformer, rerouteListener);
        }
        String string = name = request.dataStreamName() != null ? request.dataStreamName() : request.index();
        if (request.systemDataStreamDescriptor() != null) {
            return this.applyCreateIndexRequestForSystemDataStream(currentState, request, silent, metadataTransformer, rerouteListener);
        }
        SystemIndexDescriptor descriptor = this.systemIndices.findMatchingDescriptor(request.index());
        if (Objects.nonNull(descriptor) && !descriptor.allowsTemplates()) {
            return this.applyCreateIndexRequestForSystemIndex(currentState, request, silent, descriptor.getIndexPattern(), rerouteListener);
        }
        Boolean isHiddenFromRequest = IndexMetadata.INDEX_HIDDEN_SETTING.exists(request.settings()) ? IndexMetadata.INDEX_HIDDEN_SETTING.get(request.settings()) : null;
        ComposableIndexTemplate templateFromRequest = request.matchingTemplate();
        if (templateFromRequest != null) {
            return this.applyCreateIndexRequestWithV2Template(currentState, request, silent, templateFromRequest, metadataTransformer, rerouteListener);
        }
        String v2Template = MetadataIndexTemplateService.findV2Template(projectMetadata, name, isHiddenFromRequest != null && isHiddenFromRequest != false);
        if (v2Template != null) {
            return this.applyCreateIndexRequestWithV2Template(currentState, request, silent, v2Template, metadataTransformer, rerouteListener);
        }
        List<IndexTemplateMetadata> v1Templates = MetadataIndexTemplateService.findV1Templates(projectMetadata, request.index(), isHiddenFromRequest);
        if (v1Templates.size() > 1) {
            deprecationLogger.warn(DeprecationCategory.TEMPLATES, "index_template_multiple_match", "index [{}] matches multiple legacy templates [{}], composable templates will only match a single template", request.index(), v1Templates.stream().map(IndexTemplateMetadata::name).sorted().collect(Collectors.joining(", ")));
        }
        return this.applyCreateIndexRequestWithV1Templates(currentState, request, silent, v1Templates, metadataTransformer, rerouteListener);
    }

    public ClusterState applyCreateIndexRequest(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, ActionListener<Void> rerouteListener) throws Exception {
        return this.applyCreateIndexRequest(currentState, request, silent, null, rerouteListener);
    }

    private ClusterState applyCreateIndexWithTemporaryService(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, IndexMetadata sourceMetadata, IndexMetadata temporaryIndexMeta, List<CompressedXContent> mappings, Function<IndexService, List<AliasMetadata>> aliasSupplier, List<String> templatesApplied, BiConsumer<ProjectMetadata.Builder, IndexMetadata> metadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        assert (!this.indicesService.hasIndex(temporaryIndexMeta.getIndex())) : Strings.format("Index [%s] already exists", temporaryIndexMeta.getIndex().getName());
        return (ClusterState)this.indicesService.withTempIndexService(temporaryIndexMeta, indexService -> {
            IndexMetadata indexMetadata;
            if (sourceMetadata == null) {
                try {
                    MetadataCreateIndexService.updateIndexMappingsAndBuildSortOrder(indexService, request, mappings);
                }
                catch (Exception e) {
                    logger.log(silent ? Level.DEBUG : Level.INFO, "failed on parsing mappings on index creation [{}]", (Object)request.index(), (Object)e);
                    throw e;
                }
            }
            List aliases = (List)aliasSupplier.apply((IndexService)indexService);
            try {
                indexMetadata = MetadataCreateIndexService.buildIndexMetadata(request.index(), aliases, indexService.mapperService()::documentMapper, temporaryIndexMeta.getSettings(), temporaryIndexMeta.getRoutingNumShards(), sourceMetadata, temporaryIndexMeta.isSystem(), temporaryIndexMeta.getCustomData());
            }
            catch (Exception e) {
                logger.info("failed to build index metadata [{}]", (Object)request.index());
                throw e;
            }
            logger.log(silent ? Level.DEBUG : Level.INFO, "creating index [{}] in project [{}], cause [{}], templates {}, shards [{}]/[{}]", (Object)request.index(), (Object)request.projectId(), (Object)request.cause(), (Object)templatesApplied, (Object)indexMetadata.getNumberOfShards(), (Object)indexMetadata.getNumberOfReplicas());
            indexService.getIndexEventListener().beforeIndexAddedToCluster(indexMetadata.getIndex(), indexMetadata.getSettings());
            ClusterState updated = MetadataCreateIndexService.clusterStateCreateIndex(currentState, request.projectId(), indexMetadata, metadataTransformer, this.blocksTransformerUponIndexCreation, this.allocationService.getShardRoutingRoleStrategy());
            assert (this.assertHasRefreshBlock(indexMetadata, updated.projectState(request.projectId())));
            if (request.performReroute()) {
                updated = this.allocationService.reroute(updated, "index [" + indexMetadata.getIndex().getName() + "] created in project [" + String.valueOf(request.projectId()) + "]", rerouteListener);
            }
            return updated;
        });
    }

    private IndexMetadata buildAndValidateTemporaryIndexMetadata(Settings aggregatedIndexSettings, CreateIndexClusterStateUpdateRequest request, int routingNumShards) {
        boolean isHiddenAfterTemplates = IndexMetadata.INDEX_HIDDEN_SETTING.get(aggregatedIndexSettings);
        boolean isSystem = this.validateDotIndex(request.index(), isHiddenAfterTemplates);
        Settings.Builder settingsBuilder = Settings.builder().put(aggregatedIndexSettings);
        settingsBuilder.remove(IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.getKey());
        Settings indexSettings = settingsBuilder.build();
        IndexMetadata.Builder tmpImdBuilder = IndexMetadata.builder(request.index());
        tmpImdBuilder.setRoutingNumShards(routingNumShards);
        tmpImdBuilder.settings(indexSettings);
        tmpImdBuilder.transportVersion(TransportVersion.current());
        tmpImdBuilder.system(isSystem);
        IndexMetadata tempMetadata = tmpImdBuilder.build();
        MetadataCreateIndexService.validateActiveShardCount(request.waitForActiveShards(), tempMetadata);
        return tempMetadata;
    }

    private ClusterState applyCreateIndexRequestWithV1Templates(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, List<IndexTemplateMetadata> templates, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        logger.debug("applying create index request using legacy templates {}", templates.stream().map(IndexTemplateMetadata::name).toList());
        Map<String, Object> mappingsMap = MetadataCreateIndexService.parseV1Mappings(request.mappings(), templates.stream().map(IndexTemplateMetadata::getMappings).collect(Collectors.toList()), this.xContentRegistry);
        CompressedXContent mappings = mappingsMap.isEmpty() ? null : new CompressedXContent(mappingsMap);
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        RoutingTable routingTable = currentState.routingTable(request.projectId());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(metadata, projectMetadata, currentState.nodes(), currentState.blocks(), routingTable, request, MetadataIndexTemplateService.resolveSettings(templates), mappings == null ? List.of() : List.of(mappings), null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, mappings == null ? List.of() : List.of(mappings), indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), MetadataIndexTemplateService.resolveAliases(templates), projectMetadata, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(request.getNameResolvedAt()), this.systemIndices::isSystemName), templates.stream().map(IndexTemplateMetadata::getName).collect(Collectors.toList()), projectMetadataTransformer, rerouteListener);
    }

    private ClusterState applyCreateIndexRequestWithV2Template(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, String templateName, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        boolean isDataStream;
        logger.debug("applying create index request using composable template [{}]", (Object)templateName);
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        ComposableIndexTemplate template = projectMetadata.templatesV2().get(templateName);
        boolean bl = isDataStream = template.getDataStreamTemplate() != null;
        if (isDataStream && request.dataStreamName() == null) {
            throw new IllegalArgumentException("cannot create index with name [" + request.index() + "], because it matches with template [" + templateName + "] that creates data streams only, use create data stream api instead");
        }
        return this.applyCreateIndexRequestWithV2Template(currentState, request, silent, template, projectMetadataTransformer, rerouteListener);
    }

    private ClusterState applyCreateIndexRequestWithV2Template(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, ComposableIndexTemplate template, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        RoutingTable routingTable = currentState.routingTable(request.projectId());
        boolean isDataStream = template.getDataStreamTemplate() != null;
        List<CompressedXContent> mappings = MetadataCreateIndexService.collectV2Mappings(request.mappings(), projectMetadata, template, this.xContentRegistry, request.index());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(metadata, projectMetadata, currentState.nodes(), currentState.blocks(), routingTable, request, MetadataIndexTemplateService.resolveSettings(template, projectMetadata.componentTemplates()), mappings, null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, mappings, indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), isDataStream ? Set.of() : request.aliases(), isDataStream ? List.of() : MetadataIndexTemplateService.resolveAliases(projectMetadata, template), projectMetadata, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(request.getNameResolvedAt()), this.systemIndices::isSystemName), Collections.singletonList("provided in request"), projectMetadataTransformer, rerouteListener);
    }

    private ClusterState applyCreateIndexRequestForSystemIndex(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, String indexPattern, ActionListener<Void> rerouteListener) throws Exception {
        logger.debug("applying create index request for system index [{}] matching pattern [{}]", (Object)request.index(), (Object)indexPattern);
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        RoutingTable routingTable = currentState.routingTable(request.projectId());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(metadata, projectMetadata, currentState.nodes(), currentState.blocks(), routingTable, request, Settings.EMPTY, null, null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, List.of(new CompressedXContent(MapperService.parseMapping(this.xContentRegistry, request.mappings()))), indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), List.of(), projectMetadata, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(request.getNameResolvedAt()), this.systemIndices::isSystemName), List.of(), null, rerouteListener);
    }

    private ClusterState applyCreateIndexRequestForSystemDataStream(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        Objects.requireNonNull(request.systemDataStreamDescriptor());
        logger.debug("applying create index request for system data stream [{}]", (Object)request.systemDataStreamDescriptor());
        ComposableIndexTemplate template = request.systemDataStreamDescriptor().getComposableIndexTemplate();
        if (request.dataStreamName() == null && template.getDataStreamTemplate() != null) {
            throw new IllegalArgumentException("cannot create index with name [" + request.index() + "], because it matches with a system data stream");
        }
        Map<String, ComponentTemplate> componentTemplates = request.systemDataStreamDescriptor().getComponentTemplates();
        List<CompressedXContent> mappings = MetadataCreateIndexService.collectSystemV2Mappings(template, componentTemplates, this.xContentRegistry, request.index());
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        RoutingTable routingTable = currentState.routingTable(request.projectId());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(metadata, projectMetadata, currentState.nodes(), currentState.blocks(), routingTable, request, MetadataIndexTemplateService.resolveSettings(template, componentTemplates), mappings, null, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, null);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, null, tmpImd, mappings, indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), MetadataIndexTemplateService.resolveAliases(template, componentTemplates), projectMetadata, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(request.getNameResolvedAt()), this.systemIndices::isSystemName), List.of(), projectMetadataTransformer, rerouteListener);
    }

    private static List<CompressedXContent> collectSystemV2Mappings(ComposableIndexTemplate composableIndexTemplate, Map<String, ComponentTemplate> componentTemplates, NamedXContentRegistry xContentRegistry, String indexName) throws Exception {
        List<CompressedXContent> templateMappings = MetadataIndexTemplateService.collectMappings(composableIndexTemplate, componentTemplates, indexName);
        return MetadataCreateIndexService.collectV2Mappings(null, templateMappings, xContentRegistry);
    }

    private static List<CompressedXContent> collectV2Mappings(@Nullable String requestMappings, List<CompressedXContent> templateMappings, NamedXContentRegistry xContentRegistry) throws IOException {
        Map<String, Object> parsedRequestMappings;
        ArrayList<CompressedXContent> result = new ArrayList<CompressedXContent>(templateMappings.size() + 1);
        result.addAll(templateMappings);
        if (requestMappings != null && !(parsedRequestMappings = MapperService.parseMapping(xContentRegistry, requestMappings)).isEmpty()) {
            result.add(new CompressedXContent(parsedRequestMappings));
        }
        return result;
    }

    public static List<CompressedXContent> collectV2Mappings(@Nullable String requestMappings, ProjectMetadata projectMetadata, ComposableIndexTemplate template, NamedXContentRegistry xContentRegistry, String indexName) throws IOException {
        List<CompressedXContent> templateMappings = MetadataIndexTemplateService.collectMappings(template, projectMetadata.componentTemplates(), indexName);
        return MetadataCreateIndexService.collectV2Mappings(requestMappings, templateMappings, xContentRegistry);
    }

    private ClusterState applyCreateIndexRequestWithExistingMetadata(ClusterState currentState, CreateIndexClusterStateUpdateRequest request, boolean silent, IndexMetadata sourceMetadata, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ActionListener<Void> rerouteListener) throws Exception {
        logger.info("applying create index request using existing index [{}] metadata", (Object)sourceMetadata.getIndex().getName());
        Map<String, Object> mappings = MapperService.parseMapping(this.xContentRegistry, request.mappings());
        if (!mappings.isEmpty()) {
            throw new IllegalArgumentException("mappings are not allowed when creating an index from a source index, all mappings are copied from the source index");
        }
        Metadata metadata = currentState.getMetadata();
        ProjectMetadata projectMetadata = metadata.getProject(request.projectId());
        RoutingTable routingTable = currentState.routingTable(request.projectId());
        Settings aggregatedIndexSettings = MetadataCreateIndexService.aggregateIndexSettings(metadata, projectMetadata, currentState.nodes(), currentState.blocks(), routingTable, request, Settings.EMPTY, null, sourceMetadata, this.settings, this.indexScopedSettings, this.shardLimitValidator, this.indexSettingProviders);
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(aggregatedIndexSettings, sourceMetadata);
        IndexMetadata tmpImd = this.buildAndValidateTemporaryIndexMetadata(aggregatedIndexSettings, request, routingNumShards);
        return this.applyCreateIndexWithTemporaryService(currentState, request, silent, sourceMetadata, tmpImd, List.of(), indexService -> MetadataCreateIndexService.resolveAndValidateAliases(request.index(), request.aliases(), Collections.emptyList(), projectMetadata, this.xContentRegistry, indexService.newSearchExecutionContext(0, 0, null, () -> 0L, null, Collections.emptyMap()), IndexService.dateMathExpressionResolverAt(request.getNameResolvedAt()), this.systemIndices::isSystemName), List.of(), projectMetadataTransformer, rerouteListener);
    }

    public static Map<String, Object> parseV1Mappings(String mappingsJson, List<CompressedXContent> templateMappings, NamedXContentRegistry xContentRegistry) throws IOException {
        Map<String, Object> mappings = MapperService.parseMapping(xContentRegistry, mappingsJson);
        for (CompressedXContent mapping : templateMappings) {
            Map<String, Object> templateMapping;
            if (mapping == null || (templateMapping = MapperService.parseMapping(xContentRegistry, mapping)).isEmpty()) continue;
            assert (templateMapping.size() == 1) : "expected exactly one mapping value, got: " + String.valueOf(templateMapping);
            templateMapping = Collections.singletonMap("_doc", templateMapping.values().iterator().next());
            if (mappings.isEmpty()) {
                mappings = templateMapping;
                continue;
            }
            XContentHelper.mergeDefaults(mappings, templateMapping);
        }
        return mappings;
    }

    static Settings aggregateIndexSettings(Metadata metadata, ProjectMetadata projectMetadata, DiscoveryNodes nodes, ClusterBlocks clusterBlocks, RoutingTable routingTable, CreateIndexClusterStateUpdateRequest request, Settings combinedTemplateSettings, List<CompressedXContent> combinedTemplateMappings, @Nullable IndexMetadata sourceMetadata, Settings settings, IndexScopedSettings indexScopedSettings, ShardLimitValidator shardLimitValidator, Set<IndexSettingProvider> indexSettingProviders) {
        String currentTierPreference;
        Settings templateAndRequestSettings;
        boolean isDataStreamIndex = request.dataStreamName() != null;
        Settings.Builder templateSettings = Settings.builder().put(combinedTemplateSettings);
        Settings.Builder requestSettings = Settings.builder().put(request.settings());
        Settings.Builder templateAndRequestSettingsBuilder = Settings.builder().put(combinedTemplateSettings).put(request.settings());
        if (request.isFailureIndex()) {
            DataStreamFailureStoreDefinition.filterUserDefinedSettings(templateAndRequestSettingsBuilder);
        }
        IndexVersion createdVersion = IndexMetadata.SETTING_INDEX_VERSION_CREATED.exists(templateAndRequestSettings = templateAndRequestSettingsBuilder.build()) ? IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(templateAndRequestSettings) : IndexVersion.min(IndexVersion.current(), nodes.getMaxDataNodeCompatibleIndexVersion());
        Settings.Builder indexSettingsBuilder = Settings.builder();
        if (sourceMetadata == null) {
            IndexMode templateIndexMode = Optional.of(request).filter(r -> !r.isFailureIndex()).map(CreateIndexClusterStateUpdateRequest::matchingTemplate).map(projectMetadata::retrieveIndexModeFromTemplate).orElse(null);
            Settings.Builder additionalIndexSettings = Settings.builder();
            Instant resolvedAt = Instant.ofEpochMilli(request.getNameResolvedAt());
            HashSet<String> overrulingSettings = new HashSet<String>();
            for (IndexSettingProvider provider : indexSettingProviders) {
                Settings.Builder builder = Settings.builder();
                provider.provideAdditionalSettings(request.index(), request.dataStreamName(), templateIndexMode, projectMetadata, resolvedAt, templateAndRequestSettings, combinedTemplateMappings, createdVersion, builder);
                Settings newAdditionalSettings = builder.build();
                MetadataCreateIndexService.validateAdditionalSettings(provider, newAdditionalSettings, additionalIndexSettings);
                additionalIndexSettings.put(newAdditionalSettings);
                if (!provider.overrulesTemplateAndRequestSettings()) continue;
                overrulingSettings.addAll(newAdditionalSettings.keySet());
            }
            for (String explicitSetting : additionalIndexSettings.keys()) {
                if (overrulingSettings.contains(explicitSetting)) {
                    templateSettings.remove(explicitSetting);
                    requestSettings.remove(explicitSetting);
                    continue;
                }
                if (templateSettings.keys().contains(explicitSetting) && templateSettings.get(explicitSetting) == null) {
                    logger.debug("removing default [{}] setting as it is set to null in a template for [{}] creation", (Object)explicitSetting, (Object)request.index());
                    additionalIndexSettings.remove(explicitSetting);
                    templateSettings.remove(explicitSetting);
                }
                if (!requestSettings.keys().contains(explicitSetting) || requestSettings.get(explicitSetting) != null) continue;
                logger.debug("removing default [{}] setting as it is set to null in the request for [{}] creation", (Object)explicitSetting, (Object)request.index());
                additionalIndexSettings.remove(explicitSetting);
                requestSettings.remove(explicitSetting);
            }
            indexSettingsBuilder.put(additionalIndexSettings.build());
            indexSettingsBuilder.put(templateSettings.build());
        }
        if (request.isFailureIndex()) {
            DataStreamFailureStoreDefinition.filterUserDefinedSettings(indexSettingsBuilder);
        }
        indexSettingsBuilder.put(requestSettings.build());
        if (sourceMetadata == null && DataTier.parseTierList(currentTierPreference = indexSettingsBuilder.get("index.routing.allocation.include._tier_preference")).isEmpty()) {
            String newTierPreference = isDataStreamIndex ? "data_hot" : "data_content";
            logger.debug("enforcing default [{}] setting for [{}] creation, replacing [{}] with [{}]", (Object)"index.routing.allocation.include._tier_preference", (Object)request.index(), (Object)currentTierPreference, (Object)newTierPreference);
            indexSettingsBuilder.put("index.routing.allocation.include._tier_preference", newTierPreference);
        }
        indexSettingsBuilder.put("index.version.created", createdVersion);
        if (!IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(indexSettingsBuilder)) {
            indexSettingsBuilder.put("index.number_of_shards", IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(settings));
        }
        if (!IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.exists(indexSettingsBuilder)) {
            indexSettingsBuilder.put("index.number_of_replicas", IndexMetadata.INDEX_NUMBER_OF_REPLICAS_SETTING.get(settings));
        }
        if (settings.get("index.auto_expand_replicas") != null && indexSettingsBuilder.get("index.auto_expand_replicas") == null) {
            indexSettingsBuilder.put("index.auto_expand_replicas", settings.get("index.auto_expand_replicas"));
        }
        if (indexSettingsBuilder.get("index.creation_date") == null) {
            indexSettingsBuilder.put("index.creation_date", Instant.now().toEpochMilli());
        }
        indexSettingsBuilder.put("index.provided_name", request.getProvidedName());
        indexSettingsBuilder.put("index.uuid", UUIDs.randomBase64UUID());
        if (sourceMetadata != null) {
            assert (request.resizeType() != null);
            MetadataCreateIndexService.prepareResizeIndexSettings(projectMetadata, clusterBlocks, routingTable, indexSettingsBuilder, request.recoverFrom(), request.index(), request.resizeType(), request.copySettings(), indexScopedSettings);
        }
        Settings indexSettings = indexSettingsBuilder.build();
        shardLimitValidator.validateShardLimit(indexSettings, nodes, metadata);
        MetadataCreateIndexService.validateSoftDeleteSettings(indexSettings);
        MetadataCreateIndexService.validateTranslogRetentionSettings(indexSettings);
        MetadataCreateIndexService.validateStoreTypeSetting(indexSettings);
        return indexSettings;
    }

    public static void validateAdditionalSettings(IndexSettingProvider provider, Settings additionalSettings, Settings.Builder allAdditionalSettings) throws IllegalArgumentException {
        for (String settingName : additionalSettings.keySet()) {
            if (allAdditionalSettings.keys().contains(settingName)) {
                String name = provider.getClass().getSimpleName();
                String message = Strings.format("additional index setting [%s] added by [%s] is already present", settingName, name);
                throw new IllegalArgumentException(message);
            }
            if (!"index.version.created".equals(settingName)) continue;
            String name = provider.getClass().getSimpleName();
            String message = Strings.format("setting [%s] added by [%s] is not allowed to be set via an IndexSettingProvider", settingName, name);
            throw new IllegalArgumentException(message);
        }
    }

    private static void validateSoftDeleteSettings(Settings indexSettings) {
        if (!IndexSettings.INDEX_SOFT_DELETES_SETTING.get(indexSettings).booleanValue() && IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings).onOrAfter(IndexVersions.V_8_0_0)) {
            throw new IllegalArgumentException("Creating indices with soft-deletes disabled is no longer supported. Please do not specify a value for setting [index.soft_deletes.enabled].");
        }
    }

    static int getIndexNumberOfRoutingShards(Settings indexSettings, @Nullable IndexMetadata sourceMetadata) {
        int routingNumShards = MetadataCreateIndexService.getIndexNumberOfRoutingShards(indexSettings, sourceMetadata == null ? 1 : sourceMetadata.getNumberOfShards(), sourceMetadata == null ? 0 : sourceMetadata.getRoutingNumShards());
        return routingNumShards;
    }

    static int getIndexNumberOfRoutingShards(Settings indexSettings, int sourceNumShards, int sourceRoutingNumShards) {
        int routingNumShards;
        int numTargetShards = IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(indexSettings);
        IndexVersion indexVersionCreated = IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings);
        if (sourceNumShards == 1) {
            routingNumShards = indexSettings.get(IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.getKey()) != null ? IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.get(indexSettings) : MetadataCreateIndexService.calculateNumRoutingShards(numTargetShards, indexVersionCreated);
        } else {
            assert (!IndexMetadata.INDEX_NUMBER_OF_ROUTING_SHARDS_SETTING.exists(indexSettings)) : "index.number_of_routing_shards should not be present on the target index on resize";
            routingNumShards = sourceRoutingNumShards;
        }
        return routingNumShards;
    }

    public static List<AliasMetadata> resolveAndValidateAliases(String index, Set<Alias> aliases, List<Map<String, AliasMetadata>> templateAliases, ProjectMetadata projectMetadata, NamedXContentRegistry xContentRegistry, SearchExecutionContext searchExecutionContext, Function<String, String> indexNameExpressionResolver, Predicate<String> systemNamePredicate) {
        HashSet<Alias> resolvedExpressions = new HashSet<Alias>();
        ArrayList<AliasMetadata> resolvedAliases = new ArrayList<AliasMetadata>();
        for (Alias alias : aliases) {
            String resolvedExpression = indexNameExpressionResolver.apply(alias.name());
            alias = alias.name(resolvedExpression);
            AliasValidator.validateAlias(alias, index, projectMetadata);
            if (Strings.hasLength(alias.filter())) {
                AliasValidator.validateAliasFilter(alias.name(), alias.filter(), searchExecutionContext, xContentRegistry);
            }
            AliasMetadata aliasMetadata = AliasMetadata.builder(alias.name()).filter(alias.filter()).indexRouting(alias.indexRouting()).searchRouting(alias.searchRouting()).writeIndex(alias.writeIndex()).isHidden(systemNamePredicate.test(alias.name()) ? Boolean.TRUE : alias.isHidden()).build();
            resolvedAliases.add(aliasMetadata);
            resolvedExpressions.add(new Alias(resolvedExpression));
        }
        HashMap<String, AliasMetadata> templatesAliases = new HashMap<String, AliasMetadata>();
        for (Map<String, AliasMetadata> templateAliasConfig : templateAliases) {
            for (Map.Entry<String, AliasMetadata> entry : templateAliasConfig.entrySet()) {
                String resolvedTemplateExpression = indexNameExpressionResolver.apply(entry.getValue().alias());
                AliasMetadata aliasMetadata = AliasMetadata.newAliasMetadata(entry.getValue(), resolvedTemplateExpression);
                if (resolvedExpressions.contains(new Alias(aliasMetadata.alias())) || templatesAliases.containsKey(entry.getKey())) continue;
                if (aliasMetadata.alias().contains("{index}")) {
                    String templatedAlias = aliasMetadata.alias().replace("{index}", index);
                    aliasMetadata = AliasMetadata.newAliasMetadata(aliasMetadata, templatedAlias);
                }
                if (systemNamePredicate.test(aliasMetadata.alias())) {
                    aliasMetadata = AliasMetadata.builder(aliasMetadata.alias()).filter(aliasMetadata.filter()).indexRouting(aliasMetadata.indexRouting()).searchRouting(aliasMetadata.searchRouting()).writeIndex(aliasMetadata.writeIndex()).isHidden(true).build();
                }
                AliasValidator.validateAliasMetadata(aliasMetadata, index, projectMetadata);
                if (aliasMetadata.filter() != null) {
                    AliasValidator.validateAliasFilter(aliasMetadata.alias(), aliasMetadata.filter().uncompressed(), searchExecutionContext, xContentRegistry);
                }
                templatesAliases.put(aliasMetadata.alias(), aliasMetadata);
                resolvedAliases.add(aliasMetadata);
            }
        }
        return resolvedAliases;
    }

    static ClusterState clusterStateCreateIndex(ClusterState currentState, ProjectId projectId, IndexMetadata indexMetadata, BiConsumer<ProjectMetadata.Builder, IndexMetadata> projectMetadataTransformer, ClusterBlocksTransformer blocksTransformer, ShardRoutingRoleStrategy shardRoutingRoleStrategy) {
        ProjectMetadata newProjectMetadata;
        ProjectMetadata currentProjectMetadata = currentState.metadata().getProject(projectId);
        if (projectMetadataTransformer != null) {
            ProjectMetadata.Builder projectMetadataBuilder = ProjectMetadata.builder(currentProjectMetadata);
            projectMetadataBuilder.put(indexMetadata, false);
            projectMetadataTransformer.accept(projectMetadataBuilder, indexMetadata);
            newProjectMetadata = projectMetadataBuilder.build();
        } else {
            newProjectMetadata = currentProjectMetadata.withAddedIndex(indexMetadata);
        }
        ClusterBlocks.Builder blocksBuilder = ClusterBlocks.builder().blocks(currentState.blocks());
        blocksBuilder.updateBlocks(projectId, indexMetadata);
        if (blocksTransformer != null) {
            blocksTransformer.apply(blocksBuilder, projectId, indexMetadata);
        }
        RoutingTable.Builder routingTableBuilder = RoutingTable.builder(shardRoutingRoleStrategy, currentState.routingTable(projectId)).addAsNew(newProjectMetadata.index(indexMetadata.getIndex().getName()));
        return ClusterState.builder(currentState).blocks(blocksBuilder).putProjectMetadata(newProjectMetadata).routingTable(GlobalRoutingTable.builder(currentState.globalRoutingTable()).put(projectId, routingTableBuilder).build()).build();
    }

    static IndexMetadata buildIndexMetadata(String indexName, List<AliasMetadata> aliases, Supplier<DocumentMapper> documentMapperSupplier, Settings indexSettings, int routingNumShards, @Nullable IndexMetadata sourceMetadata, boolean isSystem, Map<String, DiffableStringMap> customData) {
        IndexMetadata.Builder indexMetadataBuilder = MetadataCreateIndexService.createIndexMetadataBuilder(indexName, sourceMetadata, indexSettings, routingNumShards);
        indexMetadataBuilder.system(isSystem);
        HashMap<String, Object> mappingsMetadata = new HashMap<String, Object>();
        DocumentMapper docMapper = documentMapperSupplier.get();
        if (docMapper != null) {
            MappingMetadata mappingMd = new MappingMetadata(docMapper);
            mappingsMetadata.put(docMapper.type(), mappingMd);
            indexMetadataBuilder.putInferenceFields(docMapper.mappers().inferenceFields());
        }
        for (MappingMetadata mappingMd : mappingsMetadata.values()) {
            indexMetadataBuilder.putMapping(mappingMd);
        }
        for (int i = aliases.size() - 1; i >= 0; --i) {
            indexMetadataBuilder.putAlias(aliases.get(i));
        }
        indexMetadataBuilder.state(IndexMetadata.State.OPEN);
        indexMetadataBuilder.putCustom(customData);
        return indexMetadataBuilder.build();
    }

    private static IndexMetadata.Builder createIndexMetadataBuilder(String indexName, @Nullable IndexMetadata sourceMetadata, Settings indexSettings, int routingNumShards) {
        IndexMetadata.Builder builder = IndexMetadata.builder(indexName);
        builder.setRoutingNumShards(routingNumShards);
        builder.settings(indexSettings);
        if (sourceMetadata != null) {
            long primaryTerm = IntStream.range(0, sourceMetadata.getNumberOfShards()).mapToLong(sourceMetadata::primaryTerm).max().getAsLong();
            for (int shardId = 0; shardId < builder.numberOfShards(); ++shardId) {
                builder.primaryTerm(shardId, primaryTerm);
            }
        }
        return builder;
    }

    private static void updateIndexMappingsAndBuildSortOrder(IndexService indexService, CreateIndexClusterStateUpdateRequest request, List<CompressedXContent> mappings) throws IOException {
        MapperService mapperService = indexService.mapperService();
        IndexMode indexMode = indexService.getIndexSettings() != null ? indexService.getIndexSettings().getMode() : IndexMode.STANDARD;
        ArrayList<CompressedXContent> allMappings = new ArrayList<CompressedXContent>();
        CompressedXContent defaultMapping = indexMode.getDefaultMapping(indexService.getIndexSettings());
        if (defaultMapping != null) {
            allMappings.add(defaultMapping);
        }
        allMappings.addAll(mappings);
        mapperService.merge("_doc", allMappings, MapperService.MergeReason.INDEX_TEMPLATE);
        indexMode.validateTimestampFieldMapping(request.dataStreamName() != null, mapperService.mappingLookup());
        indexService.getIndexSortSupplier().get();
    }

    private static void validateActiveShardCount(ActiveShardCount waitForActiveShards, IndexMetadata indexMetadata) {
        if (waitForActiveShards == ActiveShardCount.DEFAULT) {
            waitForActiveShards = indexMetadata.getWaitForActiveShards();
        }
        if (!waitForActiveShards.validate(indexMetadata.getNumberOfReplicas())) {
            throw new IllegalArgumentException("invalid wait_for_active_shards[" + String.valueOf(waitForActiveShards) + "]: cannot be greater than number of shard copies [" + (indexMetadata.getNumberOfReplicas() + 1) + "]");
        }
    }

    private void validate(CreateIndexClusterStateUpdateRequest request, ProjectMetadata projectMetadata, RoutingTable routingTable) {
        MetadataCreateIndexService.validateIndexName(request.index(), projectMetadata, routingTable);
        this.validateIndexSettings(request.index(), request.settings(), this.forbidPrivateIndexSettings && !request.settingsSystemProvided());
    }

    public void validateIndexSettings(String indexName, Settings settings, boolean forbidPrivateIndexSettings) throws IndexCreationException {
        List<String> validationErrors = this.getIndexSettingsValidationErrors(settings, null, forbidPrivateIndexSettings);
        if (!validationErrors.isEmpty()) {
            ValidationException validationException = new ValidationException();
            validationException.addValidationErrors(validationErrors);
            throw new IndexCreationException(indexName, validationException);
        }
    }

    List<String> getIndexSettingsValidationErrors(Settings settings, @Nullable Settings systemProvided, boolean forbidPrivateIndexSettings) {
        List<String> validationErrors = MetadataCreateIndexService.validateIndexCustomPath(settings, this.env.sharedDataDir());
        if (forbidPrivateIndexSettings) {
            validationErrors.addAll(MetadataCreateIndexService.validatePrivateSettingsNotExplicitlySet(settings, systemProvided, this.indexScopedSettings));
        }
        return validationErrors;
    }

    private static List<String> validatePrivateSettingsNotExplicitlySet(Settings settings, @Nullable Settings systemProvided, IndexScopedSettings indexScopedSettings) {
        ArrayList<String> validationErrors = new ArrayList<String>();
        for (String key : settings.keySet()) {
            Setting<?> setting = indexScopedSettings.get(key);
            if (setting == null) {
                assert (indexScopedSettings.isPrivateSetting(key)) : "expected [" + key + "] to be private but it was not";
                continue;
            }
            if (!setting.isPrivateIndex() || MetadataCreateIndexService.isSystemProvided(key, settings, systemProvided)) continue;
            validationErrors.add("private index setting [" + key + "] can not be set explicitly");
        }
        return validationErrors;
    }

    private static boolean isSystemProvided(String key, Settings settings, @Nullable Settings systemProvided) {
        return systemProvided != null && settings.get(key).equals(systemProvided.get(key));
    }

    private static List<String> validateIndexCustomPath(Settings settings, @Nullable Path sharedDataPath) {
        String customPath = IndexMetadata.INDEX_DATA_PATH_SETTING.get(settings);
        ArrayList<String> validationErrors = new ArrayList<String>();
        if (!Strings.isEmpty(customPath)) {
            if (sharedDataPath == null) {
                validationErrors.add("path.shared_data must be set in order to use custom data paths");
            } else {
                Path resolvedPath = PathUtils.get((Path[])new Path[]{sharedDataPath}, (String)customPath);
                if (resolvedPath == null) {
                    validationErrors.add("custom path [" + customPath + "] is not a sub-path of path.shared_data [" + String.valueOf(sharedDataPath) + "]");
                }
            }
        }
        return validationErrors;
    }

    static List<String> validateShrinkIndex(ProjectMetadata projectMetadata, ClusterBlocks clusterBlocks, RoutingTable routingTable, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(projectMetadata, clusterBlocks, sourceIndex, targetIndexName, targetIndexSettings);
        if (sourceMetadata.isSearchableSnapshot()) {
            throw new IllegalArgumentException("can't shrink searchable snapshot index [" + sourceIndex + "]");
        }
        assert (IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(targetIndexSettings));
        IndexMetadata.selectShrinkShards(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
        if (sourceMetadata.getNumberOfShards() == 1) {
            throw new IllegalArgumentException("can't shrink an index with only one shard");
        }
        IndexRoutingTable table = routingTable.index(sourceIndex);
        HashMap<String, AtomicInteger> nodesToNumRouting = new HashMap<String, AtomicInteger>();
        int numShards = sourceMetadata.getNumberOfShards();
        for (ShardRouting routing : table.shardsWithState(ShardRoutingState.STARTED)) {
            nodesToNumRouting.computeIfAbsent(routing.currentNodeId(), s -> new AtomicInteger(0)).incrementAndGet();
        }
        ArrayList<String> nodesToAllocateOn = new ArrayList<String>();
        for (Map.Entry entries : nodesToNumRouting.entrySet()) {
            int numAllocations = ((AtomicInteger)entries.getValue()).get();
            assert (numAllocations <= numShards) : "wait what? " + numAllocations + " is > than num shards " + numShards;
            if (numAllocations != numShards) continue;
            nodesToAllocateOn.add((String)entries.getKey());
        }
        if (nodesToAllocateOn.isEmpty()) {
            throw new IllegalStateException("index " + sourceIndex + " must have all shards allocated on the same node to shrink index");
        }
        return nodesToAllocateOn;
    }

    static void validateSplitIndex(ProjectMetadata projectMetadata, ClusterBlocks clusterBlocks, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(projectMetadata, clusterBlocks, sourceIndex, targetIndexName, targetIndexSettings);
        if (sourceMetadata.isSearchableSnapshot()) {
            throw new IllegalArgumentException("can't split searchable snapshot index [" + sourceIndex + "]");
        }
        IndexMetadata.selectSplitShard(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
    }

    static void validateCloneIndex(ProjectMetadata projectMetadata, ClusterBlocks clusterBlocks, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        IndexMetadata sourceMetadata = MetadataCreateIndexService.validateResize(projectMetadata, clusterBlocks, sourceIndex, targetIndexName, targetIndexSettings);
        if (sourceMetadata.isSearchableSnapshot()) {
            for (Setting nonCloneableSetting : Arrays.asList(IndexModule.INDEX_STORE_TYPE_SETTING, IndexModule.INDEX_RECOVERY_TYPE_SETTING)) {
                if (nonCloneableSetting.exists(targetIndexSettings)) continue;
                throw new IllegalArgumentException("can't clone searchable snapshot index [" + sourceIndex + "]; setting [" + nonCloneableSetting.getKey() + "] should be overridden");
            }
        }
        IndexMetadata.selectCloneShard(0, sourceMetadata, IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
    }

    static IndexMetadata validateResize(ProjectMetadata projectMetadata, ClusterBlocks clusterBlocks, String sourceIndex, String targetIndexName, Settings targetIndexSettings) {
        if (projectMetadata.hasIndex(targetIndexName)) {
            throw new ResourceAlreadyExistsException(projectMetadata.index(targetIndexName).getIndex());
        }
        IndexMetadata sourceMetadata = projectMetadata.index(sourceIndex);
        if (sourceMetadata == null) {
            throw new IndexNotFoundException(sourceIndex);
        }
        IndexAbstraction source = (IndexAbstraction)projectMetadata.getIndicesLookup().get(sourceIndex);
        assert (source != null);
        if (source.getParentDataStream() != null && source.getParentDataStream().getWriteIndex().equals(sourceMetadata.getIndex())) {
            throw new IllegalArgumentException(String.format(Locale.ROOT, "cannot resize the write index [%s] for data stream [%s]", sourceIndex, source.getParentDataStream().getName()));
        }
        if (!clusterBlocks.indexBlocked(projectMetadata.id(), ClusterBlockLevel.WRITE, sourceIndex)) {
            throw new IllegalStateException("index " + sourceIndex + " must be read-only to resize index. use \"index.blocks.write=true\"");
        }
        if (IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.exists(targetIndexSettings)) {
            IndexMetadata.getRoutingFactor(sourceMetadata.getNumberOfShards(), IndexMetadata.INDEX_NUMBER_OF_SHARDS_SETTING.get(targetIndexSettings));
        }
        for (String setting : UNMODIFIABLE_SETTINGS_DURING_RESIZE) {
            if (!targetIndexSettings.hasValue(setting)) continue;
            throw new IllegalArgumentException("can't change setting [" + setting + "] during resize");
        }
        return sourceMetadata;
    }

    static void prepareResizeIndexSettings(ProjectMetadata projectMetadata, ClusterBlocks clusterBlocks, RoutingTable routingTable, Settings.Builder indexSettingsBuilder, Index resizeSourceIndex, String resizeIntoName, ResizeType type, boolean copySettings, IndexScopedSettings indexScopedSettings) {
        IndexMetadata sourceMetadata = projectMetadata.index(resizeSourceIndex.getName());
        if (type == ResizeType.SHRINK) {
            List<String> nodesToAllocateOn = MetadataCreateIndexService.validateShrinkIndex(projectMetadata, clusterBlocks, routingTable, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.put(IndexMetadata.INDEX_SHRINK_INITIAL_RECOVERY_KEY, Strings.arrayToCommaDelimitedString(nodesToAllocateOn.toArray()));
        } else if (type == ResizeType.SPLIT) {
            MetadataCreateIndexService.validateSplitIndex(projectMetadata, clusterBlocks, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.putNull(IndexMetadata.INDEX_SHRINK_INITIAL_RECOVERY_KEY);
        } else if (type == ResizeType.CLONE) {
            MetadataCreateIndexService.validateCloneIndex(projectMetadata, clusterBlocks, resizeSourceIndex.getName(), resizeIntoName, indexSettingsBuilder.build());
            indexSettingsBuilder.putNull(IndexMetadata.INDEX_SHRINK_INITIAL_RECOVERY_KEY);
        } else {
            throw new IllegalStateException("unknown resize type is " + String.valueOf((Object)type));
        }
        Settings.Builder builder = Settings.builder();
        if (copySettings) {
            for (String key : sourceMetadata.getSettings().keySet()) {
                Setting<?> setting = indexScopedSettings.get(key);
                if (setting == null) {
                    assert (indexScopedSettings.isPrivateSetting(key)) : key;
                } else if (setting.getProperties().contains((Object)Setting.Property.NotCopyableOnResize)) continue;
                if (indexSettingsBuilder.keys().contains(key)) continue;
                builder.copy(key, sourceMetadata.getSettings());
            }
        } else {
            Predicate<String> sourceSettingsPredicate = s -> (s.startsWith("index.similarity.") || s.startsWith("index.analysis.") || s.startsWith("index.sort.") || s.equals("index.soft_deletes.enabled")) && !indexSettingsBuilder.keys().contains(s);
            builder.put(sourceMetadata.getSettings().filter(sourceSettingsPredicate));
        }
        indexSettingsBuilder.put("index.version.created", sourceMetadata.getCreationVersion()).put(builder.build()).put("index.routing_partition_size", sourceMetadata.getRoutingPartitionSize()).put(IndexMetadata.INDEX_RESIZE_SOURCE_NAME.getKey(), resizeSourceIndex.getName()).put(IndexMetadata.INDEX_RESIZE_SOURCE_UUID.getKey(), resizeSourceIndex.getUUID());
        if (sourceMetadata.getSettings().hasValue("index.version.compatibility")) {
            indexSettingsBuilder.put("index.version.compatibility", sourceMetadata.getCompatibilityVersion());
        }
    }

    public static int calculateNumRoutingShards(int numShards, IndexVersion indexVersionCreated) {
        if (indexVersionCreated.onOrAfter(IndexVersions.V_7_0_0)) {
            int log2MaxNumShards = 10;
            int log2NumShards = 32 - Integer.numberOfLeadingZeros(numShards - 1);
            int numSplits = log2MaxNumShards - log2NumShards;
            numSplits = Math.max(1, numSplits);
            return numShards * 1 << numSplits;
        }
        return numShards;
    }

    public static void validateTranslogRetentionSettings(Settings indexSettings) {
        if (IndexMetadata.SETTING_INDEX_VERSION_CREATED.get(indexSettings).onOrAfter(IndexVersions.V_8_0_0) && (IndexSettings.INDEX_TRANSLOG_RETENTION_AGE_SETTING.exists(indexSettings) || IndexSettings.INDEX_TRANSLOG_RETENTION_SIZE_SETTING.exists(indexSettings))) {
            throw new IllegalArgumentException("Translog retention settings [index.translog.retention.age] and [index.translog.retention.size] are no longer supported. Please do not specify values for these settings");
        }
        if (IndexSettings.INDEX_SOFT_DELETES_SETTING.get(indexSettings).booleanValue() && (IndexSettings.INDEX_TRANSLOG_RETENTION_AGE_SETTING.exists(indexSettings) || IndexSettings.INDEX_TRANSLOG_RETENTION_SIZE_SETTING.exists(indexSettings))) {
            deprecationLogger.warn(DeprecationCategory.SETTINGS, "translog_retention", "Translog retention settings [index.translog.retention.age] and [index.translog.retention.size] are deprecated and effectively ignored. They will be removed in a future version.", new Object[0]);
        }
    }

    public static void validateStoreTypeSetting(Settings indexSettings) {
        String storeType = IndexModule.INDEX_STORE_TYPE_SETTING.get(indexSettings);
        if (IndexModule.Type.SIMPLEFS.match(storeType)) {
            deprecationLogger.warn(DeprecationCategory.SETTINGS, "store_type_setting", "[simplefs] is deprecated and will be removed in 8.0. Use [niofs] or other file systems instead. Elasticsearch 7.15 or later uses [niofs] for the [simplefs] store type as it offers superior or equivalent performance to [simplefs].", new Object[0]);
        }
    }

    public static boolean useRefreshBlock(Settings settings) {
        return DiscoveryNode.isStateless(settings) && settings.getAsBoolean(USE_INDEX_REFRESH_BLOCK_SETTING_NAME, false) != false;
    }

    static ClusterBlocksTransformer createClusterBlocksTransformerForIndexCreation(Settings settings) {
        if (!MetadataCreateIndexService.useRefreshBlock(settings)) {
            return (clusterBlocks, projectId, indexMetadata) -> {};
        }
        logger.debug("applying refresh block on index creation");
        return (clusterBlocks, projectId, indexMetadata) -> {
            if (MetadataCreateIndexService.applyRefreshBlock(indexMetadata)) {
                clusterBlocks.addIndexBlock(projectId, indexMetadata.getIndex().getName(), IndexMetadata.INDEX_REFRESH_BLOCK);
            }
        };
    }

    private static boolean applyRefreshBlock(IndexMetadata indexMetadata) {
        return 0 < indexMetadata.getNumberOfReplicas() && indexMetadata.getResizeSourceIndex() == null && indexMetadata.getInSyncAllocationIds().values().stream().allMatch(Set::isEmpty);
    }

    private boolean assertHasRefreshBlock(IndexMetadata indexMetadata, ProjectState state) {
        boolean hasRefreshBlock = state.blocks().hasIndexBlock(state.projectId(), indexMetadata.getIndex().getName(), IndexMetadata.INDEX_REFRESH_BLOCK);
        if (!MetadataCreateIndexService.useRefreshBlock(this.settings) || !MetadataCreateIndexService.applyRefreshBlock(indexMetadata) ? !$assertionsDisabled && hasRefreshBlock : !$assertionsDisabled && !hasRefreshBlock) {
            throw new AssertionError(indexMetadata.getIndex());
        }
        return true;
    }

    @FunctionalInterface
    static interface ClusterBlocksTransformer {
        public void apply(ClusterBlocks.Builder var1, ProjectId var2, IndexMetadata var3);
    }
}

