"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createDashboardTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _onechatServer = require("@kbn/onechat-server");
var _common = require("../../../common");
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createDashboardSchema = _zod.z.object({
  title: _zod.z.string().describe('The title of the dashboard to create.'),
  description: _zod.z.string().describe('A description of the dashboard.'),
  panels: _zod.z.array(_zod.z.unknown()).optional().describe('An array of panel configurations (PanelJSON or lens_tool_artifact).')
});
const createDashboardTool = (dashboard, savedObjects, {
  dashboardLocator,
  spaces
}) => {
  return {
    id: _common.dashboardTools.createDashboard,
    type: _onechatCommon.ToolType.builtin,
    availability: {
      cacheMode: 'space',
      handler: _utils.checkDashboardToolsAvailability
    },
    description: `Create a dashboard with the specified title, description, and panels.

This tool will:
1. Accept a title and description for the dashboard
2. Accept an array of panel configurations
3. Create a dashboard with the provided configuration`,
    schema: createDashboardSchema,
    tags: [],
    handler: async ({
      title,
      description,
      panels,
      ...rest
    }, {
      logger,
      request,
      esClient
    }) => {
      try {
        var _spaces$spacesService;
        const coreContext = {
          savedObjects: {
            client: savedObjects.getScopedClient(request)
          }
        };

        // Create a minimal request handler context
        const requestHandlerContext = {
          core: Promise.resolve(coreContext),
          resolve: async () => ({
            core: coreContext
          })
        };
        const normalizedPanels = (0, _utils.normalizePanels)(panels);
        const dashboardCreateResponse = await dashboard.client.create(requestHandlerContext, {
          data: {
            title,
            description,
            panels: normalizedPanels
          }
        });
        logger.info(`Dashboard created successfully: ${dashboardCreateResponse.id}`);
        const spaceId = spaces === null || spaces === void 0 ? void 0 : (_spaces$spacesService = spaces.spacesService) === null || _spaces$spacesService === void 0 ? void 0 : _spaces$spacesService.getSpaceId(request);
        const dashboardUrl = await (dashboardLocator === null || dashboardLocator === void 0 ? void 0 : dashboardLocator.getRedirectUrl({
          dashboardId: dashboardCreateResponse.id
        }, {
          spaceId
        }));
        return {
          results: [{
            type: _tool_result.ToolResultType.dashboard,
            tool_result_id: (0, _onechatServer.getToolResultId)(),
            data: {
              id: dashboardCreateResponse.id,
              title,
              content: {
                url: dashboardUrl,
                description,
                panelCount: normalizedPanels.length
              }
            }
          }]
        };
      } catch (error) {
        logger.error(`Error in create_dashboard tool: ${error.message}`);
        return {
          results: [{
            type: _tool_result.ToolResultType.error,
            data: {
              message: `Failed to create dashboard: ${error.message}`,
              metadata: {
                title,
                description
              }
            }
          }]
        };
      }
    }
  };
};
exports.createDashboardTool = createDashboardTool;