"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.parseRecords = void 0;
var _lodash = require("lodash");
var _uuid = require("uuid");
var _latest = require("@kbn/cloud-security-posture-common/types/graph/latest");
var _types = require("./types");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const parseRecords = (logger, records, nodesLimit) => {
  const ctx = {
    nodesLimit,
    logger,
    nodesMap: {},
    edgeLabelsNodes: {},
    edgesMap: {},
    labelEdges: {},
    messages: []
  };
  logger.trace(`Parsing records [length: ${records.length}] [nodesLimit: ${nodesLimit !== null && nodesLimit !== void 0 ? nodesLimit : 'none'}]`);
  createNodes(records, ctx);
  createEdgesAndGroups(ctx);
  logger.trace(`Parsed [nodes: ${Object.keys(ctx.nodesMap).length}, edges: ${Object.keys(ctx.edgesMap).length}]`);

  // Sort groups to be first (fixes minor layout issue)
  const nodes = sortNodes(ctx.nodesMap);
  return {
    nodes,
    edges: Object.values(ctx.edgesMap),
    messages: ctx.messages.length > 0 ? ctx.messages : undefined
  };
};
exports.parseRecords = parseRecords;
const deriveEntityAttributesFromType = entityGroupType => {
  const mappedProps = {
    shape: 'rectangle'
  };
  if (entityGroupType) {
    const {
      icon,
      shape
    } = (0, _utils.transformEntityTypeToIconAndShape)(entityGroupType);
    if (icon) {
      mappedProps.icon = icon;
    }
    if (shape) {
      mappedProps.shape = shape;
    }
    mappedProps.tag = entityGroupType;
  }
  return mappedProps;
};

/**
 * Resolves the entity type based on enrichment data
 * Falls back to singular/plural non-enriched types based on entity count
 */
const resolveEntityType = (entityType, idsCount) => {
  if (entityType) {
    return entityType;
  }
  return idsCount === 1 ? _types.NON_ENRICHED_ENTITY_TYPE_SINGULAR : _types.NON_ENRICHED_ENTITY_TYPE_PLURAL;
};

/**
 * Generates the appropriate label for an entity node
 * Logic matches the previous ESQL EVAL calculations
 */
const generateEntityLabel = (idsCount, entityNodeId, entityType, entityName, entitySubType) => {
  // Single non-enriched entity: show the group ID (which is the entity ID for single entities)
  if (idsCount === 1 && entityType === _types.NON_ENRICHED_ENTITY_TYPE_SINGULAR) {
    return entityNodeId;
  }
  // Single enriched entity: show the name (extract first element if array)
  if (idsCount === 1 && entityType !== _types.NON_ENRICHED_ENTITY_TYPE_SINGULAR) {
    const name = Array.isArray(entityName) ? entityName[0] : entityName;
    return name || '';
  }
  // Multiple entities with subtype: show the subtype
  if (idsCount > 1 && entitySubType) {
    return entitySubType;
  }
  return '';
};
const createGroupedActorAndTargetNodes = (record, context) => {
  const {
    nodesMap
  } = context;
  const {
    // actor attributes
    actorNodeId,
    actorIdsCount,
    actorsDocData,
    actorEntityType: rawActorEntityType,
    actorEntitySubType,
    actorEntityName,
    actorHostIps,
    // target attributes
    targetNodeId,
    targetIdsCount,
    targetsDocData,
    targetEntityType: rawTargetEntityType,
    targetEntitySubType,
    targetEntityName,
    targetHostIps
  } = record;
  const actorHostIpsArray = actorHostIps ? (0, _lodash.castArray)(actorHostIps) : [];
  const targetHostIpsArray = targetHostIps ? (0, _lodash.castArray)(targetHostIps) : [];

  // Resolve entity types and labels using utility functions
  const actorEntityType = resolveEntityType(rawActorEntityType, actorIdsCount);
  const targetEntityType = resolveEntityType(rawTargetEntityType, targetIdsCount);
  const actorLabel = generateEntityLabel(actorIdsCount, actorNodeId, actorEntityType, actorEntityName, actorEntitySubType);
  const targetLabel = generateEntityLabel(targetIdsCount, targetNodeId || '', targetEntityType, targetEntityName, targetEntitySubType);
  const actorsDocDataArray = actorsDocData ? (0, _lodash.castArray)(actorsDocData).filter(actorData => actorData !== null && actorData !== undefined).map(actorData => JSON.parse(actorData)) : [];
  const targetsDocDataArray = targetsDocData ? (0, _lodash.castArray)(targetsDocData).filter(targetData => targetData !== null && targetData !== undefined).map(targetData => JSON.parse(targetData)) : [];
  const actorGroup = {
    id: actorNodeId,
    // Actor: Always use node ID from ES|QL (single entity ID or MD5 hash)
    type: actorEntityType,
    docData: actorsDocDataArray,
    hostIps: actorHostIpsArray,
    ...(actorIdsCount > 1 ? {
      count: actorIdsCount
    } : {}),
    ...(actorLabel && actorLabel !== '' ? {
      label: actorLabel
    } : {})
  };
  const targetGroup = targetIdsCount > 0 && targetNodeId ? {
    id: targetNodeId,
    type: targetEntityType,
    docData: targetsDocDataArray,
    hostIps: targetHostIpsArray,
    ...(targetIdsCount > 1 ? {
      count: targetIdsCount
    } : {}),
    ...(targetLabel && targetLabel !== '' ? {
      label: targetLabel
    } : {})
  } : {
    // Unknown target
    id: `unknown-${(0, _uuid.v4)()}`,
    type: '',
    label: 'Unknown',
    docData: [],
    hostIps: []
  };
  [actorGroup, targetGroup].forEach(({
    id,
    label,
    type,
    count,
    docData,
    hostIps
  }) => {
    if (nodesMap[id] === undefined) {
      nodesMap[id] = {
        id,
        color: 'primary',
        ...(label ? {
          label
        } : {}),
        documentsData: docData,
        ...deriveEntityAttributesFromType(type),
        ...(count && count > 1 ? {
          count
        } : {}),
        ...(hostIps.length > 0 ? {
          ips: hostIps
        } : {})
      };
    }
  });
  return {
    actorId: actorGroup.id,
    targetId: targetGroup.id
  };
};
const createLabelNode = (record, edgeId) => {
  const {
    action,
    docs,
    isAlert,
    isOrigin,
    isOriginAlert,
    badge,
    uniqueEventsCount,
    uniqueAlertsCount,
    sourceIps,
    sourceCountryCodes
  } = record;
  const labelId = edgeId + `label(${action})oe(${isOrigin ? 1 : 0})oa(${isOriginAlert ? 1 : 0})`;
  const color = uniqueAlertsCount >= 1 && uniqueEventsCount === 0 && (isOriginAlert || isAlert) ? 'danger' : 'primary';
  const sourceIpsArray = sourceIps ? (0, _lodash.castArray)(sourceIps) : [];
  const sourceCountryCodesArray = sourceCountryCodes ? (0, _lodash.castArray)(sourceCountryCodes) : [];
  return {
    id: labelId,
    label: action,
    color,
    shape: 'label',
    documentsData: parseDocumentsData(docs),
    count: badge,
    ...(uniqueEventsCount > 0 ? {
      uniqueEventsCount
    } : {}),
    ...(uniqueAlertsCount > 0 ? {
      uniqueAlertsCount
    } : {}),
    ...(sourceIpsArray.length > 0 ? {
      ips: sourceIpsArray
    } : {}),
    ...(sourceCountryCodesArray.length > 0 ? {
      countryCodes: sourceCountryCodesArray
    } : {})
  };
};
const processLabelNodes = (context, nodeData) => {
  const {
    nodesMap,
    edgeLabelsNodes,
    labelEdges
  } = context;
  const {
    edgeId,
    sourceId,
    targetId,
    labelNode
  } = nodeData;
  if (edgeLabelsNodes[edgeId] === undefined) {
    edgeLabelsNodes[edgeId] = [];
  }
  nodesMap[labelNode.id] = labelNode;
  edgeLabelsNodes[edgeId].push(labelNode.id);
  labelEdges[labelNode.id] = {
    source: sourceId,
    target: targetId,
    edgeType: 'solid'
  };
};
const isAboveAPINodesLimit = context => {
  const {
    nodesMap,
    nodesLimit
  } = context;
  return nodesLimit !== undefined && Object.keys(nodesMap).length >= nodesLimit;
};
const emitAPINodesLimitMessage = context => {
  const {
    nodesMap,
    nodesLimit,
    logger,
    messages
  } = context;
  logger.debug(`Reached nodes limit [limit: ${nodesLimit}] [current: ${Object.keys(nodesMap).length}]`);
  messages.push(_latest.ApiMessageCode.ReachedNodesLimit);
};
const createNodes = (records, context) => {
  for (const record of records) {
    if (isAboveAPINodesLimit(context)) {
      emitAPINodesLimitMessage(context);
      break;
    }
    const {
      actorId,
      targetId
    } = createGroupedActorAndTargetNodes(record, context);
    const edgeId = `a(${actorId})-b(${targetId})`;
    const labelNode = createLabelNode(record, edgeId);
    processLabelNodes(context, {
      edgeId,
      sourceId: actorId,
      targetId,
      labelNode
    });
  }
};
const sortNodes = nodesMap => {
  const groupNodes = [];
  const otherNodes = [];
  for (const node of Object.values(nodesMap)) {
    if (node.shape === 'group') {
      groupNodes.push(node);
    } else {
      otherNodes.push(node);
    }
  }
  return [...groupNodes, ...otherNodes];
};
const createEdgesAndGroups = context => {
  const {
    edgeLabelsNodes,
    edgesMap,
    nodesMap,
    labelEdges
  } = context;
  Object.entries(edgeLabelsNodes).forEach(([edgeId, edgeLabelsIds]) => {
    // When there's more than one edge label, create a group node
    if (edgeLabelsIds.length === 1) {
      const edgeLabelId = edgeLabelsIds[0];
      connectEntitiesAndLabelNode(edgesMap, nodesMap, labelEdges[edgeLabelId].source, edgeLabelId, labelEdges[edgeLabelId].target, labelEdges[edgeLabelId].edgeType);
    } else {
      const groupNode = {
        id: `grp(${edgeId})`,
        shape: 'group'
      };
      nodesMap[groupNode.id] = groupNode;
      let groupEdgesColor = 'subdued';

      // Order of creation matters when using dagre layout, first create edges to the group node,
      // then connect the group node to the label nodes
      connectEntitiesAndLabelNode(edgesMap, nodesMap, labelEdges[edgeLabelsIds[0]].source, groupNode.id, labelEdges[edgeLabelsIds[0]].target, 'solid', groupEdgesColor);
      edgeLabelsIds.forEach(edgeLabelId => {
        nodesMap[edgeLabelId].parentId = groupNode.id;
        connectEntitiesAndLabelNode(edgesMap, nodesMap, groupNode.id, edgeLabelId, groupNode.id, labelEdges[edgeLabelId].edgeType);
        if (nodesMap[edgeLabelId].color === 'danger') {
          groupEdgesColor = 'danger';
        }
      });
    }
  });
};
const connectEntitiesAndLabelNode = (edgesMap, nodesMap, sourceNodeId, labelNodeId, targetNodeId, edgeType = 'solid', colorOverride) => {
  [connectNodes(nodesMap, sourceNodeId, labelNodeId, edgeType, colorOverride), connectNodes(nodesMap, labelNodeId, targetNodeId, edgeType, colorOverride)].forEach(edge => {
    edgesMap[edge.id] = edge;
  });
};
const connectNodes = (nodesMap, sourceNodeId, targetNodeId, edgeType = 'solid', colorOverride) => {
  const sourceNode = nodesMap[sourceNodeId];
  const targetNode = nodesMap[targetNodeId];
  const color = sourceNode.shape === 'label' && sourceNode.color === 'danger' || targetNode.shape === 'label' && targetNode.color === 'danger' ? 'danger' : 'subdued';
  return {
    id: `a(${sourceNodeId})-b(${targetNodeId})`,
    source: sourceNodeId,
    target: targetNodeId,
    color: colorOverride !== null && colorOverride !== void 0 ? colorOverride : color,
    type: edgeType
  };
};
const parseDocumentsData = docs => {
  if (typeof docs === 'string') {
    return [JSON.parse(docs)];
  }
  return docs.map(doc => JSON.parse(doc));
};