"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AutomaticImportSamplesIndexService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _storage = require("./storage");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AutomaticImportSamplesIndexService {
  constructor(logger, esClientPromise) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "samplesIndexAdapter", null);
    this.logger = logger.get('samplesIndexService');
    void this.initialize(esClientPromise);
  }
  async initialize(esClientPromise) {
    const [esClient] = await Promise.all([esClientPromise]);
    // Initialize samples index adapter
    this.samplesIndexAdapter = (0, _storage.createIndexAdapter)({
      logger: this.logger,
      esClient
    });
  }

  /**
   * Creates samples documents in the samples index.
   */
  async addSamplesToDataStream(currentAuthenticatedUser, dataStream) {
    if (!this.samplesIndexAdapter) {
      throw new Error('Samples index adapter not initialized');
    }
    const operations = dataStream.logData.map(logData => {
      const document = {
        integration_id: dataStream.integrationId,
        data_stream_id: dataStream.dataStreamId,
        log_data: logData,
        created_by: currentAuthenticatedUser.username,
        original_filename: dataStream.originalFilename,
        metadata: {
          created_at: new Date().toISOString()
        }
      };
      return {
        index: {
          document
        }
      };
    });
    return this.samplesIndexAdapter.getClient().bulk({
      operations
    });
  }

  /**
   * Gets samples for a data stream
   * @param integrationId - The integration ID
   * @param dataStreamId - The data stream ID
   * @returns The samples for the data stream
   * @throws If the samples index adapter is not initialized
   */
  async getSamplesForDataStream(integrationId, dataStreamId) {
    if (!this.samplesIndexAdapter) {
      throw new Error('Samples index adapter not initialized');
    }
    const results = await this.samplesIndexAdapter.getClient().search({
      query: {
        bool: {
          must: [{
            term: {
              integration_id: integrationId
            }
          }, {
            term: {
              data_stream_id: dataStreamId
            }
          }]
        }
      },
      size: 500,
      // TODO: Make this configurable
      track_total_hits: false
    });
    const samples = results.hits.hits.map(hit => hit._source.log_data);
    return samples;
  }
}
exports.AutomaticImportSamplesIndexService = AutomaticImportSamplesIndexService;