"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getServiceMetadataDetails = getServiceMetadataDetails;
var _server = require("@kbn/observability-plugin/server");
var _common = require("@kbn/observability-plugin/common");
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _elasticAgentUtils = require("@kbn/elastic-agent-utils");
var _environment_query = require("../../../common/utils/environment_query");
var _apm = require("../../../common/es_fields/apm");
var _get_service_metadata_icons = require("./get_service_metadata_icons");
var _agent_name = require("../../../common/agent_name");
var _maybe2 = require("../../../common/utils/maybe");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getServiceMetadataDetails({
  serviceName,
  environment,
  apmEventClient,
  start,
  end
}) {
  var _maybe, _event$AGENT_NAME, _event$TELEMETRY_SDK_, _event$TELEMETRY_SDK_2, _event$AGENT_VERSION, _aggregations$faasTri;
  const filter = [{
    term: {
      [_apm.SERVICE_NAME]: serviceName
    }
  }, ...(0, _environment_query.environmentQuery)(environment), ...(0, _server.rangeQuery)(start, end)];
  const params = {
    apm: {
      events: [_common.ProcessorEvent.transaction, _common.ProcessorEvent.error, _common.ProcessorEvent.metric]
    },
    sort: [{
      _score: {
        order: 'desc'
      }
    }, {
      '@timestamp': {
        order: 'desc'
      }
    }],
    track_total_hits: 1,
    size: 1,
    query: {
      bool: {
        filter,
        should: _get_service_metadata_icons.should
      }
    },
    aggs: {
      serviceVersions: {
        terms: {
          field: _apm.SERVICE_VERSION,
          size: 10,
          order: {
            _key: 'desc'
          }
        }
      },
      availabilityZones: {
        terms: {
          field: _apm.CLOUD_AVAILABILITY_ZONE,
          size: 10
        }
      },
      containerIds: {
        terms: {
          field: _apm.CONTAINER_ID,
          size: 10
        }
      },
      regions: {
        terms: {
          field: _apm.CLOUD_REGION,
          size: 10
        }
      },
      cloudServices: {
        terms: {
          field: _apm.CLOUD_SERVICE_NAME,
          size: 1
        }
      },
      machineTypes: {
        terms: {
          field: _apm.CLOUD_MACHINE_TYPE,
          size: 10
        }
      },
      faasTriggerTypes: {
        terms: {
          field: _apm.FAAS_TRIGGER_TYPE,
          size: 10
        }
      },
      faasFunctionNames: {
        terms: {
          field: _apm.FAAS_ID,
          size: 10
        }
      },
      totalNumberInstances: {
        cardinality: {
          field: _apm.SERVICE_NODE_NAME
        }
      }
    },
    fields: ['*']
  };
  const data = await apmEventClient.search('get_service_metadata_details', params);
  if (data.hits.total.value === 0) {
    return {
      service: undefined,
      container: undefined,
      cloud: undefined
    };
  }
  const hits = (_maybe = (0, _maybe2.maybe)(data.hits.hits[0])) === null || _maybe === void 0 ? void 0 : _maybe.fields;
  const event = hits && (0, _utils.accessKnownApmEventFields)(hits);
  if (!event) {
    return {
      service: undefined,
      container: undefined,
      cloud: undefined
    };
  }
  const aggregations = data.aggregations;
  const agentName = (0, _elasticAgentUtils.getAgentName)((_event$AGENT_NAME = event[_apm.AGENT_NAME]) !== null && _event$AGENT_NAME !== void 0 ? _event$AGENT_NAME : null, (_event$TELEMETRY_SDK_ = event[_apm.TELEMETRY_SDK_LANGUAGE]) !== null && _event$TELEMETRY_SDK_ !== void 0 ? _event$TELEMETRY_SDK_ : null, (_event$TELEMETRY_SDK_2 = event[_apm.TELEMETRY_SDK_NAME]) !== null && _event$TELEMETRY_SDK_2 !== void 0 ? _event$TELEMETRY_SDK_2 : null);
  const agentVersion = (_event$AGENT_VERSION = event[_apm.AGENT_VERSION]) !== null && _event$AGENT_VERSION !== void 0 ? _event$AGENT_VERSION : event[_apm.TELEMETRY_SDK_VERSION];
  const runtimeName = event[_apm.SERVICE_RUNTIME_NAME];
  const runtimeVersion = event[_apm.SERVICE_RUNTIME_VERSION];
  const serviceMetadataDetails = {
    versions: aggregations === null || aggregations === void 0 ? void 0 : aggregations.serviceVersions.buckets.map(bucket => bucket.key),
    runtime: runtimeName || runtimeVersion ? {
      name: runtimeName,
      version: runtimeVersion
    } : undefined,
    framework: event[_apm.SERVICE_FRAMEWORK_NAME],
    agent: {
      name: agentName,
      version: agentVersion
    }
  };
  const otelDetails = (0, _agent_name.isOpenTelemetryAgentName)(agentName) ? {
    language: (0, _agent_name.hasOpenTelemetryPrefix)(agentName) ? agentName.split('/')[1] : undefined,
    sdkVersion: agentVersion,
    autoVersion: event['labels.telemetry_auto_version']
  } : undefined;
  const totalNumberInstances = aggregations === null || aggregations === void 0 ? void 0 : aggregations.totalNumberInstances.value;
  const kubernetes = event.containsFields('kubernetes');
  const container = event.containsFields('container');
  const hostOsPlatform = event[_apm.HOST_OS_PLATFORM];
  const containerDetails = hostOsPlatform || kubernetes || totalNumberInstances || container ? {
    type: kubernetes ? 'Kubernetes' : 'Docker',
    os: hostOsPlatform,
    totalNumberInstances,
    ids: aggregations === null || aggregations === void 0 ? void 0 : aggregations.containerIds.buckets.map(bucket => bucket.key)
  } : undefined;
  const cloudServiceName = event[_apm.CLOUD_SERVICE_NAME];
  const serverlessDetails = !!(aggregations !== null && aggregations !== void 0 && (_aggregations$faasTri = aggregations.faasTriggerTypes) !== null && _aggregations$faasTri !== void 0 && _aggregations$faasTri.buckets.length) && cloudServiceName ? {
    type: cloudServiceName,
    functionNames: aggregations === null || aggregations === void 0 ? void 0 : aggregations.faasFunctionNames.buckets.map(bucket => getLambdaFunctionNameFromARN(bucket.key)).filter(name => name),
    faasTriggerTypes: aggregations === null || aggregations === void 0 ? void 0 : aggregations.faasTriggerTypes.buckets.map(bucket => bucket.key),
    hostArchitecture: event[_apm.HOST_ARCHITECTURE]
  } : undefined;
  const cloudDetails = cloudServiceName ? {
    provider: event[_apm.CLOUD_PROVIDER],
    projectName: event[_apm.CLOUD_PROJECT_NAME],
    serviceName: cloudServiceName,
    availabilityZones: aggregations === null || aggregations === void 0 ? void 0 : aggregations.availabilityZones.buckets.map(bucket => bucket.key),
    regions: aggregations === null || aggregations === void 0 ? void 0 : aggregations.regions.buckets.map(bucket => bucket.key),
    machineTypes: aggregations === null || aggregations === void 0 ? void 0 : aggregations.machineTypes.buckets.map(bucket => bucket.key)
  } : undefined;
  return {
    service: serviceMetadataDetails,
    opentelemetry: otelDetails,
    container: containerDetails,
    serverless: serverlessDetails,
    cloud: cloudDetails
  };
}
function getLambdaFunctionNameFromARN(arn) {
  // Lambda function ARN example: arn:aws:lambda:us-west-2:123456789012:function:my-function
  return arn.split(':')[6] || '';
}