"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildRecoveredAlert = void 0;
var _deepmerge = _interopRequireDefault(require("deepmerge"));
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _lodash = require("lodash");
var _strip_framework_fields = require("./strip_framework_fields");
var _nanos_to_micros = require("./nanos_to_micros");
var _format_alert = require("./format_alert");
var _filter_alert_state = require("./filter_alert_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Updates an active alert document to recovered
 * Currently only populates framework fields and not any rule type specific fields
 */

const buildRecoveredAlert = ({
  alert,
  legacyAlert,
  rule,
  ruleData,
  timestamp,
  payload,
  runTimestamp,
  recoveryActionGroup,
  kibanaVersion,
  dangerouslyCreateAlertsInAllSpaces
}) => {
  var _ref, _alert$tags, _rule$ALERT_RULE_TAGS;
  const cleanedPayload = (0, _strip_framework_fields.stripFrameworkFields)(payload);

  // Make sure that any alert fields that are updateable are flattened.
  const refreshableAlertFields = (0, _format_alert.replaceRefreshableAlertFields)(alert);
  const alertState = legacyAlert.getState();
  const filteredAlertState = (0, _filter_alert_state.filterAlertState)(alertState);
  const hasAlertState = Object.keys(filteredAlertState).length > 0;

  // Preserve ALERT_MUTED from existing alert
  const alertMuted = (0, _lodash.get)(alert, _ruleDataUtils.ALERT_MUTED);
  const alertUpdates = {
    // Update the timestamp to reflect latest update time
    [_ruleDataUtils.TIMESTAMP]: timestamp,
    [_ruleDataUtils.EVENT_ACTION]: 'close',
    [_ruleDataUtils.ALERT_RULE_EXECUTION_TIMESTAMP]: runTimestamp !== null && runTimestamp !== void 0 ? runTimestamp : timestamp,
    [_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]: rule[_ruleDataUtils.ALERT_RULE_EXECUTION_UUID],
    // Set the recovery action group
    [_ruleDataUtils.ALERT_ACTION_GROUP]: recoveryActionGroup,
    // Set latest flapping state
    [_ruleDataUtils.ALERT_FLAPPING]: legacyAlert.getFlapping(),
    // Set latest flapping_history
    [_ruleDataUtils.ALERT_FLAPPING_HISTORY]: legacyAlert.getFlappingHistory(),
    // Alert is recovering from active state so by default it is improving
    [_ruleDataUtils.ALERT_SEVERITY_IMPROVING]: true,
    [_ruleDataUtils.ALERT_PREVIOUS_ACTION_GROUP]: (0, _lodash.get)(alert, _ruleDataUtils.ALERT_ACTION_GROUP),
    // Set latest maintenance window IDs
    [_ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS]: legacyAlert.getMaintenanceWindowIds(),
    // Set latest match count, should be 0
    [_ruleDataUtils.ALERT_CONSECUTIVE_MATCHES]: legacyAlert.getActiveCount(),
    [_ruleDataUtils.ALERT_PENDING_RECOVERED_COUNT]: legacyAlert.getPendingRecoveredCount(),
    // Preserve muted state from existing alert
    ...(alertMuted !== undefined ? {
      [_ruleDataUtils.ALERT_MUTED]: alertMuted
    } : {}),
    // Set status to 'recovered'
    [_ruleDataUtils.ALERT_STATUS]: _ruleDataUtils.ALERT_STATUS_RECOVERED,
    // Set latest duration as recovered alerts should have updated duration
    ...(alertState.duration ? {
      [_ruleDataUtils.ALERT_DURATION]: (0, _nanos_to_micros.nanosToMicros)(alertState.duration)
    } : {}),
    // Set end time
    ...(alertState.end && alertState.start ? {
      [_ruleDataUtils.ALERT_START]: alertState.start,
      [_ruleDataUtils.ALERT_END]: alertState.end,
      [_ruleDataUtils.ALERT_TIME_RANGE]: {
        gte: alertState.start,
        lte: alertState.end
      }
    } : {}),
    [_ruleDataUtils.SPACE_IDS]: dangerouslyCreateAlertsInAllSpaces === true ? ['*'] : rule[_ruleDataUtils.SPACE_IDS],
    // Set latest kibana version
    [_ruleDataUtils.VERSION]: kibanaVersion,
    [_ruleDataUtils.TAGS]: Array.from(new Set([...((_ref = cleanedPayload === null || cleanedPayload === void 0 ? void 0 : cleanedPayload.tags) !== null && _ref !== void 0 ? _ref : []), ...((_alert$tags = alert.tags) !== null && _alert$tags !== void 0 ? _alert$tags : []), ...((_rule$ALERT_RULE_TAGS = rule[_ruleDataUtils.ALERT_RULE_TAGS]) !== null && _rule$ALERT_RULE_TAGS !== void 0 ? _rule$ALERT_RULE_TAGS : [])])),
    ...(hasAlertState ? {
      [_ruleDataUtils.ALERT_STATE_NAMESPACE]: filteredAlertState
    } : {})
  };

  // Clean the existing alert document so any nested fields that will be updated
  // are removed, to avoid duplicate data.
  // e.g. if the existing alert document has the field:
  // {
  //   kibana: {
  //     alert: {
  //       field1: 'value1'
  //     }
  //   }
  // }
  // and the updated alert has the field
  // {
  //   'kibana.alert.field1': 'value2'
  // }
  // the expanded field from the existing alert is removed
  const cleanedAlert = (0, _format_alert.removeUnflattenedFieldsFromAlert)(alert, {
    ...cleanedPayload,
    ...alertUpdates,
    ...refreshableAlertFields
  });
  return _deepmerge.default.all([cleanedAlert, refreshableAlertFields, cleanedPayload, alertUpdates], {
    arrayMerge: (_, sourceArray) => sourceArray
  });
};
exports.buildRecoveredAlert = buildRecoveredAlert;