"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.productDocumentationTool = void 0;
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _inferenceCommon = require("@kbn/inference-common");
var _onechatServer = require("@kbn/onechat-server");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const productDocumentationSchema = _zod.z.object({
  query: _zod.z.string().describe('Search query to retrieve documentation about Elastic products'),
  product: _zod.z.enum(['kibana', 'elasticsearch', 'observability', 'security']).optional().describe('Product to filter by: "kibana", "elasticsearch", "observability", or "security"'),
  max: _zod.z.number().optional().default(3).describe('Maximum number of documents to return. Defaults to 3.')
});

// TODO make this configurable, we need a platform level setting for the embedding model
const inferenceId = _inferenceCommon.defaultInferenceEndpoints.ELSER;
const productDocumentationTool = coreSetup => {
  // Create a closure that will resolve llmTasks when the handler is called
  const getLlmTasks = async () => {
    const [, plugins] = await coreSetup.getStartServices();
    return plugins.llmTasks;
  };
  const baseTool = {
    id: _onechatCommon.platformCoreTools.productDocumentation,
    type: _onechatCommon.ToolType.builtin,
    description: `Search and retrieve documentation about Elastic products (Kibana, Elasticsearch, Elastic Security, Elastic Observability).`,
    schema: productDocumentationSchema,
    handler: async ({
      query,
      product,
      max = 3
    }, {
      modelProvider,
      logger,
      request
    }) => {
      const llmTasks = await getLlmTasks();
      if (!llmTasks) {
        return {
          results: [(0, _onechatServer.createErrorResult)({
            message: 'Product documentation tool is not available. LlmTasks plugin is not available.'
          })]
        };
      }
      try {
        // Get the default model to extract the connector
        const model = await modelProvider.getDefaultModel();
        const connector = model.connector;

        // Retrieve documentation
        const result = await llmTasks.retrieveDocumentation({
          searchTerm: query,
          products: product ? [product] : undefined,
          max,
          connectorId: connector.connectorId,
          request,
          inferenceId
        });
        if (!result.success || result.documents.length === 0) {
          return {
            results: [(0, _onechatServer.createErrorResult)({
              message: 'No documentation found for the given query.',
              metadata: {
                query,
                product: product || 'all'
              }
            })]
          };
        }

        // Return documentation results
        return {
          results: result.documents.map(doc => ({
            type: _tool_result.ToolResultType.resource,
            data: {
              reference: {
                url: doc.url,
                title: doc.title
              },
              partial: doc.summarized,
              content: {
                title: doc.title,
                url: doc.url,
                content: doc.content,
                summarized: doc.summarized
              }
            }
          }))
        };
      } catch (error) {
        logger.error(`Error retrieving product documentation: ${error.message}`);
        return {
          results: [(0, _onechatServer.createErrorResult)({
            message: `Failed to retrieve product documentation: ${error.message}`
          })]
        };
      }
    },
    tags: [],
    availability: {
      cacheMode: 'space',
      handler: async () => {
        try {
          var _await$llmTasks$retri;
          const [, plugins] = await coreSetup.getStartServices();
          const llmTasks = plugins.llmTasks;
          if (!llmTasks) {
            return {
              status: 'unavailable'
            };
          }
          const isAvailable = (_await$llmTasks$retri = await llmTasks.retrieveDocumentationAvailable({
            inferenceId
          })) !== null && _await$llmTasks$retri !== void 0 ? _await$llmTasks$retri : false;
          return {
            status: isAvailable ? 'available' : 'unavailable'
          };
        } catch (error) {
          return {
            status: 'unavailable'
          };
        }
      }
    }
  };
  return baseTool;
};
exports.productDocumentationTool = productDocumentationTool;