"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchAllPages = exports.enhanceCaseData = exports.deduplicateCases = exports.createResult = exports.createErrorResponse = exports.createCommentSummary = exports.createCommentSummariesFromArray = void 0;
exports.getCaseUrl = getCaseUrl;
exports.normalizeTimeRange = exports.getCasesClient = void 0;
var _onechatServer = require("@kbn/onechat-server");
var _tool_result = require("@kbn/onechat-common/tools/tool_result");
var _server = require("@kbn/spaces-plugin/server");
var _spaces = require("@kbn/onechat-plugin/server/utils/spaces");
var _utils = require("@kbn/cases-plugin/server/common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Normalizes and validates time range parameters for case queries.
 * Validates ISO date strings and logs warnings for invalid dates.
 *
 * @param start - ISO datetime string for start time (inclusive), optional
 * @param end - ISO datetime string for end time (exclusive), optional
 * @param logger - Logger instance for warning messages
 * @returns Normalized time range object with ISO strings and Date objects, or null if neither start nor end is provided
 */
const normalizeTimeRange = (start, end, logger) => {
  if (!start && !end) {
    return null;
  }
  let startDate = null;
  if (start) {
    startDate = new Date(start);
    if (isNaN(startDate.getTime())) {
      logger.warn(`Invalid start date: ${start}`);
      startDate = null;
    }
  }
  let endDate = null;
  if (end) {
    endDate = new Date(end);
    if (isNaN(endDate.getTime())) {
      logger.warn(`Invalid end date: ${end}`);
      endDate = null;
    }
  }
  return {
    start: startDate ? startDate.toISOString() : null,
    end: endDate ? endDate.toISOString() : null,
    startDate,
    endDate
  };
};

/**
 * Creates a standardized tool result response for the cases tool.
 * Handles success, empty, and error cases.
 *
 * @param cases - Array of enhanced case data objects (empty array for empty results)
 * @param timeRange - Normalized time range object or null if no time filtering
 * @param message - Optional message to include in the response
 * @returns Tool result object conforming to ToolResultType.other format
 */
exports.normalizeTimeRange = normalizeTimeRange;
const createResult = (cases, timeRange, message) => ({
  results: [{
    type: _tool_result.ToolResultType.other,
    data: {
      total: cases.length,
      cases,
      start: (timeRange === null || timeRange === void 0 ? void 0 : timeRange.start) || null,
      end: (timeRange === null || timeRange === void 0 ? void 0 : timeRange.end) || null,
      ...(message && {
        message
      })
    }
  }]
});

/**
 * Creates a summary object from a case attachment/comment.
 * Extracts key information including comment text (truncated to 200 chars),
 * creator username, and creation timestamp.
 *
 * @param comment - The attachment/comment object from the cases API
 * @returns A CommentSummary object with id, comment text, creator, and timestamp
 */
exports.createResult = createResult;
const createCommentSummary = comment => {
  var _comment, _ref, _comment$created_by$u;
  const commentText = comment.type === 'user' && 'comment' in comment ? ((_comment = comment.comment) === null || _comment === void 0 ? void 0 : _comment.substring(0, 200)) || '' : '';
  return {
    id: comment.id,
    comment: commentText,
    created_by: (_ref = (_comment$created_by$u = comment.created_by.username) !== null && _comment$created_by$u !== void 0 ? _comment$created_by$u : comment.created_by.email) !== null && _ref !== void 0 ? _ref : null,
    created_at: comment.created_at || null
  };
};

/**
 * Creates comment summaries from an array of attachments.
 * Filters to only user comments, limits to the first 5, and converts to summaries.
 *
 * @param comments - Array of attachment objects from the cases API
 * @returns Array of CommentSummary objects (max 5 user comments)
 */
exports.createCommentSummary = createCommentSummary;
const createCommentSummariesFromArray = comments => {
  return comments.filter(att => att.type === 'user').slice(0, 5).map(createCommentSummary);
};

/**
 * Fetches all pages of cases from a search query.
 * Handles pagination automatically up to a maximum number of pages.
 *
 * @param casesClient - The cases client instance for API calls
 * @param searchParams - Search parameters for the query
 * @param maxPages - Maximum number of pages to fetch (default: 10)
 * @returns Promise resolving to array of all cases from all pages
 */
exports.createCommentSummariesFromArray = createCommentSummariesFromArray;
const fetchAllPages = async (casesClient, searchParams, maxPages = 10) => {
  const allCases = [];
  let currentPage = 1;
  let hasMorePages = true;
  while (hasMorePages && currentPage <= maxPages) {
    var _searchParams$perPage;
    searchParams.page = currentPage;
    const searchResult = await casesClient.cases.search(searchParams);
    if (searchResult.cases.length === 0) {
      break;
    }
    allCases.push(...searchResult.cases);

    // Stop if we got fewer results than requested (last page)
    if (searchResult.cases.length < ((_searchParams$perPage = searchParams.perPage) !== null && _searchParams$perPage !== void 0 ? _searchParams$perPage : 100)) {
      hasMorePages = false;
    }
    currentPage++;
  }
  return allCases;
};

/**
 * Enhances a case object with URL and markdown link fields.
 *
 * @param caseItem - The base case object from the API
 * @param comments - Optional array of comment summaries to include
 * @param request - Kibana request object for URL generation
 * @param coreServices - Core services including CoreStart and SpacesPlugin
 * @param logger - Logger instance for warning messages
 * @returns Enhanced case data object with url and markdown_link fields
 */
exports.fetchAllPages = fetchAllPages;
const enhanceCaseData = (caseItem, comments, request, coreServices, logger) => {
  const caseUrl = getCaseUrl(request, coreServices.coreStart, coreServices.spacesPlugin, caseItem.id, caseItem.owner) || null;
  if (!caseUrl) {
    logger.warn(`[Cases Tool] Failed to generate URL for case ${caseItem.id} with owner ${caseItem.owner}`);
  }
  const markdownLink = caseUrl ? `[${caseItem.title}](${caseUrl})` : caseItem.title;
  return {
    ...caseItem,
    url: caseUrl,
    markdown_link: markdownLink,
    ...(comments && comments.length > 0 && {
      comments_summary: comments
    })
  };
};

/**
 * Creates a standardized error response for the cases tool.
 * Extracts error message, logs it with the provided prefix, and returns
 * a tool result with an error message formatted for the user.
 *
 * @param error - The error object of unknown type
 * @param logPrefix - Prefix string for the error log message
 * @param userMessage - User-friendly error message to display
 * @param logger - Logger instance for error logging
 * @returns Tool result object with error information
 */
exports.enhanceCaseData = enhanceCaseData;
const createErrorResponse = (error, logPrefix, userMessage, logger) => {
  const errorMessage = error instanceof Error ? error.message : String(error);
  logger.error(`${logPrefix}: ${errorMessage}`);
  return {
    results: [(0, _onechatServer.createErrorResult)(`${userMessage}: ${errorMessage}`)]
  };
};

/**
 * Retrieves a cases client instance from the cases plugin.
 * Checks if the cases plugin is available and creates a client scoped to the request.
 * Returns an error result if the plugin is unavailable.
 *
 * @param pluginsStart - Plugin start dependencies containing the cases plugin
 * @param request - Kibana request object for creating a scoped client
 * @param logger - Logger instance for warning messages
 * @param timeRange - Normalized time range for error responses (if plugin unavailable)
 * @returns Promise resolving to either a cases client or an error result object
 */
exports.createErrorResponse = createErrorResponse;
const getCasesClient = async (pluginsStart, request, logger, timeRange) => {
  const casesPlugin = pluginsStart.cases;
  if (!casesPlugin) {
    logger.warn('[Cases Tool] Cases plugin not available, returning empty results');
    return {
      error: createResult([], timeRange, 'Cases plugin not available')
    };
  }
  const casesClient = await casesPlugin.getCasesClientWithRequest(request);
  return {
    casesClient
  };
};

/**
 * Deduplicates cases across multiple arrays by case ID.
 * Takes an array of case arrays (e.g., from multiple alert ID queries)
 * and returns a single array with unique cases based on their ID.
 *
 * @param casesArrays - Array of arrays containing RelatedCase objects
 * @returns Array of unique RelatedCase objects (first occurrence kept)
 */
exports.getCasesClient = getCasesClient;
const deduplicateCases = casesArrays => {
  const casesMap = new Map();
  for (const relatedCases of casesArrays) {
    for (const relatedCase of relatedCases) {
      if (!casesMap.has(relatedCase.id)) {
        casesMap.set(relatedCase.id, relatedCase);
      }
    }
  }
  return Array.from(casesMap.values());
};

// CASE URL

/**
 * App routes for different Kibana applications
 */
exports.deduplicateCases = deduplicateCases;
const APP_ROUTES = {
  security: '/app/security',
  observability: '/app/observability',
  management: '/app/management/insightsAndAlerting'
};

/**
 * Get the app route based on owner/case type
 */
function getAppRoute(owner) {
  const ownerToRoute = {
    securitySolution: APP_ROUTES.security,
    observability: APP_ROUTES.observability,
    cases: APP_ROUTES.management
  };
  return ownerToRoute[owner] || APP_ROUTES.management;
}

/**
 * Build a full URL from base components
 */
function buildFullUrl(request, core, spaceId, path) {
  const publicBaseUrl = core.http.basePath.publicBaseUrl;
  const serverBasePath = core.http.basePath.serverBasePath;

  // First try using publicBaseUrl if configured
  if (publicBaseUrl) {
    const pathWithSpace = (0, _server.addSpaceIdToPath)(serverBasePath, spaceId, path);
    return `${publicBaseUrl}${pathWithSpace}`;
  }

  // Fallback: construct URL from request
  const protocol = request.headers['x-forwarded-proto'] || 'http';
  const host = request.headers.host || 'localhost:5601';
  const baseUrl = `${protocol}://${host}`;
  const pathWithSpace = (0, _server.addSpaceIdToPath)(serverBasePath, spaceId, path);
  return `${baseUrl}${pathWithSpace}`;
}

/**
 * Generate a URL to a case
 */
function getCaseUrl(request, core, spaces, caseId, owner) {
  try {
    const spaceId = (0, _spaces.getCurrentSpaceId)({
      request,
      spaces
    });
    const publicBaseUrl = core.http.basePath.publicBaseUrl;

    // getCaseViewPath returns a full URL when publicBaseUrl is provided
    if (publicBaseUrl) {
      return (0, _utils.getCaseViewPath)({
        publicBaseUrl,
        spaceId,
        caseId,
        owner
      });
    }

    // Fallback: construct URL manually
    const appRoute = getAppRoute(owner);
    const path = `${appRoute}/cases/${caseId}`;
    return buildFullUrl(request, core, spaceId, path);
  } catch (error) {
    return null;
  }
}