"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAxiosInstance = getAxiosInstance;
var _get_action_kibana_privileges = require("../../../../lib/get_action_kibana_privileges");
var _is_preconfigured = require("../../../../lib/is_preconfigured");
var _is_system_action = require("../../../../lib/is_system_action");
var _saved_objects = require("../../../../constants/saved_objects");
var _lib = require("../../../../lib");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isWorkflowsOnlyConnectorType = ({
  supportedFeatureIds
}) => (supportedFeatureIds === null || supportedFeatureIds === void 0 ? void 0 : supportedFeatureIds.length) === 1 && supportedFeatureIds[0] === 'workflows';
async function getAxiosInstance(context, connectorId) {
  const {
    actionTypeRegistry,
    authorization,
    encryptedSavedObjectsClient,
    getAxiosInstanceWithAuth,
    inMemoryConnectors,
    isESOCanEncrypt,
    logger: log,
    request,
    spaces,
    unsecuredSavedObjectsClient,
    connectorTokenClient
  } = context;
  let actionTypeId;
  try {
    if ((0, _is_preconfigured.isPreconfigured)(context, connectorId) || (0, _is_system_action.isSystemAction)(context, connectorId)) {
      const connector = inMemoryConnectors.find(inMemoryConnector => inMemoryConnector.id === connectorId);
      actionTypeId = connector === null || connector === void 0 ? void 0 : connector.actionTypeId;
    } else {
      const {
        attributes
      } = await unsecuredSavedObjectsClient.get(_saved_objects.ACTION_SAVED_OBJECT_TYPE, connectorId);
      actionTypeId = attributes.actionTypeId;
    }
  } catch (err) {
    log.debug(`Failed to retrieve actionTypeId for action [${connectorId}]`, err);
    throw err;
  }
  const actionType = actionTypeRegistry.get(actionTypeId);
  if (!isWorkflowsOnlyConnectorType(actionType)) {
    throw new Error(`Unable to get axios instance for ${actionTypeId}. This function is exclusive for workflows-only connectors.`);
  }
  await authorization.ensureAuthorized({
    operation: 'execute',
    additionalPrivileges: (0, _get_action_kibana_privileges.getActionKibanaPrivileges)(context, actionTypeId),
    actionTypeId
  });

  // check to see if it's in memory connector before fetching secrets
  const inMemoryAction = inMemoryConnectors.find(inMemoryConnector => inMemoryConnector.id === connectorId);
  let secrets;
  if (inMemoryAction) {
    secrets = inMemoryAction.secrets;
  } else {
    if (!isESOCanEncrypt) {
      throw new Error(`Unable to get axios instance action because the Encrypted Saved Objects plugin is missing encryption key. Please set xpack.encryptedSavedObjects.encryptionKey in the kibana.yml or use the bin/kibana-encryption-keys command.`);
    }
    const spaceId = spaces && spaces.getSpaceId(request);
    const rawAction = await encryptedSavedObjectsClient.getDecryptedAsInternalUser('action', connectorId, spaceId && spaceId !== 'default' ? {
      namespace: spaceId
    } : {});
    secrets = rawAction.attributes.secrets;
  }
  const configurationUtilities = actionTypeRegistry.getUtils();
  const validatedSecrets = (0, _lib.validateSecrets)(actionType, secrets, {
    configurationUtilities
  });
  return await getAxiosInstanceWithAuth({
    connectorId,
    connectorTokenClient,
    secrets: validatedSecrets
  });
}