/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.tasks;

import java.io.IOException;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Executor;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchSecurityException;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionResponse;
import org.elasticsearch.action.ResultDeduplicator;
import org.elasticsearch.action.support.ChannelActionListener;
import org.elasticsearch.action.support.CountDownActionListener;
import org.elasticsearch.action.support.GroupedActionListener;
import org.elasticsearch.action.support.RefCountingRunnable;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.util.concurrent.EsExecutors;
import org.elasticsearch.common.util.concurrent.ListenableFuture;
import org.elasticsearch.core.Releasable;
import org.elasticsearch.core.Strings;
import org.elasticsearch.tasks.CancellableTask;
import org.elasticsearch.tasks.Task;
import org.elasticsearch.tasks.TaskId;
import org.elasticsearch.tasks.TaskManager;
import org.elasticsearch.transport.AbstractTransportRequest;
import org.elasticsearch.transport.NodeDisconnectedException;
import org.elasticsearch.transport.NodeNotConnectedException;
import org.elasticsearch.transport.Transport;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportRequestHandler;
import org.elasticsearch.transport.TransportRequestOptions;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class TaskCancellationService {
    public static final String BAN_PARENT_ACTION_NAME = "internal:admin/tasks/ban";
    public static final String REMOTE_CLUSTER_BAN_PARENT_ACTION_NAME = "cluster:internal/admin/tasks/ban";
    public static final String CANCEL_CHILD_ACTION_NAME = "internal:admin/tasks/cancel_child";
    public static final String REMOTE_CLUSTER_CANCEL_CHILD_ACTION_NAME = "cluster:internal/admin/tasks/cancel_child";
    private static final Logger logger = LogManager.getLogger(TaskCancellationService.class);
    private final TransportService transportService;
    private final TaskManager taskManager;
    private final ResultDeduplicator<CancelRequest, Void> deduplicator;
    private static final TransportResponseHandler.Empty NOOP_HANDLER = TransportResponseHandler.empty(TransportResponseHandler.TRANSPORT_WORKER, ActionListener.noop());

    public TaskCancellationService(TransportService transportService) {
        this.transportService = transportService;
        this.taskManager = transportService.getTaskManager();
        this.deduplicator = new ResultDeduplicator(transportService.getThreadPool().getThreadContext());
        transportService.registerRequestHandler(BAN_PARENT_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, BanParentTaskRequest::new, new BanParentRequestHandler());
        transportService.registerRequestHandler(REMOTE_CLUSTER_BAN_PARENT_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, BanParentTaskRequest::new, new BanParentRequestHandler());
        transportService.registerRequestHandler(CANCEL_CHILD_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, CancelChildRequest::new, new CancelChildRequestHandler());
        transportService.registerRequestHandler(REMOTE_CLUSTER_CANCEL_CHILD_ACTION_NAME, EsExecutors.DIRECT_EXECUTOR_SERVICE, CancelChildRequest::new, new CancelChildRequestHandler());
    }

    private String localNodeId() {
        return this.transportService.getLocalNode().getId();
    }

    void cancelTaskAndDescendants(CancellableTask task, String reason, boolean waitForCompletion, ActionListener<Void> finalListener) {
        this.deduplicator.executeOnce(new CancelRequest(task, waitForCompletion), finalListener, (r, listener) -> this.doCancelTaskAndDescendants(task, reason, waitForCompletion, (ActionListener<Void>)listener));
    }

    void doCancelTaskAndDescendants(CancellableTask task, String reason, boolean waitForCompletion, ActionListener<Void> listener) {
        TaskId taskId = task.taskInfo(this.localNodeId(), false).taskId();
        if (task.shouldCancelChildrenOnCancellation()) {
            Collection<Transport.Connection> childConnections;
            logger.trace("cancelling task [{}] and its descendants", (Object)taskId);
            ListenableFuture<Object> completedListener = new ListenableFuture<Object>();
            ListenableFuture<Void> setBanListener = new ListenableFuture<Void>();
            try (RefCountingRunnable refs = new RefCountingRunnable(() -> setBanListener.addListener(completedListener));){
                Releasable banChildrenRef = refs.acquire();
                Releasable cancelTaskRef = refs.acquire();
                childConnections = this.taskManager.startBanOnChildTasks(task.getId(), reason, () -> {
                    logger.trace("child tasks of parent [{}] are completed", (Object)taskId);
                    banChildrenRef.close();
                });
                this.taskManager.cancel(task, reason, () -> {
                    logger.trace("task [{}] is cancelled", (Object)taskId);
                    cancelTaskRef.close();
                });
            }
            this.setBanOnChildConnections(reason, waitForCompletion, task, childConnections, setBanListener);
            completedListener.addListener(ActionListener.running(this.transportService.getThreadPool().getThreadContext().preserveContext(() -> this.removeBanOnChildConnections(task, childConnections))));
            if (waitForCompletion) {
                completedListener.addListener(listener);
            } else {
                setBanListener.addListener(listener);
            }
        } else {
            logger.trace("task [{}] doesn't have any children that should be cancelled", (Object)taskId);
            if (waitForCompletion) {
                this.taskManager.cancel(task, reason, () -> listener.onResponse(null));
            } else {
                this.taskManager.cancel(task, reason, () -> {});
                listener.onResponse(null);
            }
        }
    }

    private void setBanOnChildConnections(String reason, boolean waitForCompletion, CancellableTask task, Collection<Transport.Connection> childConnections, ActionListener<Void> listener) {
        if (childConnections.isEmpty()) {
            listener.onResponse(null);
            return;
        }
        final TaskId taskId = new TaskId(this.localNodeId(), task.getId());
        logger.trace("cancelling child tasks of [{}] on child connections {}", (Object)taskId, childConnections);
        final CountDownActionListener countDownListener = new CountDownActionListener(childConnections.size(), listener);
        BanParentTaskRequest banRequest = BanParentTaskRequest.createSetBanParentTaskRequest(taskId, reason, waitForCompletion);
        for (final Transport.Connection connection : childConnections) {
            assert (TransportService.unwrapConnection(connection) == connection) : "Child connection must be unwrapped";
            this.transportService.sendRequest(connection, BAN_PARENT_ACTION_NAME, (TransportRequest)banRequest, TransportRequestOptions.EMPTY, new TransportResponseHandler.Empty(this){

                @Override
                public Executor executor() {
                    return TransportResponseHandler.TRANSPORT_WORKER;
                }

                @Override
                public void handleResponse() {
                    logger.trace("sent ban for tasks with the parent [{}] for connection [{}]", (Object)taskId, (Object)connection);
                    countDownListener.onResponse(null);
                }

                @Override
                public void handleException(TransportException exp) {
                    Throwable cause = ExceptionsHelper.unwrapCause(exp);
                    assert (!(cause instanceof ElasticsearchSecurityException)) : new AssertionError((Object)exp);
                    if (TaskCancellationService.isUnimportantBanFailure(cause)) {
                        logger.debug(() -> Strings.format((String)"cannot send ban for tasks with the parent [%s] on connection [%s]", (Object[])new Object[]{taskId, connection}), (Throwable)exp);
                    } else if (logger.isDebugEnabled()) {
                        logger.warn(() -> Strings.format((String)"cannot send ban for tasks with the parent [%s] on connection [%s]", (Object[])new Object[]{taskId, connection}), (Throwable)exp);
                    } else {
                        logger.warn("cannot send ban for tasks with the parent [{}] on connection [{}]: {}", (Object)taskId, (Object)connection, (Object)exp.getMessage());
                    }
                    countDownListener.onFailure(exp);
                }
            });
        }
    }

    private void removeBanOnChildConnections(CancellableTask task, Collection<Transport.Connection> childConnections) {
        final BanParentTaskRequest request = BanParentTaskRequest.createRemoveBanParentTaskRequest(new TaskId(this.localNodeId(), task.getId()));
        for (final Transport.Connection connection : childConnections) {
            assert (TransportService.unwrapConnection(connection) == connection) : "Child connection must be unwrapped";
            logger.trace("Sending remove ban for tasks with the parent [{}] for connection [{}]", (Object)request.parentTaskId, (Object)connection);
            this.transportService.sendRequest(connection, BAN_PARENT_ACTION_NAME, (TransportRequest)request, TransportRequestOptions.EMPTY, new TransportResponseHandler.Empty(this){

                @Override
                public Executor executor() {
                    return TransportResponseHandler.TRANSPORT_WORKER;
                }

                @Override
                public void handleResponse() {
                }

                @Override
                public void handleException(TransportException exp) {
                    Throwable cause = ExceptionsHelper.unwrapCause(exp);
                    assert (!(cause instanceof ElasticsearchSecurityException)) : new AssertionError((Object)exp);
                    if (TaskCancellationService.isUnimportantBanFailure(cause)) {
                        logger.debug(() -> Strings.format((String)"failed to remove ban for tasks with the parent [%s] on connection [%s]", (Object[])new Object[]{request2.parentTaskId, connection}), (Throwable)exp);
                    } else if (logger.isDebugEnabled()) {
                        logger.warn(() -> Strings.format((String)"failed to remove ban for tasks with the parent [%s] on connection [%s]", (Object[])new Object[]{request2.parentTaskId, connection}), (Throwable)exp);
                    } else {
                        logger.warn("failed to remove ban for tasks with the parent [{}] on connection [{}]: {}", (Object)request.parentTaskId, (Object)connection, (Object)exp.getMessage());
                    }
                }
            });
        }
    }

    private static boolean isUnimportantBanFailure(Throwable cause) {
        return cause instanceof NodeDisconnectedException || cause instanceof NodeNotConnectedException;
    }

    public void cancelChildRemote(TaskId parentTask, long childRequestId, Transport.Connection childConnection, String reason) {
        DiscoveryNode childNode = childConnection.getNode();
        logger.debug("sending cancellation of child of parent task [{}] with request ID [{}] to node [{}] because of [{}]", (Object)parentTask, (Object)childRequestId, (Object)childNode, (Object)reason);
        CancelChildRequest request = CancelChildRequest.createCancelChildRequest(parentTask, childRequestId, reason);
        this.transportService.sendRequest(childConnection, CANCEL_CHILD_ACTION_NAME, (TransportRequest)request, TransportRequestOptions.EMPTY, NOOP_HANDLER);
    }

    private class BanParentRequestHandler
    implements TransportRequestHandler<BanParentTaskRequest> {
        private BanParentRequestHandler() {
        }

        @Override
        public void messageReceived(BanParentTaskRequest request, TransportChannel channel, Task task) throws Exception {
            if (request.ban) {
                logger.debug("Received ban for the parent [{}] on the node [{}], reason: [{}]", (Object)request.parentTaskId, (Object)TaskCancellationService.this.localNodeId(), (Object)request.reason);
                List<CancellableTask> childTasks = TaskCancellationService.this.taskManager.setBan(request.parentTaskId, request.reason, channel);
                GroupedActionListener<Void> listener = new GroupedActionListener<Void>(childTasks.size() + 1, new ChannelActionListener(channel).map(r -> ActionResponse.Empty.INSTANCE));
                for (CancellableTask childTask : childTasks) {
                    TaskCancellationService.this.cancelTaskAndDescendants(childTask, request.reason, request.waitForCompletion, listener);
                }
                listener.onResponse(null);
            } else {
                logger.debug("Removing ban for the parent [{}] on the node [{}]", (Object)request.parentTaskId, (Object)TaskCancellationService.this.localNodeId());
                TaskCancellationService.this.taskManager.removeBan(request.parentTaskId);
                channel.sendResponse(ActionResponse.Empty.INSTANCE);
            }
        }
    }

    private class CancelChildRequestHandler
    implements TransportRequestHandler<CancelChildRequest> {
        private CancelChildRequestHandler() {
        }

        @Override
        public void messageReceived(CancelChildRequest request, TransportChannel channel, Task task) throws Exception {
            TaskCancellationService.this.taskManager.cancelChildLocal(request.parentTaskId, request.childRequestId, request.reason);
            channel.sendResponse(ActionResponse.Empty.INSTANCE);
        }
    }

    private static class CancelRequest {
        final CancellableTask task;
        final boolean waitForCompletion;

        CancelRequest(CancellableTask task, boolean waitForCompletion) {
            this.task = task;
            this.waitForCompletion = waitForCompletion;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CancelRequest that = (CancelRequest)o;
            return this.waitForCompletion == that.waitForCompletion && Objects.equals(this.task, that.task);
        }

        public int hashCode() {
            return Objects.hash(this.task, this.waitForCompletion);
        }
    }

    private static class BanParentTaskRequest
    extends AbstractTransportRequest {
        private final TaskId parentTaskId;
        private final boolean ban;
        private final boolean waitForCompletion;
        private final String reason;

        static BanParentTaskRequest createSetBanParentTaskRequest(TaskId parentTaskId, String reason, boolean waitForCompletion) {
            return new BanParentTaskRequest(parentTaskId, reason, waitForCompletion);
        }

        static BanParentTaskRequest createRemoveBanParentTaskRequest(TaskId parentTaskId) {
            return new BanParentTaskRequest(parentTaskId);
        }

        private BanParentTaskRequest(TaskId parentTaskId, String reason, boolean waitForCompletion) {
            this.parentTaskId = parentTaskId;
            this.ban = true;
            this.reason = reason;
            this.waitForCompletion = waitForCompletion;
        }

        private BanParentTaskRequest(TaskId parentTaskId) {
            this.parentTaskId = parentTaskId;
            this.ban = false;
            this.reason = null;
            this.waitForCompletion = false;
        }

        private BanParentTaskRequest(StreamInput in) throws IOException {
            super(in);
            this.parentTaskId = TaskId.readFromStream(in);
            this.ban = in.readBoolean();
            this.reason = this.ban ? in.readString() : null;
            this.waitForCompletion = in.readBoolean();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.parentTaskId.writeTo(out);
            out.writeBoolean(this.ban);
            if (this.ban) {
                out.writeString(this.reason);
            }
            out.writeBoolean(this.waitForCompletion);
        }
    }

    private static class CancelChildRequest
    extends AbstractTransportRequest {
        private final TaskId parentTaskId;
        private final long childRequestId;
        private final String reason;

        static CancelChildRequest createCancelChildRequest(TaskId parentTaskId, long childRequestId, String reason) {
            return new CancelChildRequest(parentTaskId, childRequestId, reason);
        }

        private CancelChildRequest(TaskId parentTaskId, long childRequestId, String reason) {
            this.parentTaskId = parentTaskId;
            this.childRequestId = childRequestId;
            this.reason = reason;
        }

        private CancelChildRequest(StreamInput in) throws IOException {
            super(in);
            this.parentTaskId = TaskId.readFromStream(in);
            this.childRequestId = in.readLong();
            this.reason = in.readString();
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.parentTaskId.writeTo(out);
            out.writeLong(this.childRequestId);
            out.writeString(this.reason);
        }
    }
}

