/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.geo;

import java.io.IOException;
import java.util.Collections;
import org.apache.lucene.geo.GeoEncodingUtils;
import org.apache.lucene.search.DoubleValues;
import org.apache.lucene.util.SloppyMath;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.common.geo.GenericPointParser;
import org.elasticsearch.common.geo.GeoDistance;
import org.elasticsearch.common.geo.GeoPoint;
import org.elasticsearch.common.unit.DistanceUnit;
import org.elasticsearch.common.xcontent.LoggingDeprecationHandler;
import org.elasticsearch.common.xcontent.support.XContentMapValues;
import org.elasticsearch.index.fielddata.FieldData;
import org.elasticsearch.index.fielddata.GeoPointValues;
import org.elasticsearch.index.fielddata.MultiGeoPointValues;
import org.elasticsearch.index.fielddata.SortedNumericDoubleValues;
import org.elasticsearch.index.fielddata.SortingNumericDoubleValues;
import org.elasticsearch.xcontent.DeprecationHandler;
import org.elasticsearch.xcontent.NamedXContentRegistry;
import org.elasticsearch.xcontent.XContentParser;
import org.elasticsearch.xcontent.support.MapXContentParser;

public class GeoUtils {
    public static final double MAX_LAT = 90.0;
    public static final double MIN_LAT = -90.0;
    public static final double MAX_LON = 180.0;
    public static final double MIN_LON = -180.0;
    public static final double EARTH_SEMI_MAJOR_AXIS = 6378137.0;
    public static final double EARTH_SEMI_MINOR_AXIS = 6356752.314245;
    public static final double EARTH_MEAN_RADIUS = 6371008.7714;
    public static final double EARTH_EQUATOR = 4.007501668557849E7;
    public static final double EARTH_POLAR_DISTANCE = 1.9970326371122006E7;
    public static final double TOLERANCE = 1.0E-6;
    private static final int QUAD_MAX_LEVELS_POSSIBLE = 50;
    private static final int GEOHASH_MAX_LEVELS_POSSIBLE = 24;
    private static final GenericPointParser<GeoPoint> geoPointParser = new GenericPointParser<GeoPoint>("geo_point", "lon", "lat"){

        @Override
        public void assertZValue(boolean ignoreZValue, double zValue) {
            GeoPoint.assertZValue(ignoreZValue, zValue);
        }

        @Override
        public GeoPoint createPoint(double x, double y) {
            return new GeoPoint(y, x);
        }

        @Override
        public String fieldError() {
            return "field must be either lat/lon, geohash string or type/coordinates";
        }
    };

    public static boolean isValidLatitude(double latitude) {
        return !Double.isNaN(latitude) && !Double.isInfinite(latitude) && !(latitude < -90.0) && !(latitude > 90.0);
    }

    public static boolean isValidLongitude(double longitude) {
        return !Double.isNaN(longitude) && !Double.isInfinite(longitude) && !(longitude < -180.0) && !(longitude > 180.0);
    }

    public static double geoHashCellWidth(int level) {
        assert (level >= 0);
        return 4.007501668557849E7 / (double)(1L << (level + 1) / 2 * 3 + level / 2 * 2);
    }

    public static double quadTreeCellWidth(int level) {
        assert (level >= 0);
        return 4.007501668557849E7 / (double)(1L << level);
    }

    public static double geoHashCellHeight(int level) {
        assert (level >= 0);
        return 1.9970326371122006E7 / (double)(1L << (level + 1) / 2 * 2 + level / 2 * 3);
    }

    public static double quadTreeCellHeight(int level) {
        assert (level >= 0);
        return 1.9970326371122006E7 / (double)(1L << level);
    }

    public static double geoHashCellSize(int level) {
        assert (level >= 0);
        double w = GeoUtils.geoHashCellWidth(level);
        double h = GeoUtils.geoHashCellHeight(level);
        return Math.sqrt(w * w + h * h);
    }

    public static double quadTreeCellSize(int level) {
        assert (level >= 0);
        return Math.sqrt(2.0048208977185252E15) / (double)(1L << level);
    }

    public static int quadTreeLevelsForPrecision(double meters) {
        int level;
        assert (meters >= 0.0);
        if (meters == 0.0) {
            return 50;
        }
        double ratio = 1.4983235946676121;
        double width = Math.sqrt(meters * meters / 2.244973594337675);
        long part = Math.round(Math.ceil(4.007501668557849E7 / width));
        return part <= 1L << (level = 64 - Long.numberOfLeadingZeros(part) - 1) ? level : level + 1;
    }

    public static int quadTreeLevelsForPrecision(String distance) {
        return GeoUtils.quadTreeLevelsForPrecision(DistanceUnit.METERS.parse(distance, DistanceUnit.DEFAULT));
    }

    public static int geoHashLevelsForPrecision(double meters) {
        int full;
        assert (meters >= 0.0);
        if (meters == 0.0) {
            return 24;
        }
        double ratio = 1.4983235946676121;
        double width = Math.sqrt(meters * meters / 2.244973594337675);
        double part = Math.ceil(4.007501668557849E7 / width);
        if (part == 1.0) {
            return 1;
        }
        int bits = (int)Math.round(Math.ceil(Math.log(part) / Math.log(2.0)));
        int left = bits - (full = bits / 5) * 5;
        int even = full + (left > 0 ? 1 : 0);
        int odd = full + (left > 3 ? 1 : 0);
        return even + odd;
    }

    public static int geoHashLevelsForPrecision(String distance) {
        return GeoUtils.geoHashLevelsForPrecision(DistanceUnit.METERS.parse(distance, DistanceUnit.DEFAULT));
    }

    public static double normalizeLon(double lon) {
        if (lon > 180.0 || lon <= -180.0) {
            lon = GeoUtils.centeredModulus(lon, 360.0);
        }
        return lon + 0.0;
    }

    public static double normalizeLat(double lat) {
        if (lat > 90.0 || lat < -90.0) {
            if ((lat = GeoUtils.centeredModulus(lat, 360.0)) < -90.0) {
                lat = -180.0 - lat;
            } else if (lat > 90.0) {
                lat = 180.0 - lat;
            }
        }
        return lat + 0.0;
    }

    public static void normalizePoint(GeoPoint point) {
        GeoUtils.normalizePoint(point, true, true);
    }

    public static void normalizePoint(GeoPoint point, boolean normLat, boolean normLon) {
        double[] pt = new double[]{point.lon(), point.lat()};
        GeoUtils.normalizePoint(pt, normLon, normLat);
        point.reset(pt[1], pt[0]);
    }

    public static void normalizePoint(double[] lonLat) {
        GeoUtils.normalizePoint(lonLat, true, true);
    }

    public static boolean needsNormalizeLat(double lat) {
        return lat > 90.0 || lat < -90.0;
    }

    public static boolean needsNormalizeLon(double lon) {
        return lon > 180.0 || lon < -180.0;
    }

    public static void normalizePoint(double[] lonLat, boolean normLon, boolean normLat) {
        assert (lonLat != null && lonLat.length == 2);
        normLat = normLat && GeoUtils.needsNormalizeLat(lonLat[1]);
        boolean bl = normLon = normLon && (GeoUtils.needsNormalizeLon(lonLat[0]) || normLat);
        if (normLat) {
            lonLat[1] = GeoUtils.centeredModulus(lonLat[1], 360.0);
            boolean shift = true;
            if (lonLat[1] < -90.0) {
                lonLat[1] = -180.0 - lonLat[1];
            } else if (lonLat[1] > 90.0) {
                lonLat[1] = 180.0 - lonLat[1];
            } else {
                shift = false;
            }
            if (shift) {
                lonLat[0] = normLon ? lonLat[0] + 180.0 : lonLat[0] + (GeoUtils.normalizeLon(lonLat[0]) > 0.0 ? -180.0 : 180.0);
            }
        }
        if (normLon) {
            lonLat[0] = GeoUtils.centeredModulus(lonLat[0], 360.0);
        }
    }

    public static double centeredModulus(double dividend, double divisor) {
        double rtn = dividend % divisor;
        if (rtn <= 0.0) {
            rtn += divisor;
        }
        if (rtn > divisor / 2.0) {
            rtn -= divisor;
        }
        return rtn;
    }

    public static GeoPoint parseGeoPoint(XContentParser parser) throws IOException, ElasticsearchParseException {
        return GeoUtils.parseGeoPoint(parser, false);
    }

    public static GeoPoint parseGeoPoint(Object value, boolean ignoreZValue) throws ElasticsearchParseException {
        GeoPoint geoPoint;
        MapXContentParser parser = new MapXContentParser(NamedXContentRegistry.EMPTY, (DeprecationHandler)LoggingDeprecationHandler.INSTANCE, Collections.singletonMap("null_value", value), null);
        try {
            parser.nextToken();
            parser.nextToken();
            parser.nextToken();
            geoPoint = GeoUtils.parseGeoPoint((XContentParser)parser, ignoreZValue);
        }
        catch (Throwable throwable) {
            try {
                try {
                    parser.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException ex) {
                throw new ElasticsearchParseException("error parsing geopoint", (Throwable)ex, new Object[0]);
            }
        }
        parser.close();
        return geoPoint;
    }

    public static GeoPoint parseGeoPoint(XContentParser parser, boolean ignoreZValue) throws IOException, ElasticsearchParseException {
        return GeoUtils.parseGeoPoint(parser, ignoreZValue, EffectivePoint.BOTTOM_LEFT);
    }

    public static GeoPoint parseGeoPoint(XContentParser parser, boolean ignoreZValue, EffectivePoint effectivePoint) throws IOException, ElasticsearchParseException {
        return geoPointParser.parsePoint(parser, ignoreZValue, value -> new GeoPoint().resetFromString((String)value, ignoreZValue, effectivePoint));
    }

    public static GeoPoint parseFromString(String val) {
        GeoPoint point = new GeoPoint();
        return point.resetFromString(val, false, EffectivePoint.BOTTOM_LEFT);
    }

    public static int parsePrecision(XContentParser parser) throws IOException, ElasticsearchParseException {
        XContentParser.Token token = parser.currentToken();
        if (token.equals((Object)XContentParser.Token.VALUE_NUMBER)) {
            return XContentMapValues.nodeIntegerValue(parser.intValue());
        }
        String precision = parser.text();
        try {
            return XContentMapValues.nodeIntegerValue(precision);
        }
        catch (NumberFormatException e) {
            int parsedPrecision = GeoUtils.geoHashLevelsForPrecision(precision);
            try {
                return GeoUtils.checkPrecisionRange(parsedPrecision);
            }
            catch (IllegalArgumentException e2) {
                throw new IllegalArgumentException("precision too high [" + precision + "]", e2);
            }
        }
    }

    public static int checkPrecisionRange(int precision) {
        if (precision < 1 || precision > 12) {
            throw new IllegalArgumentException("Invalid geohash aggregation precision of " + precision + ". Must be between 1 and 12.");
        }
        return precision;
    }

    public static double arcDistance(double lat1, double lon1, double lat2, double lon2) {
        return SloppyMath.haversinMeters((double)lat1, (double)lon1, (double)lat2, (double)lon2);
    }

    public static double planeDistance(double lat1, double lon1, double lat2, double lon2) {
        double x = Math.toRadians(lon2 - lon1) * Math.cos(Math.toRadians((lat2 + lat1) / 2.0));
        double y = Math.toRadians(lat2 - lat1);
        return Math.sqrt(x * x + y * y) * 6371008.7714;
    }

    public static SortedNumericDoubleValues distanceValues(final GeoDistance distance, final DistanceUnit unit, final MultiGeoPointValues geoPointValues, final GeoPoint ... fromPoints) {
        final GeoPointValues singleValues = FieldData.unwrapSingleton(geoPointValues);
        if (singleValues != null && fromPoints.length == 1) {
            return FieldData.singleton(new DoubleValues(){

                public boolean advanceExact(int doc) throws IOException {
                    return singleValues.advanceExact(doc);
                }

                public double doubleValue() throws IOException {
                    GeoPoint from = fromPoints[0];
                    GeoPoint to = singleValues.pointValue();
                    return distance.calculate(from.lat(), from.lon(), to.lat(), to.lon(), unit);
                }
            });
        }
        return new SortingNumericDoubleValues(){

            @Override
            public boolean advanceExact(int target) throws IOException {
                if (geoPointValues.advanceExact(target)) {
                    this.resize(geoPointValues.docValueCount() * fromPoints.length);
                    int v = 0;
                    for (int i = 0; i < geoPointValues.docValueCount(); ++i) {
                        GeoPoint point = geoPointValues.nextValue();
                        for (GeoPoint from : fromPoints) {
                            this.values[v] = distance.calculate(from.lat(), from.lon(), point.lat(), point.lon(), unit);
                            ++v;
                        }
                    }
                    this.sort();
                    return true;
                }
                return false;
            }
        };
    }

    public static double quantizeLon(double lon) {
        return GeoEncodingUtils.decodeLongitude((int)GeoEncodingUtils.encodeLongitude((double)lon));
    }

    public static double quantizeLat(double lat) {
        return GeoEncodingUtils.decodeLatitude((int)GeoEncodingUtils.encodeLatitude((double)lat));
    }

    public static double quantizeLonDown(double lon) {
        return GeoEncodingUtils.decodeLongitude((int)(GeoEncodingUtils.encodeLongitude((double)lon) - 1));
    }

    public static double quantizeLatUp(double lat) {
        return GeoEncodingUtils.decodeLatitude((int)(GeoEncodingUtils.encodeLatitude((double)lat) + 1));
    }

    private GeoUtils() {
    }

    public static enum EffectivePoint {
        TOP_LEFT,
        TOP_RIGHT,
        BOTTOM_LEFT,
        BOTTOM_RIGHT;

    }
}

