"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizePanels = exports.getMarkdownPanelHeight = exports.filterOutMarkdownPanels = exports.checkDashboardToolsAvailability = exports.buildMarkdownPanel = void 0;
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _config_builder = require("@kbn/lens-embeddable-utils/config_builder");
var _page_bundle_constants = require("@kbn/dashboard-plugin/common/page_bundle_constants");
var _constants = require("@kbn/dashboard-markdown/common/constants");
var _constants2 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getPanelWidth = chartType => {
  return _constants2.SMALL_CHART_TYPES.has(chartType) ? _constants2.SMALL_PANEL_WIDTH : _constants2.LARGE_PANEL_WIDTH;
};

/**
 * Shared availability handler for all dashboard tools.
 * Checks if dashboard tools are enabled via UI settings.
 */
const checkDashboardToolsAvailability = async ({
  uiSettings
}) => {
  const enabled = await uiSettings.get(_managementSettingsIds.AGENT_BUILDER_DASHBOARD_TOOLS_SETTING_ID);
  return {
    status: enabled ? 'available' : 'unavailable'
  };
};

/**
 * Calculates the height of a markdown panel based on content length.
 */
exports.checkDashboardToolsAvailability = checkDashboardToolsAvailability;
const calculateMarkdownPanelHeight = content => {
  const lineCount = content.split('\n').length;
  const estimatedHeight = lineCount + 2;
  return Math.max(_constants2.MARKDOWN_MIN_HEIGHT, Math.min(_constants2.MARKDOWN_MAX_HEIGHT, estimatedHeight));
};

/**
 * Builds a markdown panel for dashboard summaries with dynamic height based on content.
 */
const buildMarkdownPanel = content => ({
  type: _constants.MARKDOWN_EMBEDDABLE_TYPE,
  config: {
    content
  },
  grid: {
    x: 0,
    y: 0,
    w: _constants2.MARKDOWN_PANEL_WIDTH,
    h: calculateMarkdownPanelHeight(content)
  }
});

/**
 * Returns the height of a markdown panel for use in offset calculations.
 */
exports.buildMarkdownPanel = buildMarkdownPanel;
const getMarkdownPanelHeight = content => calculateMarkdownPanelHeight(content);

/**
 * Filters out markdown panels from an array of dashboard panels.
 * Used when replacing the markdown summary during dashboard updates.
 * Note: Dashboard panels array can contain both panels and sections.
 */
exports.getMarkdownPanelHeight = getMarkdownPanelHeight;
const filterOutMarkdownPanels = panels => {
  var _panels$filter;
  return (_panels$filter = panels === null || panels === void 0 ? void 0 : panels.filter(item => !('type' in item) || item.type !== _constants.MARKDOWN_EMBEDDABLE_TYPE)) !== null && _panels$filter !== void 0 ? _panels$filter : [];
};

/**
 * Normalizes panel configurations to the correct DashboardPanel format.
 * This is a temporary function to handle lens API schema conversion.
 * @param panels - Array of panel configurations
 * @param yOffset - Optional Y offset for positioning (e.g., when a markdown panel is prepended)
 */
exports.filterOutMarkdownPanels = filterOutMarkdownPanels;
const normalizePanels = (panels, yOffset = 0) => {
  const panelConfigs = panels !== null && panels !== void 0 ? panels : [];
  const dashboardPanels = [];
  let currentX = 0;
  let currentY = yOffset;
  for (const panel of panelConfigs) {
    const config = panel;
    const w = getPanelWidth(config.type);

    // Check if panel fits in current row, if not move to next row
    if (currentX + w > _page_bundle_constants.DASHBOARD_GRID_COLUMN_COUNT) {
      currentX = 0;
      currentY += _constants2.DEFAULT_PANEL_HEIGHT;
    }
    dashboardPanels.push(buildLensPanelFromApi(config, {
      x: currentX,
      y: currentY,
      w,
      h: _constants2.DEFAULT_PANEL_HEIGHT
    }));
    currentX += w;
  }
  return dashboardPanels;
};
exports.normalizePanels = normalizePanels;
const buildLensPanelFromApi = (config, grid) => {
  var _ref, _lensAttributes$title;
  const lensAttributes = new _config_builder.LensConfigBuilder().fromAPIFormat(config);
  const lensConfig = {
    title: (_ref = (_lensAttributes$title = lensAttributes.title) !== null && _lensAttributes$title !== void 0 ? _lensAttributes$title : config.title) !== null && _ref !== void 0 ? _ref : 'Generated panel',
    attributes: lensAttributes
  };
  return {
    type: 'lens',
    grid,
    config: lensConfig
  };
};