"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.buildActiveMappings = buildActiveMappings;
exports.getBaseMappings = getBaseMappings;
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Creates an index mapping with the core properties required by saved object
 * indices, as well as the specified additional properties.
 *
 * @param typeDefinitions - the type definitions to build mapping from.
 */
function buildActiveMappings(typeDefinitions, _meta) {
  const mapping = getBaseMappings();
  return (0, _lodash.cloneDeep)({
    ...mapping,
    properties: validateAndMerge(mapping.properties, typeDefinitions),
    ...(_meta && {
      _meta
    })
  });
}

/**
 * Defines the mappings for the root fields, common to all saved objects.
 * These are present in all SO indices.
 *
 * @returns {IndexMapping}
 */
function getBaseMappings() {
  // Important: the ZDT algorithm won't trigger a reindex on documents
  // when changes on root field mappings are detected, meaning that adding
  // a non-indexed root field and then later switching it to indexed is
  // not support atm and would require changes to the ZDT algo.
  return {
    dynamic: 'strict',
    properties: {
      type: {
        type: 'keyword'
      },
      namespace: {
        type: 'keyword'
      },
      namespaces: {
        type: 'keyword'
      },
      originId: {
        type: 'keyword'
      },
      updated_at: {
        type: 'date'
      },
      updated_by: {
        type: 'keyword'
      },
      created_at: {
        type: 'date'
      },
      created_by: {
        type: 'keyword'
      },
      references: {
        type: 'nested',
        properties: {
          name: {
            type: 'keyword'
          },
          type: {
            type: 'keyword'
          },
          id: {
            type: 'keyword'
          }
        }
      },
      coreMigrationVersion: {
        type: 'keyword'
      },
      typeMigrationVersion: {
        type: 'version'
      },
      managed: {
        type: 'boolean'
      },
      accessControl: {
        dynamic: 'false',
        properties: {
          owner: {
            type: 'keyword'
          }
        }
      }
    }
  };
}
function validateAndMerge(dest, source) {
  Object.keys(source).forEach(k => {
    if (k.startsWith('_')) {
      throw new Error(`Invalid mapping "${k}". Mappings cannot start with _.`);
    }
    if (Object.hasOwn(dest, k)) {
      throw new Error(`Cannot redefine core mapping "${k}".`);
    }
  });
  return Object.assign(dest, source);
}