"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkDelete = void 0;
var _pMap = _interopRequireDefault(require("p-map"));
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsApiServer = require("@kbn/core-saved-objects-api-server");
var _constants = require("../constants");
var _utils = require("./utils");
var _delete_legacy_url_aliases = require("./internals/delete_legacy_url_aliases");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const performBulkDelete = async ({
  objects,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  extensions = {},
  logger,
  mappings
}) => {
  const {
    common: commonHelper,
    preflight: preflightHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const {
    refresh = _constants.DEFAULT_REFRESH_SETTING,
    force
  } = options;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const expectedBulkGetResults = presortObjectsByNamespaceType(objects, allowedTypes, registry, securityExtension);
  if (expectedBulkGetResults.length === 0) {
    return {
      statuses: []
    };
  }
  const multiNamespaceDocsResponse = await preflightHelper.preflightCheckForBulkDelete({
    expectedBulkGetResults,
    namespace
  });

  // First round of filtering (Left: object doesn't exist/doesn't exist in namespace, Right: good to proceed)
  const expectedMultiNamespaceResults = getExpectedBulkDeleteMultiNamespaceDocsResults({
    expectedBulkGetResults,
    multiNamespaceDocsResponse,
    namespace,
    force
  }, registry);
  let expectedResults;
  if (securityExtension) {
    // Perform Auth Check (on both L/R, we'll deal with that later)
    const authObjects = expectedMultiNamespaceResults.map(element => {
      var _preflightResult$_sou, _preflightResult$_sou2, _preflightResult$_sou3, _preflightResult$_sou4;
      const index = element.value.esRequestIndex;
      const {
        type,
        id
      } = element.value;
      const preflightResult = index !== undefined ? multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.body.docs[index] : undefined;

      // @ts-expect-error MultiGetHit._source is optional
      const accessControl = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou = preflightResult._source) === null || _preflightResult$_sou === void 0 ? void 0 : _preflightResult$_sou.accessControl;
      const name = preflightResult ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.getName(registry.getNameAttribute(type),
      // @ts-expect-error MultiGetHit._source is optional
      {
        attributes: preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou2 = preflightResult._source) === null || _preflightResult$_sou2 === void 0 ? void 0 : _preflightResult$_sou2[type]
      }) : undefined;
      return {
        type,
        id,
        name,
        ...(accessControl ? {
          accessControl
        } : {}),
        // @ts-expect-error MultiGetHit._source is optional
        existingNamespaces: (_preflightResult$_sou3 = preflightResult === null || preflightResult === void 0 ? void 0 : (_preflightResult$_sou4 = preflightResult._source) === null || _preflightResult$_sou4 === void 0 ? void 0 : _preflightResult$_sou4.namespaces) !== null && _preflightResult$_sou3 !== void 0 ? _preflightResult$_sou3 : []
      };
    });
    const authorizationResult = await securityExtension.authorizeBulkDelete({
      namespace,
      objects: authObjects
    });
    const inaccessibleObjects = authorizationResult !== null && authorizationResult !== void 0 && authorizationResult.inaccessibleObjects ? Array.from(authorizationResult.inaccessibleObjects) : [];
    expectedResults = await securityExtension.filterInaccessibleObjectsForBulkAction(expectedMultiNamespaceResults, inaccessibleObjects, 'bulk_delete', true // reindex of esRequestIndex field needed to map subsequent bulk delete results below
    );
  } else expectedResults = expectedMultiNamespaceResults;

  // Filter valid objects
  const validObjects = expectedResults.filter(_coreSavedObjectsApiServer.isRight);
  if (validObjects.length === 0) {
    // We only have error results; return early.
    const savedObjects = expectedResults.map(expectedResult => {
      return {
        ...expectedResult.value,
        success: false
      };
    });
    return {
      statuses: [...savedObjects]
    };
  }

  // Create the bulkDeleteParams
  const bulkDeleteParams = [];
  validObjects.map(expectedResult => {
    bulkDeleteParams.push({
      delete: {
        _id: serializer.generateRawId(namespace, expectedResult.value.type, expectedResult.value.id),
        _index: commonHelper.getIndexForType(expectedResult.value.type),
        ...(0, _utils.getExpectedVersionProperties)(undefined)
      }
    });
  });
  const bulkDeleteResponse = bulkDeleteParams.length ? await client.bulk({
    refresh,
    operations: bulkDeleteParams,
    require_alias: true
  }) : undefined;

  // extracted to ensure consistency in the error results returned
  let errorResult;
  const objectsToDeleteAliasesFor = [];
  const savedObjects = expectedResults.map(expectedResult => {
    if ((0, _coreSavedObjectsApiServer.isLeft)(expectedResult)) {
      return {
        ...expectedResult.value,
        success: false
      };
    }
    const {
      type,
      id,
      namespaces,
      esRequestIndex: esBulkDeleteRequestIndex
    } = expectedResult.value;
    // we assume this wouldn't happen but is needed to ensure type consistency
    if (bulkDeleteResponse === undefined) {
      throw new Error(`Unexpected error in bulkDelete saved objects: bulkDeleteResponse is undefined`);
    }
    const rawResponse = Object.values(bulkDeleteResponse.items[esBulkDeleteRequestIndex])[0];
    const error = (0, _utils.getBulkOperationError)(type, id, rawResponse);
    if (error) {
      errorResult = {
        success: false,
        type,
        id,
        error
      };
      return errorResult;
    }
    if (rawResponse.result === 'not_found') {
      errorResult = {
        success: false,
        type,
        id,
        error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
      };
      return errorResult;
    }
    if (rawResponse.result === 'deleted') {
      // `namespaces` should only exist in the expectedResult.value if the type is multi-namespace.
      if (namespaces) {
        objectsToDeleteAliasesFor.push({
          type,
          id,
          ...(namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) ? {
            namespaces: [],
            deleteBehavior: 'exclusive'
          } : {
            namespaces,
            deleteBehavior: 'inclusive'
          })
        });
      }
    }
    const successfulResult = {
      success: true,
      id,
      type
    };
    return successfulResult;
  });

  // Delete aliases if necessary, ensuring we don't have too many concurrent operations running.
  const mapper = async ({
    type,
    id,
    namespaces,
    deleteBehavior
  }) => {
    await (0, _delete_legacy_url_aliases.deleteLegacyUrlAliases)({
      mappings,
      registry,
      client,
      getIndexForType: commonHelper.getIndexForType.bind(commonHelper),
      type,
      id,
      namespaces,
      deleteBehavior
    }).catch(err => {
      logger.error(`Unable to delete aliases when deleting an object: ${err.message}`);
    });
  };
  await (0, _pMap.default)(objectsToDeleteAliasesFor, mapper, {
    concurrency: _constants.MAX_CONCURRENT_ALIAS_DELETIONS
  });
  return {
    statuses: [...savedObjects]
  };
};

/**
 * Performs initial checks on object type validity and flags multi-namespace objects for preflight checks by adding an `esRequestIndex`
 * @returns array BulkDeleteExpectedBulkGetResult[]
 */
exports.performBulkDelete = performBulkDelete;
function presortObjectsByNamespaceType(objects, allowedTypes, registry, securityExtension) {
  let bulkGetRequestIndexCounter = 0;
  return objects.map(object => {
    const {
      type,
      id
    } = object;
    if (!allowedTypes.includes(type)) {
      return (0, _coreSavedObjectsApiServer.left)({
        id,
        type,
        error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createUnsupportedTypeError(type))
      });
    }
    const requiresNamespacesCheck = registry.isMultiNamespace(type);
    return (0, _coreSavedObjectsApiServer.right)({
      type,
      id,
      fields: securityExtension !== null && securityExtension !== void 0 && securityExtension.includeSavedObjectNames() ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.getIncludedNameFields(type, registry.getNameAttribute(type)) : [],
      ...(requiresNamespacesCheck && {
        esRequestIndex: bulkGetRequestIndexCounter++
      })
    });
  });
}

/**
 * @returns array of objects sorted by expected delete success or failure result
 * @internal
 */
function getExpectedBulkDeleteMultiNamespaceDocsResults(params, registry) {
  const {
    expectedBulkGetResults,
    multiNamespaceDocsResponse,
    namespace,
    force
  } = params;
  let indexCounter = 0;
  const expectedBulkDeleteMultiNamespaceDocsResults = expectedBulkGetResults.map(expectedBulkGetResult => {
    if ((0, _coreSavedObjectsApiServer.isLeft)(expectedBulkGetResult)) {
      return {
        ...expectedBulkGetResult
      };
    }
    const {
      esRequestIndex: esBulkGetRequestIndex,
      id,
      type,
      accessControl
    } = expectedBulkGetResult.value;
    let namespaces;
    if (esBulkGetRequestIndex !== undefined) {
      var _source$namespaces;
      const indexFound = (multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.statusCode) !== 404;
      const actualResult = indexFound ? multiNamespaceDocsResponse === null || multiNamespaceDocsResponse === void 0 ? void 0 : multiNamespaceDocsResponse.body.docs[esBulkGetRequestIndex] : undefined;
      const docFound = indexFound && (0, _utils.isMgetDoc)(actualResult) && actualResult.found;

      // return an error if the doc isn't found at all or the doc doesn't exist in the namespaces
      if (!docFound) {
        return (0, _coreSavedObjectsApiServer.left)({
          id,
          type,
          error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
        });
      }
      // the following check should be redundant since we're retrieving the docs from elasticsearch but we check just to make sure
      if (!(0, _utils.rawDocExistsInNamespace)(registry, actualResult, namespace)) {
        return (0, _coreSavedObjectsApiServer.left)({
          id,
          type,
          error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
        });
      }
      // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
      namespaces = (_source$namespaces = actualResult._source.namespaces) !== null && _source$namespaces !== void 0 ? _source$namespaces : [_coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace)];
      const useForce = force && force === true;
      // the document is shared to more than one space and can only be deleted by force.
      if (!useForce && (namespaces.length > 1 || namespaces.includes(_coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING))) {
        return (0, _coreSavedObjectsApiServer.left)({
          success: false,
          id,
          type,
          error: (0, _coreSavedObjectsServer.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('Unable to delete saved object that exists in multiple namespaces, use the `force` option to delete it anyway'))
        });
      }
    }
    // contains all objects that passed initial preflight checks, including single namespace objects that skipped the mget call
    // single namespace objects will have namespaces:undefined
    const expectedResult = {
      type,
      id,
      namespaces,
      ...(accessControl ? {
        accessControl
      } : {}),
      esRequestIndex: indexCounter++
    };
    return (0, _coreSavedObjectsApiServer.right)(expectedResult);
  });
  return expectedBulkDeleteMultiNamespaceDocsResults;
}