"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useResponsiveMenu = useResponsiveMenu;
var _react = require("react");
var _cache_menu_item_heights = require("../utils/cache_menu_item_heights");
var _count_visible_menu_items = require("../utils/count_visible_menu_items");
var _get_style_property = require("../utils/get_style_property");
var _use_raf_debounced = require("./use_raf_debounced");
var _use_stable_menu_items_reference = require("./use_stable_menu_items_reference");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/**
 * Custom hook that measures the primary nav container and decides which items can stay visible.
 * Items that no longer fit are moved into the overflow "More" menu so the sidebar keeps its size
 * limits when resizing, zooming, or collapsing.
 *
 * @param isCollapsed - whether the side nav is currently collapsed (affects layout recalculation).
 * @param items - all primary navigation items, in priority order.
 * @returns an object containing:
 * - `primaryMenuRef` - a ref to the primary menu.
 * - `visibleMenuItems` - the visible menu items.
 * - `overflowMenuItems` - the overflow menu items.
 */
function useResponsiveMenu(isCollapsed, items) {
  const primaryMenuRef = (0, _react.useRef)(null);
  const heightsCacheRef = (0, _react.useRef)([]);
  const [visibleCount, setVisibleCount] = (0, _react.useState)(items.length);
  const visibleMenuItems = (0, _react.useMemo)(() => items.slice(0, visibleCount), [items, visibleCount]);
  const overflowMenuItems = (0, _react.useMemo)(() => items.slice(visibleCount), [items, visibleCount]);
  const stableItemsReference = (0, _use_stable_menu_items_reference.useStableMenuItemsReference)(items);
  const recalculateMenuLayout = (0, _react.useCallback)(() => {
    if (!primaryMenuRef.current) return;

    // Primary menu
    const menu = primaryMenuRef.current;
    const menuHeight = menu.clientHeight;

    // 1. Cache the heights of all children
    (0, _cache_menu_item_heights.cacheMenuItemHeights)(heightsCacheRef, menu, stableItemsReference);
    if (heightsCacheRef.current.length === 0) return;

    // Primary menu items
    const childrenHeights = heightsCacheRef.current;
    const childrenGap = (0, _get_style_property.getStyleProperty)(menu, 'gap');

    // 2. Calculate the number of visible menu items
    const nextVisibleCount = (0, _count_visible_menu_items.countVisibleMenuItems)(childrenHeights, childrenGap, menuHeight);

    // 3. Update the visible count if needed
    setVisibleCount(nextVisibleCount);
  }, [stableItemsReference]);
  const [scheduleRecalculation, cancelRecalculation] = (0, _use_raf_debounced.useRafDebouncedCallback)(recalculateMenuLayout);
  (0, _react.useLayoutEffect)(() => {
    // Invalidate the cache when items change
    setVisibleCount(stableItemsReference.length);
    heightsCacheRef.current = [];
    const observer = new ResizeObserver(() => {
      scheduleRecalculation();
    });
    if (primaryMenuRef.current) {
      observer.observe(primaryMenuRef.current);
    }

    // Initial calculation
    scheduleRecalculation();
    return () => {
      observer.disconnect();
      cancelRecalculation();
    };
  }, [isCollapsed, stableItemsReference, scheduleRecalculation, cancelRecalculation]);
  return {
    primaryMenuRef,
    visibleMenuItems,
    overflowMenuItems
  };
}