"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataStreamsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pLimit = _interopRequireDefault(require("p-limit"));
var _dataStreams = require("@kbn/data-streams");
var _rxjs = require("rxjs");
var _config = require("./config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

/** @internal */
class DataStreamsService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "dataStreamDefinitions", new Map());
    (0, _defineProperty2.default)(this, "dataStreamClients", new Map());
    this.coreContext = coreContext;
    this.logger = this.coreContext.logger.get('data-streams');
  }
  async setup() {
    const dataStreamsConfig = await (0, _rxjs.firstValueFrom)(this.coreContext.configService.atPath(_config.config.path));
    this.config = new _config.DataStreamsConfig(dataStreamsConfig);
    return {
      registerDataStream: dataStreamDefinition => {
        if (this.dataStreamDefinitions.has(dataStreamDefinition.name)) {
          throw new Error(`Data stream ${dataStreamDefinition.name} is already registered.`);
        }
        this.dataStreamDefinitions.set(dataStreamDefinition.name, dataStreamDefinition);
        this.dataStreamClients.set(dataStreamDefinition.name, undefined);
      }
    };
  }
  async initializeDataStream(dataStreamName, elasticsearchClient, lazyCreation) {
    if (this.dataStreamClients.get(dataStreamName)) {
      // already initialized
      return;
    }
    const dataStreamDefinition = this.dataStreamDefinitions.get(dataStreamName);
    if (!dataStreamDefinition) {
      throw new Error(`Data stream ${dataStreamName} is not registered.`);
    }
    const maybeInitializedClient = await _dataStreams.DataStreamClient.initialize({
      dataStream: dataStreamDefinition,
      elasticsearchClient,
      logger: this.logger,
      lazyCreation
    });
    this.dataStreamClients.set(dataStreamName, maybeInitializedClient);
  }
  async initializeAllDataStreams(elasticsearchClient) {
    const limit = (0, _pLimit.default)(5);
    const setupPromises = [];
    const allDataStreamNames = Array.from(this.dataStreamDefinitions.keys());
    for (const dataStreamName of allDataStreamNames) {
      setupPromises.push(limit(() => this.initializeDataStream(dataStreamName, elasticsearchClient, true)));
    }
    await Promise.all(setupPromises);
  }
  async start({
    elasticsearch
  }) {
    var _this$config;
    const elasticsearchClient = elasticsearch.client.asInternalUser;
    const skipInitialization = (_this$config = this.config) === null || _this$config === void 0 ? void 0 : _this$config.migrations.skip;
    if (skipInitialization) {
      this.logger.warn('Skipping eager initialization and migrations of all data streams on startup.');
    } else {
      this.logger.debug('Initializing all data streams');
      await this.initializeAllDataStreams(elasticsearchClient);
    }
    return {
      initializeClient: async dataStreamName => {
        if (!this.dataStreamDefinitions.has(dataStreamName)) {
          throw new Error(`Data stream ${dataStreamName} is not registered.`);
        }

        // initialize the data stream if it is not already initialized, disable lazy creation
        await this.initializeDataStream(dataStreamName, elasticsearchClient, false);
        const dataStreamClient = this.dataStreamClients.get(dataStreamName);
        if (!dataStreamClient) {
          throw new Error(`Data stream client for ${dataStreamName} is not initialized. Are you sure you are providing the same definition as in setup?`);
        }
        return dataStreamClient;
      }
    };
  }
  stop() {}
}
exports.DataStreamsService = DataStreamsService;