"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.NotionConnector = void 0;
var _i18n = require("@kbn/i18n");
var _v = require("@kbn/zod/v4");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const NotionConnector = exports.NotionConnector = {
  metadata: {
    id: '.notion',
    displayName: 'Notion',
    description: _i18n.i18n.translate('core.kibanaConnectorSpecs.notion.metadata.description', {
      defaultMessage: 'Explore content and databases in Notion'
    }),
    minimumLicense: 'enterprise',
    supportedFeatureIds: ['workflows']
  },
  auth: {
    types: ['bearer'],
    headers: {
      'Notion-Version': '2025-09-03'
    }
  },
  actions: {
    // https://developers.notion.com/reference/post-search
    searchPageOrDSByTitle: {
      isTool: false,
      input: _v.z.object({
        query: _v.z.string(),
        queryObjectType: _v.z.enum(['page', 'data_source']),
        startCursor: _v.z.string().optional(),
        pageSize: _v.z.number().optional()
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.post('https://api.notion.com/v1/search', {
          query: typedInput.query,
          filter: {
            value: typedInput.queryObjectType,
            property: 'object'
          },
          ...(typedInput.startCursor && {
            start_cursor: typedInput.startCursor
          }),
          ...(typedInput.pageSize && {
            page_size: typedInput.pageSize
          })
        });
        return response.data;
      }
    },
    // https://developers.notion.com/reference/retrieve-a-page
    getPage: {
      isTool: false,
      input: _v.z.object({
        pageId: _v.z.string()
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.get(`https://api.notion.com/v1/pages/${typedInput.pageId}`, {});
        return response.data;
      }
    },
    // https://developers.notion.com/reference/retrieve-a-data-source
    getDataSource: {
      isTool: false,
      input: _v.z.object({
        dataSourceId: _v.z.string()
      }),
      handler: async (ctx, input) => {
        const typedInput = input;
        const response = await ctx.client.get(`https://api.notion.com/v1/data_sources/${typedInput.dataSourceId}`, {});
        return response.data;
      }
    },
    // https://developers.notion.com/reference/query-a-data-source
    queryDataSource: {
      isTool: false,
      input: _v.z.object({
        dataSourceId: _v.z.string(),
        filter: _v.z.string().optional(),
        startCursor: _v.z.string().optional(),
        pageSize: _v.z.number().optional()
      }),
      handler: async (ctx, input) => {
        const typedInput = input;

        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        let requestData = {
          page_size: typedInput.pageSize,
          start_cursor: typedInput.startCursor
        };
        if (typedInput.filter) {
          requestData = {
            ...requestData,
            filter: JSON.parse(typedInput.filter)
          };
        }
        const response = await ctx.client.post(`https://api.notion.com/v1/data_sources/${typedInput.dataSourceId}/query`, requestData);
        return response.data;
      }
    }
  },
  test: {
    description: _i18n.i18n.translate('ore.kibanaConnectorSpecs.notion.test.description', {
      defaultMessage: 'Verifies Notion connection by fetching metadata about given data source'
    }),
    // TODO: might need to accept some input here in order to pass to the API endpoint to test
    // if listing all users feels a bit too much
    handler: async ctx => {
      ctx.log.debug('Notion test handler');
      try {
        const response = await ctx.client.get('https://api.notion.com/v1/users');
        const numOfUsers = response.data.results.length;
        return {
          ok: true,
          message: `Successfully connected to Notion API: found ${numOfUsers} users`
        };
      } catch (error) {
        return {
          ok: false,
          message: error.message
        };
      }
    }
  }
};