"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CloudUrlsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("../common/constants");
var _utils = require("../common/utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Service that manages all URLs for the Cloud plugin.
 */
class CloudUrlsService {
  constructor() {
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "coreSetup", void 0);
    (0, _defineProperty2.default)(this, "kibanaUrl", void 0);
  }
  setup(config, coreSetup, kibanaUrl) {
    this.config = config;
    this.coreSetup = coreSetup;
    this.kibanaUrl = kibanaUrl;
  }

  /**
   * Returns the set of "basic" URLs. No special privileges needed
   */
  getUrls() {
    if (!this.config) {
      throw new Error('Cloud configuration is not set up');
    }
    const kibanaUrl = this.kibanaUrl;
    const {
      base_url: baseUrl,
      profile_url: profileUrl,
      organization_url: organizationUrl,
      deployments_url: deploymentsUrl,
      deployment_url: deploymentUrl,
      performance_url: performanceUrl,
      users_and_roles_url: usersAndRolesUrl,
      projects_url: projectsUrl
    } = this.config;
    const fullCloudDeploymentsUrl = (0, _utils.getFullCloudUrl)(baseUrl, deploymentsUrl);
    const fullCloudDeploymentUrl = (0, _utils.getFullCloudUrl)(baseUrl, deploymentUrl);
    const fullCloudProfileUrl = (0, _utils.getFullCloudUrl)(baseUrl, profileUrl);
    const fullCloudOrganizationUrl = (0, _utils.getFullCloudUrl)(baseUrl, organizationUrl);
    const fullCloudPerformanceUrl = (0, _utils.getFullCloudUrl)(baseUrl, performanceUrl);
    const fullCloudUsersAndRolesUrl = (0, _utils.getFullCloudUrl)(baseUrl, usersAndRolesUrl);
    const fullCloudProjectsUrl = (0, _utils.getFullCloudUrl)(baseUrl, projectsUrl);
    const fullCloudSnapshotsUrl = `${fullCloudDeploymentUrl}/${_constants.CLOUD_SNAPSHOTS_PATH}`;
    return {
      baseUrl,
      kibanaUrl,
      deploymentsUrl: fullCloudDeploymentsUrl,
      deploymentUrl: fullCloudDeploymentUrl,
      profileUrl: fullCloudProfileUrl,
      organizationUrl: fullCloudOrganizationUrl,
      snapshotsUrl: fullCloudSnapshotsUrl,
      performanceUrl: fullCloudPerformanceUrl,
      usersAndRolesUrl: fullCloudUsersAndRolesUrl,
      projectsUrl: fullCloudProjectsUrl
    };
  }

  /**
   * Returns the set of "privilged" URLs. Each requires a specific privilege to access.
   */
  async getPrivilegedUrls() {
    if (!this.config) {
      throw new Error('Cloud configuration is not set up');
    }
    const showBillingUrl = (await this.getCurrentUserRoles()).includes(_constants.CLOUD_USER_BILLING_ADMIN_ROLE);
    const conditionalFullCloudBillingUrl = showBillingUrl ? (0, _utils.getFullCloudUrl)(this.config.base_url, this.config.billing_url) : undefined;
    return {
      billingUrl: conditionalFullCloudBillingUrl
    };
  }

  /**
   * Needed for determining access to privileged URLs, such as billing.
   */
  async getCurrentUserRoles() {
    var _this$coreSetup;
    const [coreStart] = (await ((_this$coreSetup = this.coreSetup) === null || _this$coreSetup === void 0 ? void 0 : _this$coreSetup.getStartServices())) || [];
    if (!coreStart) {
      throw new Error('Core setup is not available');
    }
    let userRoles = [];
    try {
      userRoles = (await coreStart.security.authc.getCurrentUser()).roles;
    } catch (e) {
      // If if no user is available, we just return an empty array of roles
    }
    return userRoles;
  }
}
exports.CloudUrlsService = CloudUrlsService;