"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerClustersRoute = void 0;
var _configSchema = require("@kbn/config-schema");
var _i18n = require("@kbn/i18n");
var _axios = _interopRequireDefault(require("axios"));
var _cloud_connect_client = require("../services/cloud_connect_client");
var _create_storage_service = require("../lib/create_storage_service");
var _inference_ccm = require("../services/inference_ccm");
var _update_default_llm_actions = require("../lib/update_default_llm_actions");
var _wait_for_inference_endpoint = require("../lib/wait_for_inference_endpoint");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const registerClustersRoute = ({
  router,
  logger,
  getStartServices,
  cloudApiUrl
}) => {
  router.get({
    path: '/internal/cloud_connect/cluster_details',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates to the Cloud Connect API for authentication and authorization.'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      // Initialize storage service for retrieving the API key
      const storageService = await (0, _create_storage_service.createStorageService)(context, getStartServices, logger);

      // Retrieve stored API key
      const apiKeyData = await storageService.getApiKey();
      if (!apiKeyData) {
        logger.warn('No API key found in saved object');
        return response.customError({
          statusCode: 503,
          body: {
            message: 'Failed to retrieve API key from saved object'
          }
        });
      }

      // Fetch cluster details from Cloud Connect API
      const cloudConnectClient = new _cloud_connect_client.CloudConnectClient(logger, cloudApiUrl);
      const clusterDetails = await cloudConnectClient.getClusterDetails(apiKeyData.apiKey, apiKeyData.clusterId);
      logger.debug(`Successfully retrieved cluster details: ${clusterDetails.id}`);

      // Fetch subscription state for the organization
      try {
        const subscription = await cloudConnectClient.getOrganizationSubscription(apiKeyData.apiKey, clusterDetails.metadata.organization_id);
        return response.ok({
          body: {
            ...clusterDetails,
            metadata: {
              ...clusterDetails.metadata,
              subscription: subscription.state
            }
          }
        });
      } catch (subscriptionError) {
        // Log the error but return cluster details without subscription
        logger.warn(`Failed to fetch subscription for organization ${clusterDetails.metadata.organization_id}`, {
          error: subscriptionError
        });

        // Return cluster details without subscription field
        return response.ok({
          body: clusterDetails
        });
      }
    } catch (error) {
      logger.error('Failed to retrieve cluster details', {
        error
      });
      if (_axios.default.isAxiosError(error)) {
        var _error$response, _error$response2, _errorData$errors, _errorData$errors$;
        const errorData = (_error$response = error.response) === null || _error$response === void 0 ? void 0 : _error$response.data;
        const apiStatusCode = (_error$response2 = error.response) === null || _error$response2 === void 0 ? void 0 : _error$response2.status;

        // Extract error message from backend response
        const errorMessage = (errorData === null || errorData === void 0 ? void 0 : (_errorData$errors = errorData.errors) === null || _errorData$errors === void 0 ? void 0 : (_errorData$errors$ = _errorData$errors[0]) === null || _errorData$errors$ === void 0 ? void 0 : _errorData$errors$.message) || (errorData === null || errorData === void 0 ? void 0 : errorData.message) || 'Failed to retrieve cluster details';

        // Use 500 for 401 errors to prevent Kibana logout
        // For all other errors, use the status code from the API
        const statusCode = apiStatusCode === 401 ? 500 : apiStatusCode || 500;
        return response.customError({
          statusCode,
          body: {
            message: errorMessage
          }
        });
      }
      return response.customError({
        statusCode: 500,
        body: {
          message: error
        }
      });
    }
  });
  router.delete({
    path: '/internal/cloud_connect/cluster',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates to the Cloud Connect API for authentication and authorization.'
      }
    },
    validate: false,
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      // Initialize storage service for retrieving and deleting the API key
      const storageService = await (0, _create_storage_service.createStorageService)(context, getStartServices, logger);

      // Retrieve stored API key
      const apiKeyData = await storageService.getApiKey();
      if (!apiKeyData) {
        logger.warn('No API key found in saved object');
        return response.customError({
          statusCode: 503,
          body: {
            message: 'Failed to retrieve API key from saved object'
          }
        });
      }

      // Delete cluster from Cloud Connect API first
      const cloudConnectClient = new _cloud_connect_client.CloudConnectClient(logger, cloudApiUrl);
      await cloudConnectClient.deleteCluster(apiKeyData.apiKey, apiKeyData.clusterId);
      logger.debug(`Successfully deleted cluster from Cloud API: ${apiKeyData.clusterId}`);

      // Only delete the saved object after successful Cloud API deletion
      await storageService.deleteApiKey();
      logger.info('Cluster disconnected successfully - API key removed');
      return response.ok({
        body: {
          success: true,
          message: 'Cluster disconnected successfully'
        }
      });
    } catch (error) {
      logger.error('Failed to disconnect cluster', {
        error
      });
      if (_axios.default.isAxiosError(error)) {
        var _error$response3;
        const errorData = (_error$response3 = error.response) === null || _error$response3 === void 0 ? void 0 : _error$response3.data;
        return response.customError({
          statusCode: 500,
          body: errorData || {
            message: 'An error occurred while disconnecting the cluster'
          }
        });
      }
      return response.customError({
        statusCode: 500,
        body: {
          message: 'An error occurred while disconnecting the cluster'
        }
      });
    }
  });
  router.put({
    path: '/internal/cloud_connect/cluster_details',
    security: {
      authz: {
        enabled: false,
        reason: 'This route delegates to the Cloud Connect API for authentication and authorization.'
      }
    },
    validate: {
      body: _configSchema.schema.object({
        services: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.object({
          enabled: _configSchema.schema.boolean()
        }))
      })
    },
    options: {
      access: 'internal'
    }
  }, async (context, request, response) => {
    try {
      const coreContext = await context.core;
      const storageService = await (0, _create_storage_service.createStorageService)(context, getStartServices, logger);

      // Retrieve stored API key
      const apiKeyData = await storageService.getApiKey();
      if (!apiKeyData) {
        logger.warn('No API key found in saved object');
        return response.customError({
          statusCode: 503,
          body: {
            message: 'Failed to retrieve API key from saved object'
          }
        });
      }

      // Update cluster services via Cloud Connect API
      const cloudConnectClient = new _cloud_connect_client.CloudConnectClient(logger, cloudApiUrl);
      const updatedCluster = await cloudConnectClient.updateCluster(apiKeyData.apiKey, apiKeyData.clusterId, {
        services: request.body.services
      });
      logger.debug(`Successfully updated cluster services: ${updatedCluster.id}`);

      // If EIS service is enabled, the response will return a keys.eis string that
      // needs to be stored in ES for inference nodes to use EIS.
      const eisRequest = request.body.services.eis;
      if (eisRequest) {
        var _updatedCluster$keys;
        const esClient = coreContext.elasticsearch.client.asCurrentUser;
        const eisKey = (_updatedCluster$keys = updatedCluster.keys) === null || _updatedCluster$keys === void 0 ? void 0 : _updatedCluster$keys.eis;

        // When enabling EIS, it should(TM) always return an eisKey. But in case it
        // doesnt, lets rollback the change and return an error since we wont be able
        // to configure inference CCM without the key.
        if (eisRequest !== null && eisRequest !== void 0 && eisRequest.enabled && !eisKey) {
          logger.error('EIS was enabled but Cloud API did not return an API key for Cloud Connect inference');
          try {
            await cloudConnectClient.updateCluster(apiKeyData.apiKey, apiKeyData.clusterId, {
              services: {
                eis: {
                  enabled: false
                }
              }
            });
            logger.info('Successfully rolled back EIS enablement in Cloud API');
          } catch (rollbackError) {
            logger.error('Failed to rollback Cloud API changes', {
              error: rollbackError
            });
          }
          return response.customError({
            statusCode: 500,
            body: {
              message: 'EIS was enabled but Cloud API did not return an API key'
            }
          });
        }

        // Update elastic inference ccm settings
        try {
          if (eisRequest !== null && eisRequest !== void 0 && eisRequest.enabled) {
            await (0, _inference_ccm.enableInferenceCCM)(esClient, eisKey, logger);
          } else {
            await (0, _inference_ccm.disableInferenceCCM)(esClient, logger);
          }
        } catch (inferenceError) {
          logger.error('Failed to update Cloud Connect inference settings, rolling back', {
            error: inferenceError
          });

          // If enabling the inference CCM settings failed, we need to rollback the service state
          const rollbackEnabled = !eisRequest.enabled;
          try {
            await cloudConnectClient.updateCluster(apiKeyData.apiKey, apiKeyData.clusterId, {
              services: {
                eis: {
                  enabled: rollbackEnabled
                }
              }
            });
            logger.info(`Successfully rolled back EIS to enabled=${rollbackEnabled} in Cloud API`);
          } catch (rollbackError) {
            logger.error('Failed to rollback Cloud API changes', {
              error: rollbackError
            });
            return response.customError({
              statusCode: 500,
              body: {
                message: 'Failed to update Cloud Connect inference settings and rollback also failed',
                attributes: {
                  inferenceError: inferenceError.message,
                  rollbackError: rollbackError.message
                }
              }
            });
          }
          return response.customError({
            statusCode: 500,
            body: {
              message: inferenceError.message
            }
          });
        }

        // Update default LLM actions if needed, this always needs to happen
        // after updating the service.
        //
        // We poll for a known inference endpoint to verify CCM setup is complete
        // before creating the default LLM connectors. This avoids race conditions
        // where the inference endpoints aren't ready yet.
        try {
          if (eisRequest !== null && eisRequest !== void 0 && eisRequest.enabled) {
            await (0, _wait_for_inference_endpoint.waitForInferenceEndpoint)(esClient, logger);
            await (0, _update_default_llm_actions.updateDefaultLLMActions)(getStartServices, request, logger);
          }
        } catch (llmActionsError) {
          logger.warn('Failed to update default LLM actions', {
            error: llmActionsError
          });
        }
      }
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      logger.error('Failed to update cluster services', {
        error
      });
      if (_axios.default.isAxiosError(error)) {
        var _error$response4, _errorData$errors2, _errorData$errors2$;
        const errorData = (_error$response4 = error.response) === null || _error$response4 === void 0 ? void 0 : _error$response4.data;

        // Extract error code from Cloud Connect API error format
        // API returns: { "errors": [{ "code": "...", "message": "..." }] }
        const errorCode = errorData === null || errorData === void 0 ? void 0 : (_errorData$errors2 = errorData.errors) === null || _errorData$errors2 === void 0 ? void 0 : (_errorData$errors2$ = _errorData$errors2[0]) === null || _errorData$errors2$ === void 0 ? void 0 : _errorData$errors2$.code;

        // Check for specific error codes and return user-friendly messages
        let errorMessage;
        if (errorCode === 'clusters.patch_cluster.invalid_state') {
          errorMessage = _i18n.i18n.translate('xpack.cloudConnect.clusterUpdate.invalidState', {
            defaultMessage: 'The API is still completing an operation, please try again later'
          });
        } else {
          var _errorData$errors3, _errorData$errors3$;
          errorMessage = (errorData === null || errorData === void 0 ? void 0 : (_errorData$errors3 = errorData.errors) === null || _errorData$errors3 === void 0 ? void 0 : (_errorData$errors3$ = _errorData$errors3[0]) === null || _errorData$errors3$ === void 0 ? void 0 : _errorData$errors3$.message) || (errorData === null || errorData === void 0 ? void 0 : errorData.message) || 'An error occurred while updating cluster services';
        }
        return response.customError({
          statusCode: 500,
          body: {
            message: errorMessage,
            ...(errorCode && {
              code: errorCode
            })
          }
        });
      }
      return response.customError({
        statusCode: 500,
        body: {
          message: 'An error occurred while updating cluster services'
        }
      });
    }
  });
};
exports.registerClustersRoute = registerClustersRoute;