"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createRuleRoute = void 0;
var _create = require("../../../../../common/routes/rule/apis/create");
var _response = require("../../../../../common/routes/rule/response");
var _lib = require("../../../../lib");
var _types = require("../../../../types");
var _constants = require("../../../constants");
var _lib2 = require("../../../lib");
var _validate_internal_rule_type = require("../../../lib/validate_internal_rule_type");
var _transforms = require("../../transforms");
var _validation = require("../../validation");
var _transforms2 = require("./transforms");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createRuleRoute = ({
  router,
  licenseState,
  usageCounter
}) => {
  router.post({
    path: `${_types.BASE_ALERTING_API_PATH}/rule/{id?}`,
    security: _constants.DEFAULT_ALERTING_ROUTE_SECURITY,
    options: {
      access: 'public',
      summary: `Create a rule`,
      tags: ['oas-tag:alerting'],
      oasOperationObject: _create.createRuleParamsExamplesV1
    },
    validate: {
      request: {
        body: _create.createBodySchemaV1,
        params: _create.createParamsSchemaV1
      },
      response: {
        200: {
          body: () => _response.ruleResponseSchemaV1,
          description: 'Indicates a successful call.'
        },
        400: {
          description: 'Indicates an invalid schema or parameters.'
        },
        403: {
          description: 'Indicates that this call is forbidden.'
        },
        409: {
          description: 'Indicates that the rule id is already in use.'
        }
      }
    }
  }, (0, _lib2.handleDisabledApiKeysError)(router.handleLegacyErrors((0, _lib2.verifyAccessAndContext)(licenseState, async function (context, req, res) {
    const alertingContext = await context.alerting;
    const rulesClient = await alertingContext.getRulesClient();
    const actionsClient = (await context.actions).getActionsClient();
    const ruleTypes = alertingContext.listTypes();

    // Assert versioned inputs
    const createRuleData = req.body;
    const params = req.params;
    (0, _lib2.countUsageOfPredefinedIds)({
      predefinedId: params === null || params === void 0 ? void 0 : params.id,
      spaceId: rulesClient.getSpaceId(),
      usageCounter
    });
    try {
      (0, _validate_internal_rule_type.validateInternalRuleType)({
        ruleTypeId: createRuleData.rule_type_id,
        ruleTypes,
        operationText: 'create'
      });

      /**
       * Throws an error if the group is not defined in default actions
       */
      const {
        actions: allActions = []
      } = createRuleData;
      (0, _validation.validateRequiredGroupInDefaultActionsV1)({
        actions: allActions,
        isSystemAction: connectorId => actionsClient.isSystemAction(connectorId)
      });
      const actions = allActions.filter(action => !actionsClient.isSystemAction(action.id));
      const systemActions = allActions.filter(action => actionsClient.isSystemAction(action.id));

      // TODO (http-versioning): Remove this cast, this enables us to move forward
      // without fixing all of other solution types
      const createdRule = await rulesClient.create({
        data: (0, _transforms2.transformCreateBodyV1)({
          createBody: createRuleData,
          actions,
          systemActions
        }),
        options: {
          id: params === null || params === void 0 ? void 0 : params.id
        }
      });

      // Assert versioned response type
      const response = {
        body: (0, _transforms.transformRuleToRuleResponseV1)(createdRule)
      };
      return res.ok(response);
    } catch (e) {
      if (e instanceof _lib.RuleTypeDisabledError) {
        return e.sendResponse(res);
      }
      throw e;
    }
  }))));
};
exports.createRuleRoute = createRuleRoute;