"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearAlertFlappingHistory = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const clearAlertFlappingHistoryQuery = ruleIds => {
  const shouldStatusTerms = [_ruleDataUtils.ALERT_STATUS_ACTIVE, _ruleDataUtils.ALERT_STATUS_RECOVERED].map(status => {
    return {
      term: {
        [_ruleDataUtils.ALERT_STATUS]: {
          value: status
        }
      }
    };
  });
  const shouldRuleIdsTerms = ruleIds.map(ruleId => {
    return {
      term: {
        [_ruleDataUtils.ALERT_RULE_UUID]: {
          value: ruleId
        }
      }
    };
  });
  return {
    bool: {
      must: [{
        bool: {
          should: shouldStatusTerms
        }
      }, {
        bool: {
          should: shouldRuleIdsTerms
        }
      }]
    }
  };
};
const clearAlertFlappingHistory = async params => {
  const {
    indices,
    ruleIds,
    esClient,
    logger
  } = params;
  if (!ruleIds.length || !indices.length) {
    throw new Error('Rule Ids and indices must be provided');
  }
  try {
    let total = 0;
    for (let retryCount = 0; retryCount < 3; retryCount++) {
      var _response$total, _response$updated;
      const response = await esClient.updateByQuery({
        index: indices,
        allow_no_indices: true,
        conflicts: 'proceed',
        script: {
          source: `
            ctx._source['${_ruleDataUtils.ALERT_FLAPPING}'] = false;
            ctx._source['${_ruleDataUtils.ALERT_FLAPPING_HISTORY}'] = new ArrayList();
          `,
          lang: 'painless'
        },
        query: clearAlertFlappingHistoryQuery(ruleIds),
        refresh: true
      });
      if (total === 0 && response.total === 0) {
        logger.debug('No active or recovered alerts matched the query');
        break;
      }
      if (response.total) {
        total = response.total;
      }
      if (response.total === response.updated) {
        break;
      }
      logger.warn(`Attempt ${retryCount + 1}: Failed to clear flapping ${((_response$total = response.total) !== null && _response$total !== void 0 ? _response$total : 0) - ((_response$updated = response.updated) !== null && _response$updated !== void 0 ? _response$updated : 0)} of ${response.total}; indices: ${indices}, ruleIds: ${ruleIds}
        }`);
    }
    if (total === 0) {
      return [];
    }
  } catch (err) {
    logger.error(`Error clearing alert flapping for indices: ${indices}, ruleIds: ${ruleIds} - ${err.message}`);
    throw err;
  }
};
exports.clearAlertFlappingHistory = clearAlertFlappingHistory;