"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MAX_INITIALS = void 0;
exports.toInitials = toInitials;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

var MAX_INITIALS = exports.MAX_INITIALS = 2;

/**
 * This function calculates the initials/acronym for a given name.
 * It defaults to only 2 characters and will take the first character (of each word).
 * If only one word is supplied for the name, it will only pass back the first letter of the word,
 * unless forced to 2 letters by setting `initialsLength` to `2`.
 * It will pass back the characters with the same casing as the original string
 * unless otherwise specified.
 *
 * @param {string} name The full name of the item to turn into initials
 * @param {number} initialsLength (Optional) How many characters to show (max 2 allowed)
 * @param {string} initials (Optional) Custom initials (max 2 characters)
 */
function toInitials(name, initialsLength, initials) {
  // If `initials` provided, check if it's a single emoji
  // in order to support complex, "multi-character" ones
  if (initials && typeof Intl !== 'undefined' && 'Segmenter' in Intl) {
    var segmenter = new Intl.Segmenter('en', {
      granularity: 'grapheme'
    });
    var segments = Array.from(segmenter.segment(initials));
    if (segments.length === 1 && isEmoji(segments[0].segment)) {
      return segments[0].segment;
    }
  }

  // Calculate the number of initials to show, maxing out at MAX_INITIALS
  var calculatedInitialsLength = initials ? initials.split('').length : name.split(' ').length;
  calculatedInitialsLength = calculatedInitialsLength > MAX_INITIALS ? MAX_INITIALS : calculatedInitialsLength;

  // Check if initialsLength was passed and set to calculated, unless greater than MAX_INITIALS
  if (initialsLength) {
    calculatedInitialsLength = initialsLength <= MAX_INITIALS ? initialsLength : MAX_INITIALS;
  }
  var calculatedInitials;
  // A. Set to initials prop if exists (but truncate to 2 characters max unless length is supplied)
  if (initials) {
    calculatedInitials = initials.substring(0, calculatedInitialsLength);
  } else {
    if (name.trim() && name.split(' ').length > 1) {
      // B. If there are any spaces in the name, set to first letter of each word
      calculatedInitials = name.match(/\b(\w)/g);
      calculatedInitials = calculatedInitials && calculatedInitials.join('').substring(0, calculatedInitialsLength);
    } else {
      // C. Set to the name's initials truncated based on calculated length
      calculatedInitials = name.substring(0, calculatedInitialsLength);
    }
  }
  return calculatedInitials || '';
}
function isEmoji(str) {
  /**
   * The \p escape sequence allows matching a character based on its Unicode properties
   * @see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Regular_expressions/Unicode_character_class_escape
   * @see https://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-data.txt
   * @see https://www.unicode.org/reports/tr51/#Definitions
   * @see https://util.unicode.org/UnicodeJsps/character.jsp?a=1F440&B1=Show
   */
  var emojiRegex = /(?:[\xA9\xAE\u203C\u2049\u2122\u2139\u2194-\u2199\u21A9\u21AA\u231A\u231B\u2328\u2388\u23CF\u23E9-\u23F3\u23F8-\u23FA\u24C2\u25AA\u25AB\u25B6\u25C0\u25FB-\u25FE\u2600-\u2605\u2607-\u2612\u2614-\u2685\u2690-\u2705\u2708-\u2712\u2714\u2716\u271D\u2721\u2728\u2733\u2734\u2744\u2747\u274C\u274E\u2753-\u2755\u2757\u2763-\u2767\u2795-\u2797\u27A1\u27B0\u27BF\u2934\u2935\u2B05-\u2B07\u2B1B\u2B1C\u2B50\u2B55\u3030\u303D\u3297\u3299]|\uD83C[\uDC00-\uDCFF\uDD0D-\uDD0F\uDD2F\uDD6C-\uDD71\uDD7E\uDD7F\uDD8E\uDD91-\uDD9A\uDDAD-\uDDFF\uDE01-\uDE0F\uDE1A\uDE2F\uDE32-\uDE3A\uDE3C-\uDE3F\uDE49-\uDFFF]|\uD83D[\uDC00-\uDD3D\uDD46-\uDE4F\uDE80-\uDEFF\uDF74-\uDF7F\uDFD5-\uDFFF]|\uD83E[\uDC0C-\uDC0F\uDC48-\uDC4F\uDC5A-\uDC5F\uDC88-\uDC8F\uDCAE-\uDCFF\uDD0C-\uDD3A\uDD3C-\uDD45\uDD47-\uDEFF]|\uD83F[\uDC00-\uDFFD])/;
  return emojiRegex.test(str);
}