"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAttachmentStateManager = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _uuid = require("uuid");
var _attachments = require("@kbn/onechat-common/attachments");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Input for updating an existing attachment.
 */

/**
 * Result of resolving an attachment reference.
 */

/**
 * Interface for managing conversation attachment state.
 * Provides CRUD operations with version tracking.
 */

/**
 * Private implementation of AttachmentStateManager.
 */
class AttachmentStateManagerImpl {
  constructor(initialAttachments = []) {
    (0, _defineProperty2.default)(this, "attachments", void 0);
    (0, _defineProperty2.default)(this, "dirty", false);
    // Deep clone to avoid external mutation
    this.attachments = new Map();
    for (const attachment of initialAttachments) {
      this.attachments.set(attachment.id, structuredClone(attachment));
    }
  }
  get(id) {
    return this.attachments.get(id);
  }
  getLatest(id) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return undefined;
    }
    return (0, _attachments.getLatestVersion)(attachment);
  }
  getVersion(id, version) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return undefined;
    }
    return (0, _attachments.getVersion)(attachment, version);
  }
  getActive() {
    return Array.from(this.attachments.values()).filter(_attachments.isAttachmentActive);
  }
  getAll() {
    return Array.from(this.attachments.values());
  }
  getDiff(id, fromVersion, toVersion) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return undefined;
    }
    const fromVer = (0, _attachments.getVersion)(attachment, fromVersion);
    const toVer = (0, _attachments.getVersion)(attachment, toVersion);
    if (!fromVer || !toVer) {
      return undefined;
    }
    let changeType;
    let summary;
    const changedFields = [];
    if (fromVersion === 0 || !fromVer) {
      changeType = 'create';
      summary = `Created attachment "${attachment.description || attachment.id}"`;
    } else if (attachment.active === false && fromVersion < toVersion) {
      changeType = 'delete';
      summary = `Deleted attachment "${attachment.description || attachment.id}"`;
    } else if (attachment.active !== false && fromVersion < toVersion) {
      if (fromVer.content_hash !== toVer.content_hash) {
        changeType = 'update';
        summary = `Updated attachment "${attachment.description || attachment.id}"`;
        changedFields.push('data');
      } else {
        changeType = 'restore';
        summary = `Restored attachment "${attachment.description || attachment.id}"`;
      }
    } else {
      changeType = 'update';
      summary = `Updated attachment "${attachment.description || attachment.id}"`;
      if (fromVer.content_hash !== toVer.content_hash) {
        changedFields.push('data');
      }
    }
    return {
      change_type: changeType,
      summary,
      ...(changedFields.length > 0 && {
        changed_fields: changedFields
      })
    };
  }
  add(input) {
    const id = input.id || (0, _uuid.v4)();
    const now = new Date().toISOString();
    const contentHash = (0, _attachments.hashContent)(input.data);
    const tokens = (0, _attachments.estimateTokens)(input.data);
    const version = {
      version: 1,
      data: input.data,
      created_at: now,
      content_hash: contentHash,
      estimated_tokens: tokens
    };
    const attachment = {
      id,
      type: input.type,
      versions: [version],
      current_version: 1,
      active: true,
      ...(input.description && {
        description: input.description
      }),
      ...(input.hidden !== undefined && {
        hidden: input.hidden
      })
    };
    this.attachments.set(id, attachment);
    this.dirty = true;
    return attachment;
  }
  update(id, input) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return undefined;
    }
    if (input.description !== undefined) {
      attachment.description = input.description;
      this.dirty = true;
    }
    if (input.hidden !== undefined) {
      attachment.hidden = input.hidden;
      this.dirty = true;
    }
    if (input.data !== undefined) {
      const newHash = (0, _attachments.hashContent)(input.data);
      const currentVersion = (0, _attachments.getLatestVersion)(attachment);

      // Only create new version if content actually changed
      if (!currentVersion || currentVersion.content_hash !== newHash) {
        const now = new Date().toISOString();
        const tokens = (0, _attachments.estimateTokens)(input.data);
        const newVersionNum = attachment.current_version + 1;
        const newVersion = {
          version: newVersionNum,
          data: input.data,
          created_at: now,
          content_hash: newHash,
          estimated_tokens: tokens
        };
        attachment.versions.push(newVersion);
        attachment.current_version = newVersionNum;
        this.dirty = true;
      }
    }
    return attachment;
  }
  delete(id) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return false;
    }
    if (attachment.active === false) {
      return false;
    }
    attachment.active = false;
    this.dirty = true;
    return true;
  }
  restore(id) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return false;
    }
    if (attachment.active !== false) {
      return false;
    }
    attachment.active = true;
    this.dirty = true;
    return true;
  }
  permanentDelete(id) {
    if (!this.attachments.has(id)) {
      return false;
    }
    this.attachments.delete(id);
    this.dirty = true;
    return true;
  }
  rename(id, description) {
    const attachment = this.attachments.get(id);
    if (!attachment) {
      return false;
    }
    attachment.description = description;
    this.dirty = true;
    return true;
  }
  resolveRefs(refs) {
    const results = [];
    for (const ref of refs) {
      const attachment = this.attachments.get(ref.attachment_id);
      if (!attachment) {
        continue;
      }
      const version = (0, _attachments.getVersion)(attachment, ref.version);
      if (!version) {
        continue;
      }
      results.push({
        id: attachment.id,
        type: attachment.type,
        version,
        active: (0, _attachments.isAttachmentActive)(attachment)
      });
    }
    return results;
  }
  getTotalTokenEstimate() {
    let total = 0;
    for (const attachment of this.attachments.values()) {
      if ((0, _attachments.isAttachmentActive)(attachment)) {
        const latest = (0, _attachments.getLatestVersion)(attachment);
        if (latest !== null && latest !== void 0 && latest.estimated_tokens) {
          total += latest.estimated_tokens;
        }
      }
    }
    return total;
  }
  hasChanges() {
    return this.dirty;
  }
  markClean() {
    this.dirty = false;
  }
}

/**
 * Factory function to create an AttachmentStateManager.
 */
const createAttachmentStateManager = (initialAttachments = []) => {
  return new AttachmentStateManagerImpl(initialAttachments);
};
exports.createAttachmentStateManager = createAttachmentStateManager;