"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AnalyticsService = void 0;
var _onechatCommon = require("@kbn/onechat-common");
var _error_utils = require("./error_utils");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Server-side analytics wrapper for Agent Builder telemetry.
 *
 * This service centralizes event type registration and reporting for
 * `AGENT_BUILDER_EVENT_TYPES` so call sites can be kept small, typed, and safe.
 */
class AnalyticsService {
  constructor(analytics, logger) {
    this.analytics = analytics;
    this.logger = logger;
  }

  /**
   * Register Agent Builder server event types with core analytics.
   */
  registerAgentBuilderEventTypes() {
    _onechatCommon.agentBuilderServerEbtEvents.forEach(eventConfig => {
      this.analytics.registerEventType(eventConfig);
    });
  }
  reportAgentCreated({
    agentId,
    toolSelection
  }) {
    try {
      const normalizedAgentId = (0, _utils.normalizeAgentIdForTelemetry)(agentId);
      const toolIds = toolSelection.flatMap(selection => selection.tool_ids);
      const toolsIncluded = Array.from(new Set(toolIds)).map(toolId => (0, _utils.normalizeToolIdForTelemetry)(toolId));
      this.analytics.reportEvent(_onechatCommon.AGENT_BUILDER_EVENT_TYPES.AgentCreated, {
        agent_id: normalizedAgentId,
        tool_ids: toolsIncluded
      });
    } catch (error) {
      // Do not fail the request if telemetry fails
      this.logger.debug('Failed to report AgentCreated telemetry event', {
        error
      });
    }
  }
  reportAgentUpdated({
    agentId,
    toolSelection
  }) {
    try {
      const normalizedAgentId = (0, _utils.normalizeAgentIdForTelemetry)(agentId);
      const toolIds = toolSelection.flatMap(selection => selection.tool_ids);
      const toolsIncluded = Array.from(new Set(toolIds)).map(toolId => (0, _utils.normalizeToolIdForTelemetry)(toolId));
      this.analytics.reportEvent(_onechatCommon.AGENT_BUILDER_EVENT_TYPES.AgentUpdated, {
        agent_id: normalizedAgentId,
        tool_ids: toolsIncluded
      });
    } catch (error) {
      // Do not fail the request if telemetry fails
      this.logger.debug('Failed to report AgentUpdated telemetry event', {
        error
      });
    }
  }
  reportToolCreated({
    toolId,
    toolType
  }) {
    try {
      this.analytics.reportEvent(_onechatCommon.AGENT_BUILDER_EVENT_TYPES.ToolCreated, {
        tool_id: (0, _utils.normalizeToolIdForTelemetry)(toolId),
        tool_type: toolType
      });
    } catch (error) {
      // Do not fail the request if telemetry fails
      this.logger.debug('Failed to report ToolCreated telemetry event', {
        error
      });
    }
  }
  reportRoundComplete({
    agentId,
    conversationId,
    modelProvider,
    round,
    roundCount
  }) {
    try {
      var _round$steps$filter$m, _round$steps, _round$input$attachme;
      const normalizedAgentId = (0, _utils.normalizeAgentIdForTelemetry)(agentId);
      // NOTE: `tools_invoked` is intentionally an array that can include duplicates (one per tool
      // call). This allows downstream telemetry analysis to compute per-tool invocation counts by
      // aggregating over the array values.
      const toolsInvoked = (_round$steps$filter$m = (_round$steps = round.steps) === null || _round$steps === void 0 ? void 0 : _round$steps.filter(step => step.type === _onechatCommon.ConversationRoundStepType.toolCall).map(step => (0, _utils.normalizeToolIdForTelemetry)(step.tool_id))) !== null && _round$steps$filter$m !== void 0 ? _round$steps$filter$m : [];
      const attachments = (_round$input$attachme = round.input.attachments) !== null && _round$input$attachme !== void 0 && _round$input$attachme.length ? round.input.attachments.map(a => a.type || 'unknown') : undefined;
      this.analytics.reportEvent(_onechatCommon.AGENT_BUILDER_EVENT_TYPES.RoundComplete, {
        agent_id: normalizedAgentId,
        attachments,
        conversation_id: conversationId,
        input_tokens: round.model_usage.input_tokens,
        llm_calls: round.model_usage.llm_calls,
        message_length: round.input.message.length,
        model: round.model_usage.model,
        model_provider: modelProvider,
        output_tokens: round.model_usage.output_tokens,
        round_id: round.id,
        response_length: round.response.message.length,
        round_number: roundCount,
        started_at: round.started_at,
        time_to_first_token: round.time_to_first_token,
        time_to_last_token: round.time_to_last_token,
        tools_invoked: toolsInvoked
      });
    } catch (error) {
      // Do not fail the request if telemetry fails
      this.logger.debug('Failed to report RoundComplete telemetry event', {
        error
      });
    }
  }
  reportRoundError({
    agentId,
    conversationId,
    error,
    modelProvider,
    roundId
  }) {
    try {
      const normalizedAgentId = (0, _utils.normalizeAgentIdForTelemetry)(agentId);
      const errorType = (0, _error_utils.sanitizeForCounterName)((0, _error_utils.normalizeErrorType)(error));
      const errorMessage = (error instanceof Error ? error.message : String(error)).slice(0, 500);
      this.analytics.reportEvent(_onechatCommon.AGENT_BUILDER_EVENT_TYPES.RoundError, {
        agent_id: normalizedAgentId,
        conversation_id: conversationId,
        model_provider: modelProvider,
        error_message: errorMessage,
        error_type: errorType,
        ...(roundId ? {
          round_id: roundId
        } : {})
      });
    } catch (err) {
      // Do not fail the request if telemetry fails
      this.logger.debug('Failed to report RoundError telemetry event', {
        error: err
      });
    }
  }
}
exports.AnalyticsService = AnalyticsService;