"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateTitle = void 0;
var _rxjs = require("rxjs");
var _zod = require("@kbn/zod");
var _inferenceTracing = require("@kbn/inference-tracing");
var _langchain = require("@kbn/onechat-genai-utils/langchain");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Generates a title for a conversation
 */
const generateTitle = ({
  nextInput,
  conversation,
  chatModel
}) => {
  return (0, _rxjs.defer)(async () => {
    try {
      return await generateConversationTitle({
        previousRounds: conversation.rounds,
        nextInput,
        chatModel
      });
    } catch (e) {
      return conversation.title;
    }
  }).pipe((0, _rxjs.shareReplay)());
};
exports.generateTitle = generateTitle;
const generateConversationTitle = async ({
  previousRounds,
  nextInput,
  chatModel
}) => {
  return (0, _inferenceTracing.withActiveInferenceSpan)('GenerateTitle', {
    attributes: {
      [_inferenceTracing.ElasticGenAIAttributes.InferenceSpanKind]: 'CHAIN'
    }
  }, async span => {
    const structuredModel = chatModel.withStructuredOutput(_zod.z.object({
      title: _zod.z.string().describe('The title for the conversation')
    }).describe('Tool to use to provide the title for the conversation'), {
      name: 'set_title'
    });
    const prompt = [['system', `You are a title-generation utility. Your ONLY purpose is to create a short, relevant title for the provided conversation.

You MUST call the 'set_title' tool to provide the title. Do NOT respond with plain text or any other conversational language.

Here is an example:
Conversation:
- User: "Hey, can you help me find out how to configure a new role in Kibana for read-only access to dashboards?"
- Assistant: "Of course! To create a read-only role..."
=> Your response MUST be a call to the 'set_title' tool like this: {"title": "Kibana Read-Only Role Configuration"}

Now, generate a title for the following conversation.`], (0, _langchain.createUserMessage)(nextInput.message)];
    const {
      title
    } = await structuredModel.invoke(prompt);
    span === null || span === void 0 ? void 0 : span.setAttribute('output.value', title);
    return title;
  });
};