"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createChatService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _onechatCommon = require("@kbn/onechat-common");
var _inferenceCommon = require("@kbn/inference-common");
var _tracing = require("../../tracing");
var _utils = require("./utils");
var _events = require("./utils/events");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createChatService = options => {
  return new ChatServiceImpl(options);
};
exports.createChatService = createChatService;
class ChatServiceImpl {
  constructor(deps) {
    (0, _defineProperty2.default)(this, "dependencies", void 0);
    this.dependencies = deps;
  }
  converse({
    agentId = _onechatCommon.oneChatDefaultAgentId,
    conversationId,
    connectorId,
    capabilities,
    request,
    abortSignal,
    nextInput,
    autoCreateConversationWithId = false,
    browserApiTools
  }) {
    const {
      trackingService,
      analyticsService
    } = this.dependencies;
    const requestId = trackingService === null || trackingService === void 0 ? void 0 : trackingService.trackQueryStart();
    return (0, _tracing.withConverseSpan)({
      agentId,
      conversationId
    }, span => {
      // Resolve scoped services
      return (0, _rxjs.defer)(async () => {
        const services = await (0, _utils.resolveServices)({
          agentId,
          connectorId,
          request,
          ...this.dependencies
        });
        span === null || span === void 0 ? void 0 : span.setAttribute('elastic.connector.id', services.selectedConnectorId);

        // Get conversation and determine operation (CREATE or UPDATE)
        const conversation = await (0, _utils.getConversation)({
          agentId,
          conversationId,
          autoCreateConversationWithId,
          conversationClient: services.conversationClient
        });

        // Build conversation context
        const context = {
          conversation,
          conversationClient: services.conversationClient,
          chatModel: services.chatModel,
          selectedConnectorId: services.selectedConnectorId
        };
        return context;
      }).pipe((0, _rxjs.switchMap)(context => {
        // Emit conversation ID for new conversations
        const conversationIdEvent$ = context.conversation.operation === 'CREATE' ? (0, _rxjs.of)((0, _events.createConversationIdSetEvent)(context.conversation.id)) : _rxjs.EMPTY;

        // Execute agent
        const agentEvents$ = (0, _utils.executeAgent$)({
          agentId,
          request,
          nextInput,
          capabilities,
          abortSignal,
          conversation: context.conversation,
          defaultConnectorId: context.selectedConnectorId,
          agentService: this.dependencies.agentService,
          browserApiTools
        });

        // Generate title (for CREATE) or use existing title (for UPDATE)
        const title$ = context.conversation.operation === 'CREATE' ? (0, _utils.generateTitle)({
          chatModel: context.chatModel,
          conversation: context.conversation,
          nextInput
        }) : (0, _rxjs.of)(context.conversation.title);

        // Persist conversation
        const persistenceEvents$ = persistConversation({
          agentId,
          conversation: context.conversation,
          conversationClient: context.conversationClient,
          conversationId,
          title$,
          agentEvents$
        });

        // Merge all event streams
        const effectiveConversationId = context.conversation.operation === 'CREATE' ? context.conversation.id : conversationId;
        const modelProvider = (0, _inferenceCommon.getConnectorProvider)(context.chatModel.getConnector());
        return (0, _rxjs.merge)(conversationIdEvent$, agentEvents$, persistenceEvents$).pipe((0, _utils.handleCancellation)(abortSignal), (0, _utils.convertErrors)({
          agentId,
          logger: this.dependencies.logger,
          analyticsService,
          trackingService,
          modelProvider,
          conversationId: effectiveConversationId
        }), (0, _rxjs.tap)(event => {
          // Track round completion and query-to-result time
          try {
            if ((0, _onechatCommon.isRoundCompleteEvent)(event)) {
              var _context$conversation, _context$conversation2;
              if (requestId) trackingService === null || trackingService === void 0 ? void 0 : trackingService.trackQueryEnd(requestId);
              const currentRoundCount = ((_context$conversation = (_context$conversation2 = context.conversation.rounds) === null || _context$conversation2 === void 0 ? void 0 : _context$conversation2.length) !== null && _context$conversation !== void 0 ? _context$conversation : 0) + 1;
              if (conversationId) {
                trackingService === null || trackingService === void 0 ? void 0 : trackingService.trackConversationRound(conversationId, currentRoundCount);
              }
              analyticsService === null || analyticsService === void 0 ? void 0 : analyticsService.reportRoundComplete({
                conversationId: effectiveConversationId,
                roundCount: currentRoundCount,
                agentId,
                round: event.data.round,
                modelProvider
              });
            }
          } catch (error) {
            this.dependencies.logger.error(error);
          }
        }), (0, _rxjs.shareReplay)());
      }));
    });
  }
}

/**
 * Creates events for conversation persistence (create/update)
 */
const persistConversation = ({
  agentId,
  conversation,
  conversationClient,
  conversationId,
  title$,
  agentEvents$
}) => {
  const roundCompletedEvents$ = agentEvents$.pipe((0, _rxjs.filter)(_onechatCommon.isRoundCompleteEvent));
  if (conversation.operation === 'CREATE') {
    return (0, _utils.createConversation$)({
      agentId,
      conversationClient,
      conversationId: conversationId || conversation.id,
      title$,
      roundCompletedEvents$
    });
  }
  return (0, _utils.updateConversation$)({
    conversationClient,
    conversation,
    title$,
    roundCompletedEvents$
  });
};