"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentRoutes = registerAgentRoutes;
var _configSchema = require("@kbn/config-schema");
var _nodePath = _interopRequireDefault(require("node:path"));
var _wrap_handler = require("./wrap_handler");
var _constants = require("../../common/constants");
var _features = require("../../common/features");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TOOL_SELECTION_SCHEMA = _configSchema.schema.arrayOf(_configSchema.schema.object({
  tool_ids: _configSchema.schema.arrayOf(_configSchema.schema.string({
    meta: {
      description: 'Tool ID to be available to the agent.'
    }
  }), {
    meta: {
      description: 'Array of tool IDs that the agent can use.'
    }
  })
}, {
  meta: {
    description: 'Tool selection configuration for the agent.'
  }
}));
function registerAgentRoutes({
  router,
  getInternalServices,
  logger,
  analyticsService
}) {
  const wrapHandler = (0, _wrap_handler.getHandlerWrapper)({
    logger
  });

  // List agents
  router.versioned.get({
    path: `${_constants.publicApiPath}/agents`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'List agents',
    description: 'List all available agents. Use this endpoint to retrieve complete agent information including their current configuration and assigned tools.',
    options: {
      tags: ['agent', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: false,
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/agents_list.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents: agentsService
    } = getInternalServices();
    const service = await agentsService.getRegistry({
      request
    });
    const agents = await service.list();
    return response.ok({
      body: {
        results: agents
      }
    });
  }));

  // Get agent by id
  router.versioned.get({
    path: `${_constants.publicApiPath}/agents/{id}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.readOnechat]
      }
    },
    access: 'public',
    summary: 'Get an agent by ID',
    description: 'Get a specific agent by ID. Use this endpoint to retrieve the complete agent definition including all configuration details and tool assignments.',
    options: {
      tags: ['agent', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the agent to retrieve.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/agents_get_by_id.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const service = await agents.getRegistry({
      request
    });
    const profile = await service.get(request.params.id);
    return response.ok({
      body: profile
    });
  }));

  // Create agent
  router.versioned.post({
    path: `${_constants.publicApiPath}/agents`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Create an agent',
    description: "Create a new agent. Use this endpoint to define the agent's behavior, appearance, and capabilities through comprehensive configuration options.",
    options: {
      tags: ['agent', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        body: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'Unique identifier for the agent.'
            }
          }),
          name: _configSchema.schema.string({
            meta: {
              description: 'Display name for the agent.'
            }
          }),
          description: _configSchema.schema.string({
            meta: {
              description: 'Description of what the agent does.'
            }
          }),
          avatar_color: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Optional hex color code for the agent avatar.'
            }
          })),
          avatar_symbol: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Optional symbol/initials for the agent avatar.'
            }
          })),
          labels: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
            meta: {
              description: 'Label for categorizing the agent.'
            }
          }), {
            meta: {
              description: 'Optional labels for categorizing and organizing agents.'
            }
          })),
          configuration: _configSchema.schema.object({
            instructions: _configSchema.schema.maybe(_configSchema.schema.string({
              meta: {
                description: 'Optional system instructions that define the agent behavior.'
              }
            })),
            tools: TOOL_SELECTION_SCHEMA
          }, {
            meta: {
              description: 'Configuration settings for the agent.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/agents_create.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const service = await agents.getRegistry({
      request
    });
    const profile = await service.create(request.body);
    analyticsService === null || analyticsService === void 0 ? void 0 : analyticsService.reportAgentCreated({
      agentId: request.body.id,
      toolSelection: request.body.configuration.tools
    });
    return response.ok({
      body: profile
    });
  }));

  // Update agent
  router.versioned.put({
    path: `${_constants.publicApiPath}/agents/{id}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Update an agent',
    description: "Update an existing agent configuration. Use this endpoint to modify any aspect of the agent's behavior, appearance, or capabilities.",
    options: {
      tags: ['agent', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the agent to update.'
            }
          })
        }),
        body: _configSchema.schema.object({
          name: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Updated display name for the agent.'
            }
          })),
          description: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Updated description of what the agent does.'
            }
          })),
          avatar_color: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Updated hex color code for the agent avatar.'
            }
          })),
          avatar_symbol: _configSchema.schema.maybe(_configSchema.schema.string({
            meta: {
              description: 'Updated symbol/initials for the agent avatar.'
            }
          })),
          labels: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string({
            meta: {
              description: 'Updated label for categorizing the agent.'
            }
          }), {
            meta: {
              description: 'Updated labels for categorizing and organizing agents.'
            }
          })),
          configuration: _configSchema.schema.maybe(_configSchema.schema.object({
            instructions: _configSchema.schema.maybe(_configSchema.schema.string({
              meta: {
                description: 'Updated system instructions that define the agent behavior.'
              }
            })),
            tools: _configSchema.schema.maybe(TOOL_SELECTION_SCHEMA)
          }, {
            meta: {
              description: 'Updated configuration settings for the agent.'
            }
          }))
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/agents_update.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const service = await agents.getRegistry({
      request
    });
    const profile = await service.update(request.params.id, request.body);
    analyticsService === null || analyticsService === void 0 ? void 0 : analyticsService.reportAgentUpdated({
      agentId: profile.id,
      toolSelection: profile.configuration.tools
    });
    return response.ok({
      body: profile
    });
  }));

  // Delete agent
  router.versioned.delete({
    path: `${_constants.publicApiPath}/agents/{id}`,
    security: {
      authz: {
        requiredPrivileges: [_features.apiPrivileges.manageOnechat]
      }
    },
    access: 'public',
    summary: 'Delete an agent',
    description: 'Delete an agent by ID. This action cannot be undone.',
    options: {
      tags: ['agent', 'oas-tag:agent builder'],
      availability: {
        since: '9.2.0'
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            meta: {
              description: 'The unique identifier of the agent to delete.'
            }
          })
        })
      }
    },
    options: {
      oasOperationObject: () => _nodePath.default.join(__dirname, 'examples/agents_delete.yaml')
    }
  }, wrapHandler(async (ctx, request, response) => {
    const {
      agents
    } = getInternalServices();
    const service = await agents.getRegistry({
      request
    });
    const result = await service.delete({
      id: request.params.id
    });
    return response.ok({
      body: {
        success: result
      }
    });
  }));
}