"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.resolveResource = void 0;
var _esErrors = require("@kbn/es-errors");
var _onechatCommon = require("@kbn/onechat-common");
var _mappings = require("../mappings");
var _field_caps = require("../field_caps");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Retrieve the field list and other relevant info from the given resource name (index, alias or datastream)
 * Note: this can target a single resource, the resource name must not be a pattern.
 */
const resolveResource = async ({
  resourceName,
  esClient
}) => {
  if (resourceName.includes(',') || resourceName.includes('*')) {
    throw new Error(`Tried to resolve resource for multiple resources using pattern ${resourceName}`);
  }
  let resolveRes;
  try {
    resolveRes = await esClient.indices.resolveIndex({
      name: [resourceName],
      allow_no_indices: false
    });
  } catch (e) {
    if ((0, _esErrors.isNotFoundError)(e)) {
      throw new Error(`No resource found for '${resourceName}'`);
    }
    throw e;
  }
  const resourceCount = resolveRes.indices.length + resolveRes.aliases.length + resolveRes.data_streams.length;
  if (resourceCount !== 1) {
    throw new Error(`Found multiple targets when trying to resolve resource for ${resourceName}`);
  }

  // target is an index
  if (resolveRes.indices.length > 0) {
    var _mappings$_meta;
    const indexName = resolveRes.indices[0].name;
    const mappingRes = await (0, _mappings.getIndexMappings)({
      indices: [indexName],
      esClient,
      cleanup: true
    });
    const mappings = mappingRes[indexName].mappings;
    const fields = (0, _mappings.flattenMapping)(mappings);
    return {
      name: resourceName,
      type: _onechatCommon.EsResourceType.index,
      fields,
      description: (_mappings$_meta = mappings._meta) === null || _mappings$_meta === void 0 ? void 0 : _mappings$_meta.description
    };
  }
  // target is a datastream
  if (resolveRes.data_streams.length > 0) {
    var _mappings$_meta2;
    const datastream = resolveRes.data_streams[0].name;
    const mappingRes = await (0, _mappings.getDataStreamMappings)({
      datastreams: [datastream],
      esClient,
      cleanup: true
    });
    const mappings = mappingRes[datastream].mappings;
    const fields = (0, _mappings.flattenMapping)(mappings);
    return {
      name: resourceName,
      type: _onechatCommon.EsResourceType.dataStream,
      fields,
      description: (_mappings$_meta2 = mappings._meta) === null || _mappings$_meta2 === void 0 ? void 0 : _mappings$_meta2.description
    };
  }
  // target is an alias
  if (resolveRes.aliases.length > 0) {
    const alias = resolveRes.aliases[0].name;
    const fieldCapRes = await esClient.fieldCaps({
      index: alias,
      fields: ['*']
    });
    const {
      fields
    } = (0, _field_caps.processFieldCapsResponse)(fieldCapRes);
    return {
      name: resourceName,
      type: _onechatCommon.EsResourceType.alias,
      fields
    };
  }
  throw new Error(`No resource found for pattern ${resourceName}`);
};
exports.resolveResource = resolveResource;