"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processFieldCapsResponse = void 0;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * response for {@link processFieldCapsResponse}
 */

/**
 * Process a field caps response and return the list of targeted indices and
 * the list of corresponding non-conflicting fields.
 */
const processFieldCapsResponse = fieldCapsRes => {
  const indices = Array.isArray(fieldCapsRes.indices) ? fieldCapsRes.indices : [fieldCapsRes.indices];
  const fields = Object.entries(fieldCapsRes.fields).filter(([path, entry]) => {
    // exclude conflicting fields
    if (Object.keys(entry).length !== 1) {
      return false;
    }
    // exclude fields with internal types
    if (Object.keys(entry)[0].startsWith('_')) {
      return false;
    }
    return true;
  }).map(([path, entry]) => {
    return processField(path, entry);
  });
  return {
    indices,
    fields
  };
};
exports.processFieldCapsResponse = processFieldCapsResponse;
const processField = (path, entry) => {
  // filtered by caller
  if (Object.keys(entry).length > 1 || Object.keys(entry).length === 0) {
    throw new Error(`Trying to process field with conflicting types: ${path}`);
  }
  const fieldCaps = Object.values(entry)[0];
  let meta = {};
  if (fieldCaps.meta) {
    meta = Object.entries(fieldCaps.meta).reduce((acc, [key, value]) => {
      acc[key] = Array.isArray(value) ? value.join(',') : `${value}`;
      return acc;
    }, {});
  }
  return {
    path,
    type: fieldCaps.type,
    meta
  };
};