"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.indexExplorer = exports.formatResource = exports.createIndexSelectorPrompt = void 0;
var _lodash = require("lodash");
var _zod = require("@kbn/zod");
var _onechatCommon = require("@kbn/onechat-common");
var _list_search_sources = require("./steps/list_search_sources");
var _mappings = require("./utils/mappings");
var _xml = require("./utils/formatting/xml");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createIndexSummaries = async ({
  indices,
  esClient
}) => {
  const allMappings = await (0, _mappings.getIndexMappings)({
    indices: indices.map(index => index.name),
    cleanup: true,
    esClient
  });
  return indices.map(({
    name: indexName
  }) => {
    var _indexMappings$mappin;
    const indexMappings = allMappings[indexName];
    const flattened = (0, _mappings.flattenMapping)(indexMappings.mappings);
    return {
      type: _onechatCommon.EsResourceType.index,
      name: indexName,
      description: indexMappings === null || indexMappings === void 0 ? void 0 : (_indexMappings$mappin = indexMappings.mappings._meta) === null || _indexMappings$mappin === void 0 ? void 0 : _indexMappings$mappin.description,
      fields: flattened.map(field => field.path)
    };
  });
};
const createAliasSummaries = async ({
  aliases
}) => {
  // for now aliases are only described by the list of indices they target
  return aliases.map(({
    name: aliasName,
    indices
  }) => {
    return {
      type: _onechatCommon.EsResourceType.alias,
      name: aliasName,
      description: `Point to the following indices: ${indices.join(', ')}`
    };
  });
};
const createDatastreamSummaries = async ({
  datastreams,
  esClient
}) => {
  const allMappings = await (0, _mappings.getDataStreamMappings)({
    datastreams: datastreams.map(stream => stream.name),
    cleanup: true,
    esClient
  });
  return datastreams.map(({
    name
  }) => {
    var _mappings$mappings$_m;
    const mappings = allMappings[name];
    const flattened = (0, _mappings.flattenMapping)(mappings.mappings);
    return {
      type: _onechatCommon.EsResourceType.dataStream,
      name,
      description: mappings === null || mappings === void 0 ? void 0 : (_mappings$mappings$_m = mappings.mappings._meta) === null || _mappings$mappings$_m === void 0 ? void 0 : _mappings$mappings$_m.description,
      fields: flattened.map(field => field.path)
    };
  });
};
const indexExplorer = async ({
  nlQuery,
  indexPattern = '*',
  includeAliases = true,
  includeDatastream = true,
  limit = 1,
  esClient,
  model,
  logger
}) => {
  logger === null || logger === void 0 ? void 0 : logger.trace(() => `index_explorer - query="${nlQuery}", pattern="${indexPattern}"`);
  const sources = await (0, _list_search_sources.listSearchSources)({
    pattern: indexPattern,
    excludeIndicesRepresentedAsDatastream: true,
    excludeIndicesRepresentedAsAlias: false,
    esClient,
    includeKibanaIndices: indexPattern !== '*'
  });
  const indexCount = sources.indices.length;
  const aliasCount = sources.aliases.length;
  const dataStreamCount = sources.data_streams.length;
  const totalCount = indexCount + aliasCount + dataStreamCount;
  logger === null || logger === void 0 ? void 0 : logger.trace(() => `index_explorer - found ${indexCount} indices, ${aliasCount} aliases, ${dataStreamCount} datastreams for query="${nlQuery}"`);
  if (totalCount <= limit) {
    return {
      resources: [...sources.indices, ...sources.aliases, ...sources.data_streams].map(resource => {
        return {
          type: resource.type,
          name: resource.name,
          reason: `Index pattern matched less resources that the specified limit of ${limit}.`
        };
      })
    };
  }
  const resources = [];
  if (indexCount > 0) {
    const indexDescriptors = await createIndexSummaries({
      indices: sources.indices,
      esClient
    });
    resources.push(...indexDescriptors);
  }
  if (dataStreamCount > 0 && includeDatastream) {
    const dsDescriptors = await createDatastreamSummaries({
      datastreams: sources.data_streams,
      esClient
    });
    resources.push(...dsDescriptors);
  }
  if (aliasCount > 0 && includeAliases) {
    const aliasDescriptors = await createAliasSummaries({
      aliases: sources.aliases
    });
    resources.push(...aliasDescriptors);
  }
  const selectedResources = await selectResources({
    resources,
    nlQuery,
    model,
    limit
  });
  return {
    resources: selectedResources
  };
};
exports.indexExplorer = indexExplorer;
// Helper function to format each resource in an XML-like block
const formatResource = res => {
  var _res$fields, _res$description;
  const topFields = (0, _lodash.take)((_res$fields = res.fields) !== null && _res$fields !== void 0 ? _res$fields : [], 10);
  return (0, _xml.generateXmlTree)({
    tagName: 'resource',
    attributes: {
      type: res.type,
      name: res.name,
      description: (_res$description = res.description) !== null && _res$description !== void 0 ? _res$description : 'No description provided.'
    },
    children: [{
      tagName: 'sample_fields',
      children: topFields.length > 0 ? topFields.map(field => ({
        tagName: 'field',
        children: [field]
      })) : ['(No fields available)']
    }]
  });
};
exports.formatResource = formatResource;
const createIndexSelectorPrompt = ({
  resources,
  nlQuery,
  limit = 1
}) => {
  return [['system', `You are an AI assistant for Elasticsearch. Your task is to select the most relevant Elasticsearch resources based on a user query.

## CRITICAL INSTRUCTIONS

You MUST call the 'select_resources' tool. Do NOT respond with text or explanations.

## Tool Schema

The 'select_resources' tool expects this exact structure:
{
  "targets": [
    {
      "name": "resource_name",
      "type": "index" | "alias" | "data_stream",
      "reason": "why this resource is relevant"
    }
  ]
}

## Rules

1. ALWAYS call the 'select_resources' tool - never respond with plain text
2. The 'targets' property MUST be an array at the root level
3. Select at most ${limit} resource(s)
4. If no resources match, call the tool with: {"targets": []}
5. Each target must have: "name", "type", and "reason" fields
`], ['human', `## Query

"${nlQuery}"

## Available Resources

${resources.map(formatResource).join('\n')}

Call the 'select_resources' tool now with your selection. Maximum ${limit} target(s). Use an empty targets array if none match.`]];
};
exports.createIndexSelectorPrompt = createIndexSelectorPrompt;
const selectResources = async ({
  resources,
  nlQuery,
  model,
  limit = 1
}) => {
  const {
    chatModel
  } = model;
  const indexSelectorModel = chatModel.withStructuredOutput(_zod.z.object({
    targets: _zod.z.array(_zod.z.object({
      reason: _zod.z.string().describe('brief explanation of why this resource could be relevant'),
      type: _zod.z.enum([_onechatCommon.EsResourceType.index, _onechatCommon.EsResourceType.alias, _onechatCommon.EsResourceType.dataStream]).describe('the type of the resource'),
      name: _zod.z.string().describe('name of the resource')
    })).default([]).describe('The list of selected resources (indices, aliases and/or datastreams). Must be an array. Use an empty array if no resources match.')
  }).describe('Tool to select the relevant Elasticsearch resources to search against'), {
    name: 'select_resources'
  });
  const promptContent = createIndexSelectorPrompt({
    resources,
    nlQuery,
    limit
  });
  const response = await indexSelectorModel.invoke(promptContent);
  // Handle case where response might be malformed or targets is not an array
  return Array.isArray(response === null || response === void 0 ? void 0 : response.targets) ? response.targets : [];
};