"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.continueConversation = continueConversation;
exports.executeFunctionAndCatchError = executeFunctionAndCatchError;
var _gptTokenizer = require("gpt-tokenizer");
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _inferenceTracing = require("@kbn/inference-tracing");
var _errors = require("@kbn/inference-plugin/common/chat_complete/errors");
var _inferenceCommon = require("@kbn/inference-common");
var _get_inference_connector = require("../../../../common/utils/get_inference_connector");
var _tool_call = require("../../../analytics/tool_call");
var _common = require("../../../../common");
var _conversation_complete = require("../../../../common/conversation_complete");
var _create_function_response_message = require("../../../../common/utils/create_function_response_message");
var _emit_with_concatenated_message = require("../../../../common/utils/emit_with_concatenated_message");
var _create_server_side_function_response_error = require("../../util/create_server_side_function_response_error");
var _catch_function_not_found_error = require("./catch_function_not_found_error");
var _extract_messages = require("./extract_messages");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_FUNCTION_RESPONSE_TOKEN_COUNT = 4000;
const EXIT_LOOP_FUNCTION_NAME = 'exit_loop';
function executeFunctionAndCatchError({
  name,
  args,
  functionClient,
  messages,
  chat,
  signal,
  logger,
  connectorId,
  simulateFunctionCalling,
  analytics,
  connector,
  scopes
}) {
  return (0, _inferenceTracing.withExecuteToolSpan)(name, {
    tool: {
      input: args
    }
  }, span => {
    // hide token count events from functions to prevent them from
    // having to deal with it as well

    const executeFunctionResponse$ = (0, _rxjs.from)(functionClient.executeFunction({
      name,
      chat: (operationName, params) => {
        return chat(operationName, {
          ...params,
          connectorId
        });
      },
      args,
      signal,
      logger,
      messages,
      connectorId,
      simulateFunctionCalling
    }));
    return executeFunctionResponse$.pipe((0, _rxjs.tap)(() => {
      analytics.reportEvent(_tool_call.toolCallEventType, {
        toolName: name,
        connector: (0, _get_inference_connector.getInferenceConnectorInfo)(connector),
        scopes
      });
    }), (0, _rxjs.catchError)(error => {
      span === null || span === void 0 ? void 0 : span.recordException(error);
      logger.error(`Encountered error running function ${name}: ${JSON.stringify(error)}`);
      if ((0, _inferenceCommon.isToolValidationError)(error)) {
        return (0, _rxjs.of)((0, _create_function_response_message.createFunctionResponseMessage)({
          name,
          content: {
            message: error.message,
            errors: error.meta
          }
        }));
      }
      // We want to catch the error only when a promise occurs
      // if it occurs in the Observable, we cannot easily recover
      // from it because the function may have already emitted
      // values which could lead to an invalid conversation state,
      // so in that case we let the stream fail.
      return (0, _rxjs.of)((0, _create_server_side_function_response_error.createServerSideFunctionResponseError)({
        name,
        error
      }));
    }), (0, _rxjs.switchMap)(response => {
      if ((0, _rxjs.isObservable)(response)) {
        return response;
      }

      // is messageAdd event
      if ('type' in response) {
        return (0, _rxjs.of)(response);
      }
      const encoded = (0, _gptTokenizer.encode)(JSON.stringify(response.content || {}));
      const exceededTokenLimit = encoded.length >= MAX_FUNCTION_RESPONSE_TOKEN_COUNT;
      return (0, _rxjs.of)((0, _create_function_response_message.createFunctionResponseMessage)({
        name,
        content: exceededTokenLimit ? {
          message: 'Function response exceeded the maximum length allowed and was truncated',
          truncated: (0, _gptTokenizer.decode)((0, _lodash.take)(encoded, MAX_FUNCTION_RESPONSE_TOKEN_COUNT))
        } : response.content,
        data: response.data
      }));
    }));
  });
}
function getFunctionOptions({
  functionClient,
  disableFunctions,
  functionLimitExceeded
}) {
  if (disableFunctions === true) {
    return {};
  }
  if (functionLimitExceeded) {
    return {
      functionCall: EXIT_LOOP_FUNCTION_NAME,
      functions: [{
        name: EXIT_LOOP_FUNCTION_NAME,
        description: `You've run out of tool calls. Call this tool, and explain to the user you've run out of budget.`,
        parameters: {
          type: 'object',
          properties: {
            response: {
              type: 'string',
              description: 'Your textual response'
            }
          },
          required: ['response']
        }
      }]
    };
  }
  const systemFunctions = functionClient.getFunctions().map(fn => fn.definition).filter(({
    isInternal
  }) => !isInternal);
  const actions = functionClient.getActions();
  const allDefinitions = systemFunctions.concat(actions).map(definition => (0, _lodash.pick)(definition, 'name', 'description', 'parameters'));
  return {
    functions: allDefinitions
  };
}
function continueConversation({
  messages: initialMessages,
  functionClient,
  chat,
  signal,
  functionCallsLeft,
  apiUserInstructions = [],
  kbUserInstructions,
  logger,
  disableFunctions,
  connectorId,
  simulateFunctionCalling,
  analytics,
  connector,
  scopes
}) {
  var _last;
  let nextFunctionCallsLeft = functionCallsLeft;
  const functionLimitExceeded = functionCallsLeft <= 0;
  const functionOptions = getFunctionOptions({
    functionClient,
    disableFunctions,
    functionLimitExceeded
  });
  const lastMessage = (_last = (0, _lodash.last)(initialMessages)) === null || _last === void 0 ? void 0 : _last.message;
  const isUserMessage = (lastMessage === null || lastMessage === void 0 ? void 0 : lastMessage.role) === _common.MessageRole.User;
  return executeNextStep().pipe(handleEvents());
  function executeNextStep() {
    var _lastMessage$function;
    if (isUserMessage) {
      const operationName = lastMessage.name && lastMessage.name !== _common.CONTEXT_FUNCTION_NAME ? `function_response ${lastMessage.name}` : 'user_message';
      return chat(operationName, {
        messages: initialMessages,
        connectorId,
        stream: true,
        ...functionOptions
      }).pipe((0, _emit_with_concatenated_message.emitWithConcatenatedMessage)(), (0, _catch_function_not_found_error.catchFunctionNotFoundError)(functionLimitExceeded));
    }
    const functionCallName = lastMessage === null || lastMessage === void 0 ? void 0 : (_lastMessage$function = lastMessage.function_call) === null || _lastMessage$function === void 0 ? void 0 : _lastMessage$function.name;
    if (!functionCallName) {
      // reply from the LLM without a function request,
      // so we can close the stream and wait for input from the user
      return _rxjs.EMPTY;
    }

    // we know we are executing a function here, so we can already
    // subtract one, and reference the old count for if clauses
    const currentFunctionCallsLeft = nextFunctionCallsLeft;
    nextFunctionCallsLeft--;
    const isAction = functionCallName && functionClient.hasAction(functionCallName);
    if (currentFunctionCallsLeft === 0) {
      // create a function call response error so the LLM knows it needs to stop calling functions
      return (0, _rxjs.of)((0, _create_server_side_function_response_error.createServerSideFunctionResponseError)({
        name: functionCallName,
        error: (0, _conversation_complete.createFunctionLimitExceededError)()
      }));
    }
    if (currentFunctionCallsLeft < 0) {
      // LLM tried calling it anyway, throw an error
      return (0, _rxjs.throwError)(() => (0, _conversation_complete.createFunctionLimitExceededError)());
    }

    // if it's an action, we close the stream and wait for the action response
    // from the client/browser
    if (isAction) {
      try {
        functionClient.validate(functionCallName, JSON.parse(lastMessage.function_call.arguments || '{}'));
      } catch (error) {
        // return a function response error for the LLM to handle
        return (0, _rxjs.of)((0, _create_server_side_function_response_error.createServerSideFunctionResponseError)({
          name: functionCallName,
          error
        }));
      }
      return _rxjs.EMPTY;
    }
    if (!functionClient.hasFunction(functionCallName)) {
      var _arguments;
      // tell the LLM the function was not found
      return (0, _rxjs.of)((0, _create_server_side_function_response_error.createServerSideFunctionResponseError)({
        name: functionCallName,
        error: (0, _errors.createToolNotFoundError)({
          name: functionCallName,
          args: (_arguments = lastMessage.function_call.arguments) !== null && _arguments !== void 0 ? _arguments : ''
        })
      }));
    }
    return executeFunctionAndCatchError({
      name: functionCallName,
      args: lastMessage.function_call.arguments,
      chat,
      functionClient,
      messages: initialMessages,
      signal,
      logger,
      connectorId,
      simulateFunctionCalling,
      analytics,
      connector,
      scopes
    });
  }
  function handleEvents() {
    return events$ => {
      const shared$ = events$.pipe((0, _rxjs.shareReplay)(), (0, _rxjs.map)(event => {
        if (event.type === _common.StreamingChatResponseEventType.MessageAdd) {
          var _message$message$func;
          const message = event.message;
          if (((_message$message$func = message.message.function_call) === null || _message$message$func === void 0 ? void 0 : _message$message$func.name) === EXIT_LOOP_FUNCTION_NAME) {
            var _message$message$func2, _args$response;
            const args = JSON.parse((_message$message$func2 = message.message.function_call.arguments) !== null && _message$message$func2 !== void 0 ? _message$message$func2 : '{}');
            return {
              ...event,
              message: {
                ...message,
                message: {
                  ...(0, _lodash.omit)(message.message, 'function_call', 'content'),
                  content: (_args$response = args.response) !== null && _args$response !== void 0 ? _args$response : `The model returned an empty response`
                }
              }
            };
          }
        }
        return event;
      }));
      return (0, _rxjs.concat)(shared$, shared$.pipe((0, _extract_messages.extractMessages)(), (0, _rxjs.switchMap)(extractedMessages => {
        if (!extractedMessages.length) {
          return _rxjs.EMPTY;
        }
        return continueConversation({
          messages: initialMessages.concat(extractedMessages),
          chat,
          functionCallsLeft: nextFunctionCallsLeft,
          functionClient,
          signal,
          kbUserInstructions,
          apiUserInstructions,
          logger,
          disableFunctions,
          connectorId,
          simulateFunctionCalling,
          analytics,
          connector,
          scopes
        });
      })));
    };
  }
}