"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SCORE_SUGGESTIONS_FUNCTION_NAME = void 0;
exports.scoreSuggestions = scoreSuggestions;
var t = _interopRequireWildcard(require("io-ts"));
var _dedent = _interopRequireDefault(require("dedent"));
var _rxjs = require("rxjs");
var _ioTsUtils = require("@kbn/io-ts-utils");
var _lodash = require("lodash");
var _common = require("../../../../common");
var _parse_suggestion_scores = require("./parse_suggestion_scores");
var _short_id_table = require("../../../../common/utils/short_id_table");
var _get_last_user_message = require("./get_last_user_message");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SCORE_SUGGESTIONS_FUNCTION_NAME = exports.SCORE_SUGGESTIONS_FUNCTION_NAME = 'score_suggestions';
const scoreFunctionRequestRt = t.type({
  message: t.type({
    function_call: t.type({
      name: t.literal(SCORE_SUGGESTIONS_FUNCTION_NAME),
      arguments: t.string
    })
  })
});
const scoreFunctionArgumentsRt = t.type({
  scores: t.string
});
async function scoreSuggestions({
  suggestions,
  messages,
  screenDescription,
  chat,
  signal,
  logger
}) {
  const shortIdTable = new _short_id_table.ShortIdTable();
  const userPrompt = (0, _get_last_user_message.getLastUserMessage)(messages);
  const scoreFunction = {
    name: SCORE_SUGGESTIONS_FUNCTION_NAME,
    description: `Scores documents for relevance based on the user's prompt, conversation history, and screen context.`,
    parameters: {
      type: 'object',
      properties: {
        scores: {
          description: `A CSV string of document IDs and their integer scores (0-7). One per line, with no header. Example:
            my_id,7
            my_other_id,3
            my_third_id,0
          `,
          type: 'string'
        }
      },
      required: ['scores']
    }
  };
  const response = await (0, _rxjs.lastValueFrom)(chat('score_suggestions', {
    systemMessage: (0, _dedent.default)(`You are a Document Relevance Scorer.
        Your sole task is to compare each *document* in <DocumentsToScore> against three sources of context:

        1.  **<UserPrompt>** - what the user is asking right now.
        2.  **<ConversationHistory>** - the ongoing dialogue (including earlier assistant responses).
        3.  **<ScreenDescription>** - what the user is currently looking at in the UI.

        For every document you must assign one integer score from 0 to 7 (inclusive) that answers the question
        “*How helpful is this document for the user's current need, given their prompt <UserPrompt>, conversation history <ConversationHistory> and screen description <ScreenDescription>?*”

        ### Scoring rubric
        Use the following scale to assign a score to each document. Be critical and consistent.
        - **7:** Directly and completely answers the user's current need; almost certainly the top answer. 
        - **5-6:** Highly relevant; addresses most aspects of the prompt or clarifies a key point. 
        - **3-4:** Somewhat relevant; tangential, partial answer, or needs other docs to be useful. 
        - **1-2:** Barely relevant; vague thematic overlap only. 
        - **0:** Irrelevant; no meaningful connection.

        ### Mandatory rules
        1.  **Base every score only on the text provided**. Do not rely on outside knowledge.
        2.  **Never alter, summarise, copy, or quote the documents**. Your output is *only* the scores.
        3.  **Return the result exclusively by calling the provided function** \`${SCORE_SUGGESTIONS_FUNCTION_NAME}\`.
            * Populate the single argument 'scores' with a CSV string.
            * Format: '<documentId>,<score>' - one line per document, no header, no extra whitespace.
        4.  **Do not output anything else** (no explanations, no JSON wrappers, no markdown). The function call itself is the entire response.

        If you cannot parse any part of the input, still score whatever you can and give obviously unparsable docs a 0.

        ---
        CONTEXT AND DOCUMENTS TO SCORE
        ---

        <UserPrompt>
        ${userPrompt}
        </UserPrompt>

        <ConversationHistory>
        ${JSON.stringify(messages, null, 2)}
        </ConversationHistory>

        <ScreenDescription>
        ${screenDescription}
        </ScreenDescription>

        <DocumentsToScore>
        ${JSON.stringify(suggestions.map(suggestion => ({
      ...(0, _lodash.omit)(suggestion, 'esScore'),
      // Omit ES score to not bias the LLM
      id: shortIdTable.take(suggestion.id) // Shorten id to save tokens
    })), null, 2)}
        </DocumentsToScore>`),
    messages: [{
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common.MessageRole.User,
        content: userPrompt
      }
    }],
    functions: [scoreFunction],
    functionCall: SCORE_SUGGESTIONS_FUNCTION_NAME,
    signal,
    stream: true
  }).pipe((0, _common.concatenateChatCompletionChunks)()));
  const scoreFunctionRequest = (0, _ioTsUtils.decodeOrThrow)(scoreFunctionRequestRt)(response);
  const {
    scores: scoresAsString
  } = (0, _ioTsUtils.decodeOrThrow)(_ioTsUtils.jsonRt.pipe(scoreFunctionArgumentsRt))(scoreFunctionRequest.message.function_call.arguments);
  const llmScores = (0, _parse_suggestion_scores.parseSuggestionScores)(scoresAsString)
  // Restore original IDs (added fallback to id for testing purposes)
  .map(({
    id,
    llmScore
  }) => ({
    id: shortIdTable.lookup(id) || id,
    llmScore
  }));
  if (llmScores.length === 0) {
    // seemingly invalid or no scores, return all
    return {
      relevantDocuments: suggestions,
      llmScores: []
    };
  }

  // get top 5 documents ids
  const relevantDocuments = llmScores.filter(({
    llmScore
  }) => llmScore > 4).sort((a, b) => b.llmScore - a.llmScore).slice(0, 5).map(({
    id,
    llmScore
  }) => {
    const suggestion = suggestions.find(doc => doc.id === id);
    if (!suggestion) {
      return; // remove hallucinated documents
    }
    return {
      id,
      llmScore,
      esScore: suggestion.esScore,
      text: suggestion.text
    };
  }).filter(filterNil);
  logger.debug(() => `Relevant documents: ${JSON.stringify(relevantDocuments, null, 2)}`);
  return {
    relevantDocuments,
    llmScores: llmScores.map(score => ({
      id: score.id,
      llmScore: score.llmScore
    }))
  };
}
function filterNil(value) {
  return value !== null && value !== undefined;
}