"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getObsAIAssistantConnectorAdapter = void 0;
exports.getObsAIAssistantConnectorType = getObsAIAssistantConnectorType;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _dedent = _interopRequireDefault(require("dedent"));
var _i18n = require("@kbn/i18n");
var _configSchema = require("@kbn/config-schema");
var _zod = require("@kbn/zod");
var _common = require("@kbn/actions-plugin/common");
var _email = require("@kbn/connector-schemas/email");
var _jira = require("@kbn/connector-schemas/jira");
var _pagerduty = require("@kbn/connector-schemas/pagerduty");
var _slack_api = require("@kbn/connector-schemas/slack_api");
var _webhook = require("@kbn/connector-schemas/webhook");
var _common2 = require("@kbn/observability-ai-assistant-plugin/common");
var _aiAssistantCommon = require("@kbn/ai-assistant-common");
var _managementSettingsIds = require("@kbn/management-settings-ids");
var _concatenate_chat_completion_chunks = require("@kbn/observability-ai-assistant-plugin/common/utils/concatenate_chat_completion_chunks");
var _server = require("@kbn/observability-ai-assistant-plugin/server");
var _convert_schema_to_open_api = require("./convert_schema_to_open_api");
var _rule_connector = require("../../common/rule_connector");
var _constants = require("../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const CONNECTOR_PRIVILEGES = ['api:observabilityAIAssistant', 'app:observabilityAIAssistant'];
const connectorParamsSchemas = {
  '.slack': {
    type: 'object',
    properties: {
      id: {
        type: 'string'
      },
      params: {
        type: 'object',
        properties: {
          message: {
            type: 'string'
          }
        }
      }
    }
  },
  '.slack_api': (0, _convert_schema_to_open_api.convertSchemaToOpenApi)(_slack_api.SlackApiParamsSchema),
  '.email': (0, _convert_schema_to_open_api.convertSchemaToOpenApi)(_email.ParamsSchema),
  '.webhook': (0, _convert_schema_to_open_api.convertSchemaToOpenApi)(_webhook.ParamsSchema),
  '.jira': (0, _convert_schema_to_open_api.convertSchemaToOpenApi)(_jira.ExecutorParamsSchema),
  '.pagerduty': (0, _convert_schema_to_open_api.convertSchemaToOpenApi)(_pagerduty.ParamsSchema)
};
const ParamsSchema = _configSchema.schema.object({
  connector: _configSchema.schema.string(),
  prompts: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({
    statuses: _configSchema.schema.arrayOf(_configSchema.schema.string()),
    message: _configSchema.schema.string({
      minLength: 1
    })
  }))),
  status: _configSchema.schema.maybe(_configSchema.schema.string()),
  message: _configSchema.schema.maybe(_configSchema.schema.string({
    minLength: 1
  })) // this is a legacy field
});
const RuleSchema = _zod.z.object({
  id: _zod.z.string(),
  name: _zod.z.string(),
  tags: _zod.z.array(_zod.z.string()).default([]),
  ruleUrl: _zod.z.string().nullable().default(null)
}).strict();
const AlertSchema = _zod.z.record(_zod.z.string(), _zod.z.any());
const AlertSummarySchema = _zod.z.object({
  new: _zod.z.array(AlertSchema),
  recovered: _zod.z.array(AlertSchema)
}).strict();
const ConnectorParamsSchema = _zod.z.object({
  connector: _zod.z.string(),
  prompts: _zod.z.array(_zod.z.object({
    statuses: _zod.z.array(_zod.z.string()),
    message: _zod.z.string().min(1)
  })),
  rule: RuleSchema,
  alerts: AlertSummarySchema
}).strict();
function getObsAIAssistantConnectorType(initResources, alertDetailsContextService) {
  return {
    id: _rule_connector.OBSERVABILITY_AI_ASSISTANT_CONNECTOR_ID,
    isSystemActionType: true,
    getKibanaPrivileges: () => CONNECTOR_PRIVILEGES,
    minimumLicenseRequired: 'enterprise',
    name: _i18n.i18n.translate('xpack.observabilityAiAssistant.alertConnector.title', {
      defaultMessage: 'Observability AI Assistant'
    }),
    supportedFeatureIds: [_common.AlertingConnectorFeatureId],
    validate: {
      config: {
        schema: _zod.z.object({}).strict(),
        customValidator: () => {}
      },
      params: {
        schema: ConnectorParamsSchema
      },
      secrets: {
        schema: _zod.z.object({}).strict()
      }
    },
    renderParameterTemplates,
    executor(options) {
      return executor(options, initResources, alertDetailsContextService);
    }
  };
}
function renderParameterTemplates(logger, params, variables) {
  return {
    connector: params.connector,
    prompts: params.prompts,
    rule: params.rule,
    alerts: params.alerts
  };
}
async function executor(execOptions, initResources, alertDetailsContextService) {
  var _params$alerts, _params$alerts2;
  const {
    request,
    params
  } = execOptions;
  if ((((_params$alerts = params.alerts) === null || _params$alerts === void 0 ? void 0 : _params$alerts.new) || []).length === 0 && (((_params$alerts2 = params.alerts) === null || _params$alerts2 === void 0 ? void 0 : _params$alerts2.recovered) || []).length === 0) {
    // connector could be executed with only ongoing actions. we use this path as
    // dedup mechanism to prevent triggering the same worfklow for an ongoing alert
    return {
      actionId: execOptions.actionId,
      status: 'ok'
    };
  }
  if (!request) {
    throw new Error('AI Assistant connector requires a kibana request');
  }
  const resources = await initResources(request);
  const coreContext = await resources.context.core;
  const chatExperience = await coreContext.uiSettings.client.get(_managementSettingsIds.AI_CHAT_EXPERIENCE_TYPE);
  if (chatExperience === _aiAssistantCommon.AIChatExperience.Agent) {
    execOptions.logger.debug('Skipping Observability AI Assistant rule connector execution because chat experience is set to Agents.');
    return {
      actionId: execOptions.actionId,
      status: 'ok'
    };
  }
  const client = await resources.service.getClient({
    request,
    scopes: ['observability']
  });
  const functionClient = await resources.service.getFunctionClient({
    signal: new AbortController().signal,
    resources,
    client,
    screenContexts: [],
    scopes: ['observability']
  });
  const actionsClient = await (await resources.plugins.actions.start()).getActionsClientWithRequest(request);
  await Promise.all(params.prompts.map(prompt => executeAlertsChatCompletion(resources, prompt, params, alertDetailsContextService, client, functionClient, actionsClient, execOptions.logger)));
  return {
    actionId: execOptions.actionId,
    status: 'ok'
  };
}
async function executeAlertsChatCompletion(resources, prompt, params, alertDetailsContextService, client, functionClient, actionsClient, logger) {
  var _params$alerts3, _params$alerts4;
  const alerts = {
    new: [...(((_params$alerts3 = params.alerts) === null || _params$alerts3 === void 0 ? void 0 : _params$alerts3.new) || [])],
    recovered: [...(((_params$alerts4 = params.alerts) === null || _params$alerts4 === void 0 ? void 0 : _params$alerts4.recovered) || [])]
  };
  if (_constants.ALERT_STATUSES.some(status => prompt.statuses.includes(status))) {
    alerts.new = alerts.new.filter(alert => prompt.statuses.includes((0, _lodash.get)(alert, 'kibana.alert.status')));
    alerts.recovered = alerts.recovered.filter(alert => prompt.statuses.includes((0, _lodash.get)(alert, 'kibana.alert.status')));
  }
  if (alerts.new.length === 0 && alerts.recovered.length === 0) {
    return;
  }
  const connectorsList = await actionsClient.getAll().then(connectors => {
    return connectors.map(connector => {
      if (connector.actionTypeId in connectorParamsSchemas) {
        return {
          ...connector,
          parameters: connectorParamsSchemas[connector.actionTypeId]
        };
      }
      return connector;
    });
  });
  const backgroundInstruction = (0, _dedent.default)(`You are called as a background process because alerts have changed state.
As a background process you are not interacting with a user. Because of that DO NOT ask for user
input if tasked to execute actions. You can generate multiple responses in a row.
If available, include the link of the conversation at the end of your answer.`);
  functionClient.registerInstruction(backgroundInstruction);
  const hasSlackConnector = !!connectorsList.filter(connector => connector.actionTypeId === '.slack').length;
  if (hasSlackConnector) {
    const slackConnectorInstruction = ({
      availableFunctionNames
    }) => availableFunctionNames.includes(_server.EXECUTE_CONNECTOR_FUNCTION_NAME) ? (0, _dedent.default)(`The execute_connector function can be used to invoke Kibana connectors.
        To send to the Slack connector, you need the following arguments:
        - the "id" of the connector
        - the "params" parameter that you will fill with the message
        Please include both "id" and "params.message" in the function arguments when executing the Slack connector..`) : undefined;
    functionClient.registerInstruction(slackConnectorInstruction);
  }
  const alertsContext = await getAlertsContext(params.rule, alerts, async alert => {
    const alertDetailsContext = await alertDetailsContextService.getAlertDetailsContext({
      core: resources.context.core,
      licensing: resources.context.licensing,
      request: resources.request
    }, {
      alert_started_at: (0, _lodash.get)(alert, 'kibana.alert.start'),
      'service.name': (0, _lodash.get)(alert, 'service.name'),
      'service.environment': (0, _lodash.get)(alert, 'service.environment'),
      'host.name': (0, _lodash.get)(alert, 'host.name')
    });
    return alertDetailsContext.map(({
      description,
      data
    }) => `${description}:\n${JSON.stringify(data, null, 2)}`).join('\n\n');
  });
  const {
    response$
  } = client.complete({
    functionClient,
    persist: true,
    isPublic: true,
    connectorId: params.connector,
    signal: new AbortController().signal,
    kibanaPublicUrl: (await resources.plugins.core.start()).http.basePath.publicBaseUrl,
    messages: [{
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common2.MessageRole.User,
        content: prompt.message
      }
    }, {
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common2.MessageRole.Assistant,
        content: '',
        function_call: {
          name: 'get_alerts_context',
          arguments: JSON.stringify({}),
          trigger: _common2.MessageRole.Assistant
        }
      }
    }, {
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common2.MessageRole.User,
        name: 'get_alerts_context',
        content: JSON.stringify({
          context: alertsContext
        })
      }
    }, {
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common2.MessageRole.Assistant,
        content: '',
        function_call: {
          name: 'get_connectors',
          arguments: JSON.stringify({}),
          trigger: _common2.MessageRole.Assistant
        }
      }
    }, {
      '@timestamp': new Date().toISOString(),
      message: {
        role: _common2.MessageRole.User,
        name: 'get_connectors',
        content: JSON.stringify({
          connectors: connectorsList
        })
      }
    }]
  });
  response$.pipe((0, _rxjs.filter)(event => event.type === _common2.StreamingChatResponseEventType.ChatCompletionChunk)).pipe((0, _concatenate_chat_completion_chunks.concatenateChatCompletionChunks)()).subscribe({
    error: err => {
      logger.error(err);
    }
  });
}
const getObsAIAssistantConnectorAdapter = () => {
  return {
    connectorTypeId: _rule_connector.OBSERVABILITY_AI_ASSISTANT_CONNECTOR_ID,
    ruleActionParamsSchema: ParamsSchema,
    getKibanaPrivileges: () => CONNECTOR_PRIVILEGES,
    buildActionParams: ({
      params,
      rule,
      ruleUrl,
      alerts
    }) => {
      return {
        connector: params.connector,
        // Ensure backwards compatibility by using the message field as a prompt if prompts are missing
        prompts: params.prompts ? params.prompts : [{
          statuses: _constants.ALERT_STATUSES,
          message: params.message || ''
        }],
        rule: {
          id: rule.id,
          name: rule.name,
          tags: rule.tags,
          ruleUrl: ruleUrl !== null && ruleUrl !== void 0 ? ruleUrl : null
        },
        alerts: {
          new: alerts.new.data,
          recovered: alerts.recovered.data
        }
      };
    }
  };
};
exports.getObsAIAssistantConnectorAdapter = getObsAIAssistantConnectorAdapter;
async function getAlertsContext(rule, alerts, getAlertContext) {
  const getAlertGroupDetails = async alertGroup => {
    const formattedDetails = await Promise.all(alertGroup.map(async alert => {
      return `- ${JSON.stringify(alert)}. The following contextual information is available:\n${await getAlertContext(alert)}`;
    })).then(messages => messages.join('\n'));
    return formattedDetails;
  };
  let details = `The following alerts have changed state for the rule ${JSON.stringify(rule, null, 2)}:\n`;
  if (alerts.new.length > 0) {
    details += `- ${alerts.new.length} alerts have fired:\n${await getAlertGroupDetails(alerts.new)}\n`;
  }
  if (alerts.recovered.length > 0) {
    details += `- ${alerts.recovered.length} alerts have recovered\n: ${await getAlertGroupDetails(alerts.recovered)}\n`;
  }
  return details;
}